/*
 * Soya3D
 * Copyright (C) 1999  Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.awt;

import opale.soya.*;
import gl4java.*;

/**
 * An overlay is a 2D image drawn over a 3D scene (=over a rendering surface).
 * 
 * Hardware overlays are drawn by the monitor, and software by openGL methods (whitch can
 * use hardware acceleration). At this time, hardware overlays are not implemented.
 *
 * Warning : overlays may not be displayed if they are partially out the rendering
 * surface's screen.
 * 
 * @author Artiste on the Web
*/

public abstract class Overlay extends Object {
  /**
   * Create an overlay for the given rendering surface.
   * @param newRs the rendering surface
   */
  public Overlay(RenderingSurface newRs) {  
    rs = newRs;
    rs.addOverlay(this);
  }

  private RenderingSurface rs;
  /**
   * Gets the rendering surface over whitch this overlay is drawn.
   * @return the rendering surface
   */
  public RenderingSurface getRenderingSurface() { return rs; }
  /**
   * Removes / deletes this overlay.
   */
  public void remove() { rs.removeOverlay(this); }
  
  /**
   * Checks if it is an hardware overlay (=drawn by the monitor and not by the gl).
   * @return true if the overlay is in hardware.
   */
  public abstract boolean isHardware();

  private float left, top, width, height;
  public float getOverlayLeft  () { return left;   }
  public float getOverlayTop   () { return top;    }
  public float getOverlayWidth () { return width;  }
  public float getOverlayHeight() { return height; }
  public void setOverlayLeft  (float  f) { left = f;   computeDrawPosition(rs); }
  public void setOverlayTop   (float  f) { top = f;    computeDrawPosition(rs); }
  public void setOverlayWidth (float  f) { width = f;  computeDrawPosition(rs); }
  public void setOverlayHeight(float  f) { height = f; computeDrawPosition(rs); }
  /**
   * Moves the overlay : sets its position over its rendering surface.
   * @param newLeft the left coordinate
   * @param newTop the top coordinate
   * @param newWidth the width
   * @param newHeight the height
   */
  public void move(float newLeft, float newTop, float newWidth, float newHeight) {
    left   = newLeft;
    top    = newTop;
    width  = newWidth;
    height = newHeight;
    computeDrawPosition(rs);
  }
  
  /**
   * Constant for using abstract coordinates, independant of the size of the rendering
   * surface. Coordinate will range from (-.999999, -.999999) to (1, 1). (0, 0) is the center of the
   * overlay.
   */
  public static final int COORDINATES_MODE_ABSTRACT = 0;
  /**
   * Constant for using coordinates in pixel.
   */
  public static final int COORDINATES_MODE_PIXEL = 1;
  private int coordinatesMode = COORDINATES_MODE_ABSTRACT;
  /**
   * Gets the current coordinates mode. COORDINATES_MODE_ABSTRACT is the default.
   * @return a COORDINATES_MODE_* constant.
   */
  public int getCoordinatesMode() { return coordinatesMode; }
  /**
   * Sets the new coordinates mode.
   * @param cm a COORDINATES_MODE_* constant.
   */
  public void setCoordinatesMode(int cm) {
    coordinatesMode = cm;
    computeDrawPosition(rs);
  }
  
  /**
   * The drawing coordinates, in pixel.
   */
  protected int drawLeft, drawTop, drawWidth, drawHeight;
  /**
   * The size of the rendering surface, in pixel.
   */
  protected int renderingSurfaceWidth, renderingSurfaceHeight;
  /**
   * Checks if the drawing coordinates need to be updated, and updates them if necessary.
   * Should be called by child classes before each drawing.
   */
  protected void updateDrawPosition() {
    if((renderingSurfaceWidth != rs.getSurfaceWidth()) || (renderingSurfaceHeight != rs.getSurfaceHeight()))
      computeDrawPosition(rs);
  }
  /**
   * Compute the drawing coordinates.
   * Should be overriden by child classes if they want to change the computation.
   */
  protected void computeDrawPosition(RenderingSurface rs) {
    renderingSurfaceWidth  = rs.getSurfaceWidth ();
    renderingSurfaceHeight = rs.getSurfaceHeight();
    switch(coordinatesMode) {
    case COORDINATES_MODE_ABSTRACT:
      drawLeft   = (int) ((left + 1f) / 2f * (float) renderingSurfaceWidth );
      drawTop    = (int) ((top  + 1f) / 2f * (float) renderingSurfaceHeight);
      drawWidth  = (int) ((width    ) / 2f * (float) renderingSurfaceWidth );
      drawHeight = (int) ((height   ) / 2f * (float) renderingSurfaceHeight);
      break;
    case COORDINATES_MODE_PIXEL:
      drawLeft   = (int) left;
      drawTop    = (int) top;
      drawWidth  = (int) width;
      drawHeight = (int) height;
      break;
    }
  }
  
  protected boolean visible = true;
  /**
   * Gets if this overlay is visible. Default is true.
   * @return true if visible
   */
  public boolean isVisible () { return visible; }
  public boolean getVisible() { return visible; }
  /**
   * Sets if this overlay is visible.
   * @param b The new visibility. Non-visible overlay are not drawn.
   */
  public void setVisible(boolean b) { visible = b; }
  
  /**
   * Draws the overlay.
   * @param gl  The gl for drawing.
   * @param glu The glu for drawing.
   */
  public abstract void draw(GLFunc gl, GLUFunc glu);
}
