/*
 * Soya3D
 * Copyright (C) 1999  Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.awt;

import opale.soya.*;
import opale.soya.font.*;
import opale.soya.soya2d.*;
import gl4java.*;

/**
 * An overlay that write a text. You can also add a background color behind the text.
 * 
 * @author Artiste on the Web
 */

public abstract class TextOverlay extends Overlay {
  public TextOverlay(RenderingSurface rs) { super(rs); }
  
  /**
   * Creates and returns a new text overlay.
   * @param rs the rendering surface over whicth you want to write a text
   * @return a new text overlay
   */
  public static TextOverlay newTextOverlay(RenderingSurface rs) { // No support for hardware overlay at this time.
    // TODO : try hardware overlay here.
    return new SoftwareTextOverlay(rs);
  }
  /**
   * Creates and returns a new text overlay.
   * @param rs the rendering surface over whicth you want to write a text
   * @param text the text
   * @return a new text overlay
   */
  public static TextOverlay newTextOverlay(RenderingSurface rs, String text) {
    return newTextOverlay(rs, text, DefaultFont2D.HELVETICA_10);
  }
  /**
   * Creates and returns a new text overlay.
   * @param rs the rendering surface over whicth you want to write a text
   * @param text the text
   * @param font the font
   * @return a new text overlay
   */
  public static TextOverlay newTextOverlay(RenderingSurface rs, String text, Font2D font) {
    TextOverlay to = newTextOverlay(rs);
    to.setFont(font);
    to.setText(text);
    return to;
  }
  
  private String text;
  /**
   * Gets the text.
   * @return the text
   */
  public String getText() { return text; }
  /**
   * Sets the text. Can be null.
   * @param newText the new text
   */
  public void setText(String newText) {
    text = newText;
    computeDrawPosition(getRenderingSurface());
  }
  
  private Font2D font;
  /**
   * Gets the font.
   * @return the font
   */
  public Font2D getFont() { return font; }
  /**
   * Sets the font.
   * @param f the new font
   */
  public void setFont(Font2D f) {
    font = f;
    computeDrawPosition(getRenderingSurface());
  }
  
  private final float[] backgroundColor = (float[]) Material.NO_COLOR.clone();
  /**
   * Gets the text's back color. Default is soya.soya2d.Material.NO_COLOR .
   * @return the back color
   */
  public float[] getBackgroundColor() { return backgroundColor; }
  /**
   * Sets the text's back color, with implicit alpha = 1f.
   * @param red the red component
   * @param green the green component
   * @param blue the blue component
   */
  public void setBackgroundColor(float red, float green, float blue) { setBackgroundColor(red, green, blue, 1f); }
  /**
   * Sets the text's back color.
   * @param red the red component
   * @param green the green component
   * @param blue the blue component
   * @param alpha the alpha component
   */
  public void setBackgroundColor(float red, float green, float blue, float alpha) {
    backgroundColor[0] = red;
    backgroundColor[1] = green;
    backgroundColor[2] = blue;
    backgroundColor[3] = alpha;
  }
  /**
   * Sets the text's back color. Use soya.soya2d.Material.NO_COLOR for disabling back color;
   * this is the default.
   * @param c the 4 color's components
   */
  public void setBackgroundColor(float[] c) { System.arraycopy(c, 0, backgroundColor, 0, 4); }
  
  private final float[] color = (float[]) Material.WHITE_COLOR.clone();
  /**
   * Gets the text's color. Default is soya.soya2d.Material.WHITE_COLOR .
   * @return the color
   */
  public float[] getColor() { return color; }
  /**
   * Sets the text's color, with implicit alpha = 1f.
   * @param red the red component
   * @param green the green component
   * @param blue the blue component
   */
  public void setColor(float red, float green, float blue) { setColor(red, green, blue, 1f); }
  /**
   * Sets the text's color.
   * @param red the red component
   * @param green the green component
   * @param blue the blue component
   * @param alpha the alpha component
   */
  public void setColor(float red, float green, float blue, float alpha) {
    color[0] = red;
    color[1] = green;
    color[2] = blue;
    color[3] = alpha;
  }
  /**
   * Sets the text's color.
   * @param c the 4 color's components
   */
  public void setColor(float[] c) { System.arraycopy(c, 0, color, 0, 4); }
  
  /**
   * Sets the text's position. Coordinates must be in pixel or abstract, as defined by
   * getCoordinatesMode() .
   * @param newLeft the new left coordinate
   * @param newTop the new left coordinate
   */
  public void move(float newLeft, float newTop) { move(newLeft, newTop, 0f, 0f); }
  
  protected boolean centered;
  public boolean isCentered() { return centered; }
  public void setCentered(boolean b) {
    centered = b;
    computeDrawPosition(getRenderingSurface());
  }
  
  // Overrides :
  protected void computeDrawPosition(RenderingSurface rs) {
    if(font == null) return;
    renderingSurfaceWidth  = rs.getSurfaceWidth ();
    renderingSurfaceHeight = rs.getSurfaceHeight();
    switch(getCoordinatesMode()) {
    case COORDINATES_MODE_ABSTRACT:
      drawLeft   = (int) ((getOverlayLeft() + 1f) / 2f * ((float) renderingSurfaceWidth ));
      drawTop    = (int) ((getOverlayTop () + 1f) / 2f * ((float) renderingSurfaceHeight));
      if(text == null) {
        drawWidth  = 0;
        drawHeight = 0;
      }
      else {
        drawWidth  = font.getStringWidth (text);
        drawHeight = font.getStringHeight(text);
      }
      break;
    case COORDINATES_MODE_PIXEL:
      drawLeft   = (int) getOverlayLeft();
      drawTop    = (int) getOverlayTop ();
      if(text == null) {
        drawWidth  = 0;
        drawHeight = 0;
      }
      else {
        drawWidth  = font.getStringWidth (text);
        drawHeight = font.getStringHeight(text);
      }
      break;
    }
    if(centered) {
      drawLeft = drawLeft - drawWidth  / 2;
      drawTop  = drawTop  - drawHeight / 2;
    }
  }
}
