/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.editor;

import opale.soya.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import opale.soya.soya3d.Vector;
import opale.soya.soya3d.model.*;
import opale.soya.util.*;
import java.util.*;
import java.lang.Math;
import java.awt.event.*;

/**
 * A 3D element that corresponds to the mouse cursor. It is used by the GUIEditor.
 * 
 * You must set the cursor's handles property to a handles object ; a handles object is a
 * collection of Handle3D. Move or press the cursor (by calling the move, press or release
 * method) and the cursor will automatically transmit the event to its handles.
 * 
 * @author Artiste on the Web
 */

public class Cursor3D extends Volume3D {
  /**
   * Creates a new cursor.
   */
  public Cursor3D() {
    setShape(Handle3D.getGreenCube());
    setShowCoordinates(true);
  }

  protected Handles handles;
  /**
   * Gets the handles of this cursor.
   * @return the handles
   */
  public Handles getHandles() { return handles; }
  /**
   * Sets the handles of this cursor.
   * @param hs the new handles
   */
  public void setHandles(Handles hs) { 
    handles = hs;
    firePropertyChange("handles", null, null);
  }

  /**
   * Recovers the nearest handle of this cursor, in its handles collection.
   * @return the nearest handle in the handles
   */
  public Handle3D getNearestHandle() { return handles.getNearestHandle(this); }
  /**
   * Recovers the nearest selected handle of this cursor, in its handles collection.
   * @return the nearest selected handle in the handles
   */
  public Handle3D getNearestSelectedHandle() { return handles.getNearestSelectedHandle(this); }
  /**
   * Recovers the nearest unselected handle of this cursor, in its handles collection.
   * @return the nearest unselected handle in the handles
   */
  public Handle3D getNearestUnselectedHandle() { return handles.getNearestUnselectedHandle(this); }
  /**
   * Recovers the nearest clicked handle of this cursor, in its handles collection.
   * @return the nearest clicked handle in the handles
   */
  public Handle3D getNearestClickedHandle() { return handles.getNearestClickedHandle(this); }
  /**
   * Recovers the nearest unclicked handle of this cursor, in its handles collection.
   * @return the nearest unclicked handle in the handles
   */
  public Handle3D getNearestUnclickedHandle() { return handles.getNearestUnclickedHandle(this); }

  /**
   * Gets the position indicates by this cursor. It is the position of a selected but not
   * clicked handle (= the cursor point on this handle), or, if there is no such handle,
   * the cursor position itself.
   * @return the position indicated by the cursor
   */
  public Position getChoosenPosition() {
    Handle3D h = getNearestUnclickedHandle(); // the clicked handles are the ones manipulated!
    if((h != null) && (h.isSelected())) return h;
    else {
      if(magnetic) {
        Position p = new Point(this);
        magnetize(p);
        return p;
      }
      else return this;
    }
  }

  protected boolean pressed;
  /**
   * Checks if the cursor is pressed = if we are between a press() and a release() .
   * @return true if the cursor is pressed
   */
  public boolean isPressed() { return pressed; }
  /**
   * Press the cursor. You should call this method when the mouse is pressed, and pass as
   * argument the mouse event
   * @param e the mouse event that has caused this pression
   */
  public synchronized void press(MouseEvent e) {
    if(!pressed) {
      pressed = true;
      for(Iterator i = handles.iterator(); i .hasNext(); ) ((Handle3D) i.next()).cursorPress(this, e);
      firePropertyChange("pressed");
    }
  }
  /**
   * Press the cursor as if it was over the given handle. It will simulate a click on the
   * handle.
   * @param e the mouse event that has caused this pression
   * @param h the handle
   */
  public synchronized void press(MouseEvent e, Handle3D h) {
    //if(!pressed) {
    move(h);
    pressed = true;
    h.cursorPress(this, e);
    firePropertyChange("pressed");
    //}
  }
  /**
   * Hold the cursor : select all handles that is under or over the cursor.
   * @param e the mouse event that has caused this pression
   * @param v the holding's vector direction
   */
  public void hold(MouseEvent e, Vector v) {
    Handle3D h;
    for(Iterator i = handles.iterator(); i.hasNext(); ) {
      h = (Handle3D) i.next();
      if(!h.isVisible()) continue;
      Vector v2 = new Vector(this, h);
      float distance = v2.length() * ((float) Math.sin(((double) v.angle(v2) * Matrix.PI / 180f)));
      if(distance < magneticStep) {
        h.setClicked(true);
        move(h);
      }
      else {
        if(!e.isShiftDown()) h.setSelected(false);
      }
    }
  }
  /**
   * Releases the cursor. You should call this method when the mouse is released, and pass as
   * argument the mouse event
   * @param e the mouse event that has caused this release
   */
  public synchronized void release(MouseEvent e) {
    if(pressed) {
      pressed = false;
      for(Iterator i = handles.iterator(); i .hasNext(); ) ((Handle3D) i.next()).cursorRelease(this, e);
      firePropertyChange("pressed");
    }
  }
  /**
   * Moves the cursor. You should call this method when the mouse is moved, and pass as
   * argument the mouse event and the new cursor position.
   * @param e the mouse event that has caused this movement
   * @param p the new cursor position
   */
  public synchronized void move(MouseEvent e, Position p) {
    Vector v = new Vector(this, p);
    move(p);
    if(light != null) light.move(p);
    if(getShowCoordinates()) computeCoordinates();
    for(Iterator i = handles.iterator(); i .hasNext(); ) ((Handle3D) i.next()).cursorMove(this, v, e);
  }
  /**
   * Moves the cursor. You should call this method when the mouse is moved, and pass as
   * argument the mouse event and the new cursor position.
   * @param e the mouse event that has caused this movement
   * @param v the translation
   */
  public synchronized void move(MouseEvent e, Vector v) {
    addVector(v);
    if(light != null) light.addVector(v);
    if(getShowCoordinates()) computeCoordinates();
    for(Iterator i = handles.iterator(); i .hasNext(); ) ((Handle3D) i.next()).cursorMove(this, v, e);
  }
  
  protected boolean magnetic;
  /**
   * Checks if this cursor is magnetic. A magnetic cursor align all the handles it moves on a
   * magnetic grid. Use setStep(float) to choose the grid step.
   * @return true if this cursor is magnetic
   */
  public boolean isMagnetic() { return magnetic; }
  /**
   * Sets if this cursor is magnetic. A magnetic cursor align all the handles it moves on a
   * magnetic grid. Use setStep(float) to choose the grid step.
   * @param b true if this cursor is magnetic
   */
  public synchronized void setMagnetic(boolean b) {
    magnetic = b;
    firePropertyChange("magnetic");
  }

  protected float magneticStep = 1f;
  /**
   * Gets this cursor's magnetic step. It correspond to the magnetic grid step, and also to
   * the size of the cursor.
   * @return the magnetic step
   */
  public float getStep() { return magneticStep; }
  /**
   * Sets this cursor's magnetic step. It correspond to the magnetic grid step, and also to
   * the size of the cursor.
   * @param f the new magnetic step
   */
  public synchronized void setStep(float f) {
    scale(1 / magneticStep);
    magneticStep = f;
    scale(magneticStep);
    firePropertyChange("step");
  }

  /**
   * Magnetizes a position. If the cursor is magnetic, it will align the position on the
   * magnetic grid.
   * @param p the position
   */
  public void magnetize(Position p) {
    if(!magnetic) return;
    CoordSyst f = p.getCoordSyst();
    if((f != null) && (f != getCoordSyst())) { // CoordSyst conversion needed.
      synchronized(p) { // p must not be changed elsewhere;
        p.setCoordSyst(getCoordSyst());
        p.move(java.lang.Math.round(p.getX() / magneticStep) * magneticStep,
               java.lang.Math.round(p.getY() / magneticStep) * magneticStep,
               java.lang.Math.round(p.getZ() / magneticStep) * magneticStep);
        p.setCoordSyst(f);
      }
    }
    else {
      p.move(java.lang.Math.round(p.getX() / magneticStep) * magneticStep,
             java.lang.Math.round(p.getY() / magneticStep) * magneticStep,
             java.lang.Math.round(p.getZ() / magneticStep) * magneticStep);
    }
  }

  private Light3D light;
  /**
   * Checks if this cursor is luminous. A luminous cursor has a light that follows him.
   * @return true if luminous
   */
  public boolean isLuminous() { return light != null; }
  /**
   * Sets if this cursor is luminous. A luminous cursor has a light that follows him.
   * @param b true if luminous
   */
  public void setLuminous(boolean b) {
    if(b != isLuminous()) {
      if(b) {
        light = new Light3D();
        light.setStatic(true);
        light.setAttenuation(0f, 1f, 0f);
        World3D parent = getParent();
        if(parent != null) parent.add(light);
      }
      else {
        light.remove();
        light = null;
      }
      firePropertyChange("lit");
    }
  }

  private Label3D coordinates;
  /**
   * Checks if this cursor shows its coordinates.
   * @return true if coordinates are shown
   */
  public boolean getShowCoordinates() { return coordinates != null; }
  /**
   * Sets if this cursor shows its coordinates.
   * @param b true if coordinates are shown
   */
  public void setShowCoordinates(boolean b) {
    if(b != getShowCoordinates()) {
      if(b) {
        coordinates = new Label3D();
        coordinates.setOverlayed(true );
        coordinates.setLit      (false);
        World3D parent = getParent();
        if(parent != null) parent.add(coordinates);
      }
      else {
        coordinates.remove();
        coordinates = null;
      }
    }
  }
  private void computeCoordinates() {
    Position p = new Point(this);
    if(isMagnetic()) magnetize(p);
    coordinates.setText(p.getX() + ", " + p.getY() + ", " + p.getZ());
    coordinates.move(this);
  }

  protected void added(World3D into) {
    super.added(into);
    if(light != null) into.add(light);
    if(coordinates != null) {
      into.add(coordinates);
      computeCoordinates();
    }
  }
  protected void removed(World3D from) {
    super.removed(from);
    if(light != null) from.remove(light);
    if(coordinates != null) from.remove(coordinates);
  }
}
