/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.editor;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import opale.soya.soya3d.model.*;
import opale.soya.soya3d.model.Shape;
import opale.soya.soya3d.animation.*;
import java.awt.*;
import java.awt.event.*;
import java.beans.*;
import java.io.*;
import java.net.URL;
import java.lang.reflect.*;

/**
 * The soya Editor. Run this class to run the editor.
 * 
 * @author Artiste on the Web
 */

public class Editor extends Frame {
  protected Menu    newMenu ;
  protected Menu    openMenu;
  protected MenuBar menuBar ;
  /**
   * Create a new Editor.
   */
  public Editor() {
    super("Soya editor");
    options.set();

    setLayout(new GridLayout(2, 2));

    Panel p1 = new Panel();
    p1.setLayout(new FlowLayout());
    p1.add(new Label("Materials path :"));
    final TextField t1 = new TextField(options.materialsPath);
    t1.addTextListener(new TextListener() {
      public void textValueChanged(TextEvent e) {
        Material.setPath(t1.getText());
        options.materialsPath = Material.path;
        System.out.println("Materials path :" + Material.path);
      }
    });
    t1.setColumns(25);
    p1.add(t1);
    add(p1);

    Panel p2 = new Panel();
    p2.setLayout(new FlowLayout());
    p2.add(new Label("Shapes path :"));
    final TextField t2 = new TextField(options.shapesPath);
    t2.addTextListener(new TextListener() {
      public void textValueChanged(TextEvent e) {
        Shape.setPath(t2.getText());
        options.shapesPath = Shape.path;
        System.out.println("Shapes path :" + Shape.path);
      }
    });
    t2.setColumns(25);
    p2.add(t2);
    add(p2);
    
    Panel p3 = new Panel();
    p3.setLayout(new FlowLayout());
    p3.add(new Label("Animations path :"));
    final TextField t3 = new TextField(options.animationsPath);
    t3.addTextListener(new TextListener() {
      public void textValueChanged(TextEvent e) {
        Animations.setPath(t3.getText());
        options.animationsPath = Animations.path;
        System.out.println("Animations path :" + Animations.path);
      }
    });
    t3.setColumns(25);
    p3.add(t3);
    add(p3);
    
    MenuBar m = new MenuBar();
    menuBar = m;
    newMenu = new Menu("New", true);
    MenuItem m1 = new MenuItem("Material");
    newMenu.add(m1);
    m1.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) { autoEdit(new Material()); }
    });
    MenuItem m2 = new MenuItem("Shape");
    newMenu.add(m2);
    m2.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) { autoEdit(new FragmentedShape()); }
    });
    MenuItem m3 = new MenuItem("World");
    newMenu.add(m3);
    m3.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) { autoEdit(new World3D()); }
    });
    MenuItem m4 = new MenuItem("Environment");
    newMenu.add(m4);
    m4.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) { autoEdit(new Environment3D()); }
    });
    m.add(newMenu);
    
    openMenu = new Menu("Open");
    MenuItem m21 = new MenuItem("Material...");
    openMenu.add(m21);
    m21.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        try {
          final Material mat = Material.get(StoredInPathObjectChooser.question(Editor.this, Material.path, "material", false)); 
          autoEdit(mat);
        }
        catch(Exception e2) { System.out.println("can't find or edit the material!"); e2.printStackTrace(); }
      }
    });
    MenuItem m22 = new MenuItem("Shape...");
    openMenu.add(m22);
    m22.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        try {	
          final Shape sha = Shape.get(StoredInPathObjectChooser.question(Editor.this, Shape.path, "shape", false)); 
          autoEdit(sha);
        }
        catch(Exception e2) { System.out.println("can't find or edit the shape!"); e2.printStackTrace(); }
      }
    });    
    MenuItem m23 = new MenuItem("File...");
    openMenu.add(m23);
    m23.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        try {
          FileDialog fd = new FileDialog(Editor.this, "Open which file?", FileDialog.LOAD);
          fd.setFilenameFilter(new FilenameFilter() {
            public boolean accept(File dir, String file) { return file.endsWith(".ser"); }
          });
          fd.setVisible(true);
          autoEdit(Soya.load(fd.getDirectory() + fd.getFile()));
        }
        catch(Exception e2) { System.out.println("can't find or edit this file!"); e2.printStackTrace(); }
      }
    });
    m.add(openMenu);
    setMenuBar(m);
    
    setSize(450,300);
    
    addWindowListener(new WindowAdapter() {
      public void windowClosing(WindowEvent e) {
        dispose();
        options.save();
      }
    });
  }
  
  private static Options options = Options.get();
  public static Options getOptions() { return options; }
  public static class Options implements Serializable {
    private static final long serialVersionUID = 5069576975559515361l;
    public String materialsPath;
    public String shapesPath;
    public String animationsPath;
    
    public float magneticStep = .1f;
    public float inclineStep  = 11.25f;
    public boolean showGround = false;
    
    public void save() {
      try {
        ObjectOutputStream oi = new ObjectOutputStream(new FileOutputStream(System.getProperty("user.home") + "/.soyaEditorOptions"));
        oi.writeObject(this);
        oi.close();
      }
      catch(Exception e) { System.out.println("Can't save the editorOptions file" + e); e.printStackTrace(); }
    }
    public static Options get() {
      try {
        InputStream i = new FileInputStream(System.getProperty("user.home") + "/.soyaEditorOptions");
        if(i == null) return new Options();
        ObjectInputStream oi = new ObjectInputStream(i);
        Options op = (Options) oi.readObject();
        if(op.magneticStep <= 0f) op.magneticStep  = .1f;
        if(op.inclineStep  <= 0f) op.inclineStep   = 11.25f;
        oi.close();
        return op;
      }
      catch(Exception e) {
        System.out.println("Can't load the editorOptions file" + e);
        e.printStackTrace();
        return new Options();
      }
    }
    public void set() {
      Material.path = materialsPath;
      Shape.path = shapesPath;
      Animations.path = animationsPath;
    }
  }
  
  /**
   * Main method that will run a new editor.
   */
  public static void main(String[] args) {
    Soya.init();
    Editor e = new Editor();
    e.setLocation(64, 64);
    e.setVisible(true);
  }
  
  /**
   * Edit a bean with the soya bean editor.
   * @param bean the bean
   * @return the bean editor window
   */
  public static Window edit(Object bean) throws IntrospectionException, InstantiationException, IllegalAccessException { return edit(bean, true); }
  /**
   * Edit a bean with the soya bean editor.
   * @param bean the bean
   * @param allowSaving true if the bean editor should ask for saving after closing.
   * @return the bean editor window
   */
  public static Window edit(final Object bean, boolean allowSaving) throws IntrospectionException, InstantiationException, IllegalAccessException {
    final BeanEditor ps = new BeanEditor(bean);
    ps.show();
    
    if(allowSaving && isSaveable(bean)) {
      ps.addWindowListener(new WindowAdapter() {
        public void windowClosed(WindowEvent e) {
          int a = YesNoCancelDialog.question((Frame) ps, "Saving?", "Save this " + bean.getClass() + " ?");
          if(a == YesNoCancelDialog.CANCEL) {
            try { edit(bean, true); }
            catch(Exception ex) { ex.printStackTrace(); }
            return;
          }
          if(a == YesNoCancelDialog.YES) {
            try { save(bean, ps); }
            catch(Exception ex) { System.out.println("Can't save this " + bean.getClass() + " !"); ex.printStackTrace(); }
          }
        }
      });
    }
    return ps;
  }
  public static Window edit(final Object bean, boolean allowSaving, Customizer customizer) throws IntrospectionException, InstantiationException, IllegalAccessException {
    final BeanEditor ps = new BeanEditor(bean, customizer);
    ps.show();
    
    if(allowSaving && isSaveable(bean)) {
      ps.addWindowListener(new WindowAdapter() {
        public void windowClosed(WindowEvent e) {
          int a = YesNoCancelDialog.question((Frame) ps, "Saving?", "Save this " + bean.getClass() + " ?");
          if(a == YesNoCancelDialog.CANCEL) {
            try { edit(bean, true); }
            catch(Exception ex) { ex.printStackTrace(); }
            return;
          }
          if(a == YesNoCancelDialog.YES) {
            try { save(bean, ps); }
            catch(Exception ex) { System.out.println("Can't save this " + bean.getClass() + " !"); ex.printStackTrace(); }
          }
        }
      });
    }
    return ps;
  }
  protected Window autoEdit(Object bean) {
    try { return edit(bean, true); }
    catch(Exception e) { System.out.println("Can't edit this bean!"); e.printStackTrace(); }
    return null;
  }
  
  /**
   * Checks if the given bean is saveable. A saveable bean is a bean that has at least one
   * of these methods :
   *   save();
   *   save(java.lang.string);
   * The string parameter of the second method is intented to be the filename.
   * @param bean the bean
   */  
  public static boolean isSaveable(Object bean) {
    try {
      Method m = bean.getClass().getMethod("save", null);
      return true;
    }
    catch(Exception e) {  }
    Class[] paramsClasses = { java.lang.String.class };
    try {
      Method m = bean.getClass().getMethod("save", paramsClasses);
      return true;
    }
    catch(Exception e) {  }
    return false;
  }
  
  /**
   * Saves the given bean. First, try to invoke this method :
   *   save();
   * And if it fail, ask for a filename and try with this method :
   *   save(java.lang.string);
   * The string parameter is intented to be the filename.
   * This method can throws any exception that the bean's save method can throws (often
   * IOException), and also NoSuchMethodException if the bean is not saveable.
   * @param bean the bean
   * @param frame the caller's frame (for showing the file dialog box).
   */  
  public static void save(Object bean, Frame frame) throws Exception {
    try {
      Method m = bean.getClass().getMethod("save", null);
      m.invoke(bean, null);
      return;
    }
    catch(Exception e) { e.printStackTrace(); }
    
    saveAs(bean, frame);
  }
  /**
   * Ask for a filename and saves the given bean, by try to invoke this method :
   *   save(java.lang.string);
   * The string parameter is intented to be the filename.
   * This method can throws any exception that the bean's save method can throws (often
   * IOException), and also NoSuchMethodException if the bean is not saveable.
   * @param bean the bean
   * @param frame the caller's frame (for showing the file dialog box).
   */
  public static void saveAs(Object bean, Frame frame) throws Exception {
    String fileName = askForFileName(frame);
    Class[] paramsClasses = { java.lang.String.class };
    Object[] params = { fileName };
    try {
      Method m = bean.getClass().getMethod("save", paramsClasses);
      m.invoke(bean, params);
    }
    catch(Exception e) { throw(e); }
  }
  private static String askForFileName(Frame frame) {
    FileDialog fd = new FileDialog(frame, "Saving", FileDialog.SAVE);
    fd.setFilenameFilter(new FilenameFilter() {
      public boolean accept(File dir, String file) { return file.endsWith(".ser"); }
    });
    fd.setVisible(true);
    return fd.getDirectory() + fd.getFile();
  }

  /**
   * Gets the name of the given bean by trying to invoke a String getName() method.
   * @param bean the bean
   * @return the bean's name
   */  
  public String getName(Object bean) {
    String name = null;
    try {
      Method m = bean.getClass().getMethod("getName", null);
      name = (String) m.invoke(bean, null);
    }
    catch(Exception e) {  }
    if(name == null) name = "(no name)";
    return name;
  }
}
