/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General PublicS License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USAint
 */

package opale.soya.editor;

import opale.soya.*;
import opale.soya.util.Lockable;
import opale.soya.util.*;
import opale.soya.awt.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import opale.soya.soya3d.Point;
import java.beans.*;
import java.lang.reflect.*;
import java.awt.*;
import java.awt.event.*;
import java.util.Iterator;

/**
 * An abstract class for all GUI editors.
 * 
 * A GUI editor is a CoordSyst window that contains a status bar, a 3D view (= a rendering canvas)
 * and a menu bar. The menu bar contains default menu (such as file, edit or view).
 * It also include a 3D cursor gestionary.
 * 
 * Many method are protected and correspond to the GUI editor action; you can call them if
 * needed, or you can override them if you want a different implementation.
 * 
 * @author Artiste on the Web
 */

public abstract class GUIEditor extends java.awt.Frame implements Lockable {
  /**
   * The menu bar. Add your own menu if needed.
   */
  protected MenuBar menuBar = new MenuBar();
  /**
   * The file menu. A close menu item is already included.
   */
  protected Menu fileMenu = new Menu("File");
  /**
   * The edit menu. Menu item for cursor properties(luminous, show-coordinate and grid step) are
   * already included.
   */
  protected Menu editMenu = new Menu("Edit"); 
  /**
   * The view menu. Basic views are already included.
   */
  protected Menu viewMenu = new Menu("View");
  
  /**
   * The orientation that will be rotated when needed. Default is world.
   */
  protected Orientation toRotate;
  
  private CheckboxMenuItem cameraOrtho;
  
  protected Cursor3D createCursor() { return new Cursor3D(); }
  
  /**
   * Creates a new GUI editor.
   */
  public GUIEditor() { this(""); }
  /**
   * Creates a new GUI editor with the given title.
   */
  public GUIEditor(String title) {
    super(title);
    setLayout(new BorderLayout());
    setSize(640, 480);
    renderingCanvas = new RenderingCanvas(300, 200);
    add(renderingCanvas);
    add(statusBar, BorderLayout.SOUTH);
    new CursorMover(renderingCanvas);
    renderingCanvas.showFPS();
    renderingCanvas.setCursor(Cursor.getPredefinedCursor(Cursor.MOVE_CURSOR));
    
    cursor = createCursor();
    
    addWindowListener(new WindowAdapter() {
      public void windowClosing(WindowEvent e) { GUIEditor.this.dispose(); }
    });
    
    // Build menu:
    setMenuBar(menuBar);
    
    menuBar.add(fileMenu);
    MenuItem file2 = new MenuItem("Close"); fileMenu.add(file2);
    file2.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) { GUIEditor.this.dispose(); }
    });
    
    menuBar.add(editMenu);
    final CheckboxMenuItem edit1 = new CheckboxMenuItem("Luminous cursor", true); editMenu.add(edit1);
    edit1.addItemListener(new ItemListener() {
      public void itemStateChanged(ItemEvent e) { cursor.setLuminous(edit1.getState()); }
    });
    final CheckboxMenuItem edit2 = new CheckboxMenuItem("Show cusor coordinates", true); editMenu.add(edit2);
    edit2.addItemListener(new ItemListener() {
      public void itemStateChanged(ItemEvent e) { cursor.setShowCoordinates(edit2.getState()); }
    });
    MenuItem edit3 = new MenuItem("Change grid step..."); editMenu.add(edit3);
    edit3.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        float f = Float.parseFloat(TextChooser.question(GUIEditor.this, "New grid step", Float.toString(getStep())));
        if(f > 0f) setStep(f);
      }
    });
    cameraOrtho = new CheckboxMenuItem("Ortho camera"); editMenu.add(cameraOrtho);
    cameraOrtho.setShortcut(new MenuShortcut(KeyEvent.VK_C));
    cameraOrtho.addItemListener(new ItemListener() {
      public void itemStateChanged(ItemEvent e) {
        camera.setOrtho(cameraOrtho.getState());
        if(!camera.isOrtho()) camera.setFOV(90f);
        render();
      }
    });
    
    menuBar.add(viewMenu);
    MenuItem view1 = new MenuItem("Reset view"); viewMenu.add(view1);
    view1.setShortcut(new MenuShortcut(KeyEvent.VK_NUMPAD0));
    view1.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) { resetView(); }
    });
    viewMenu.add(new MenuItem("-"));
    MenuItem view2 = new MenuItem("Back view"); viewMenu.add(view2);
    view2.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) { backView(); }
    });
    MenuItem view3 = new MenuItem("Front view"); viewMenu.add(view3);
    view3.setShortcut(new MenuShortcut(KeyEvent.VK_NUMPAD5));
    view3.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) { frontView(); }
    });
    MenuItem view4 = new MenuItem("Left view"); viewMenu.add(view4);
    view4.setShortcut(new MenuShortcut(KeyEvent.VK_NUMPAD4));
    view4.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) { leftView(); }
    });
    MenuItem view5 = new MenuItem("Right view"); viewMenu.add(view5);
    view5.setShortcut(new MenuShortcut(KeyEvent.VK_NUMPAD6));
    view5.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) { rightView(); }
    });
    MenuItem view6 = new MenuItem("Upper view"); viewMenu.add(view6);
    view6.setShortcut(new MenuShortcut(KeyEvent.VK_NUMPAD8));
    view6.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) { upperView(); }
    });
    MenuItem view7 = new MenuItem("Lower view"); viewMenu.add(view7);
    view7.setShortcut(new MenuShortcut(KeyEvent.VK_NUMPAD2));
    view7.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) { lowerView(); }
    });
    
    // Build scene:
    scene.setAmbientColor(0.75f, 0.75f, 0.75f);
    scene.add(camera);
    camera.move(0, 0, 2);
    renderingCanvas.setRenderer(camera);
    world.add(cursor);
    cursor.setLuminous(true);
    scene.add(world);
    //world.add(new Volume3D(AxisFragment.createShape()));
    world.add(new Axis3D());
    world.setRotationType(Orientation.ROTATION_TYPE_EXTERNAL);
    toRotate = world;
    
    renderingCanvas.requestFocus();
  }

  /**
   * Resets the view.
   */
  protected void resetView() {
    world.resetOrientation();
    camera.move(0f, 0f, 2f);
    camera.getRenderingSurface().render();
    setCursorVisible();
  }
  /**
   * Sets the view to a back view (this is the default view).
   */
  protected void backView() {
    world.lookAt(new Vector(0f, 0f, -1f, camera));       // Orientate the object so it presents its left face to the eye (eye is supposed to look at 0, 0, -1).
    
    DimensionWrapper w = world.wrapper();              // Gets a wrapper from the object, in order to perform frame-coordinate conversion on dimensions.
    w.setCoordSyst(camera.getParent());               // Convert the dimension into eye-parent frame...
    camera.move(0f, 0f, w.getMax().getZ() + 1f);     // ...and move the eye at it's biggest Z value plus 1.
    
    camera.getRenderingSurface().render();         // Update the view.
    setCursorVisible();
  }
  /**
   * Sets the view to a front view.
   */
  protected void frontView() {
    world.lookAt(new Vector(0f, 0f, 1f, camera));
    DimensionWrapper w = world.wrapper();
    w.setCoordSyst(camera.getParent());
    camera.move(0f, 0f, w.getMax().getZ() + 1f);
    camera.getRenderingSurface().render();
    setCursorVisible();
  }
  /**
   * Sets the view to a left view.
   */
  protected void leftView() {
    world.lookAt(new Vector(-1f, 0f, 0f, camera));
    DimensionWrapper w = world.wrapper();
    w.setCoordSyst(camera.getParent());
    camera.move(0f, 0f, w.getMax().getZ() + 1f);
    camera.getRenderingSurface().render();
    setCursorVisible();
  }
  /**
   * Sets the view to a right view.
   */
  protected void rightView() {
    world.lookAt(new Vector(1f, 0f, 0f, camera));
    DimensionWrapper w = world.wrapper();
    w.setCoordSyst(camera.getParent());
    camera.move(0f, 0f, w.getMax().getZ() + 1f);
    camera.getRenderingSurface().render();
    setCursorVisible();
  }
  /**
   * Sets the view to an upper view.
   */
  protected void upperView() {
    world.lookAt(new Vector(0f, 1f, 0f, camera));
    DimensionWrapper w = world.wrapper();
    w.setCoordSyst(camera.getParent());
    camera.move(0f, 0f, w.getMax().getZ() + 1f);
    camera.getRenderingSurface().render();
    setCursorVisible();
  }
  /**
   * Sets the view to a lower view.
   */
  protected void lowerView() {
    world.lookAt(new Vector(0f, -1f, 0f, camera));
    DimensionWrapper w = world.wrapper();
    w.setCoordSyst(camera.getParent());
    camera.move(0f, 0f, w.getMax().getZ() + 1f);
    camera.getRenderingSurface().render();
    setCursorVisible();
  }
  
  /*
  public abstract int getMode();
  public abstract void setMode(int newMode);
  */
	
  private boolean magnetic = false;
  /**
   * Checks if the  magnetic grid is used. This do not take into account the state of the
   * control key.
   * @return true if the magnetic grid is on
   */
  public boolean isMagnetic() { return magnetic; }
  /**
   * Sets if the  magnetic grid is used.
   * @param true to set the magnetic grid on
   */
  public void setMagnetic(boolean b) {
    magnetic = b;
  }
  
  private float step;
  /**
   * Gets the grid magnetic step.
   * @return the grid magnetic step
   */
  protected float getStep() { return step; }
  /**
   * Sets the grid magnetic step. You should override this method in order to call the
   * setStep(float) method of your handles.
   * @param the new grid magnetic step
   */
  protected void setStep(float f) {
    step = f;
    Editor.getOptions().magneticStep = step;
    cursor.setStep(step);
  }
  
  private float ax, ay;
  private MouseEvent lastMouseEvent;
  protected int mouseButton(MouseEvent e) {
    if((e.getModifiers() & MouseEvent.BUTTON3_MASK) == MouseEvent.BUTTON3_MASK) return 3;
    if((e.getModifiers() & MouseEvent.BUTTON2_MASK) == MouseEvent.BUTTON2_MASK) return 2;
    if((e.getModifiers() & MouseEvent.BUTTON1_MASK) == MouseEvent.BUTTON1_MASK) return 1;
    return 0;
  }
  /**
   * Called when the mouse is moved, and move the 3D cursor according to the new mouse location.
   * May be overriden.
   * @param e the mouse event
   */
  protected void mouseMoved(MouseEvent e) {
    boolean buttonPressed = (mouseButton(e) == 1);
    if(cursor.isPressed()) lock();
    
    if(e.isControlDown()) cursor.setMagnetic(!magnetic);
    else cursor.setMagnetic(magnetic);
    
    Position p  = cursor.clone(camera);
    float sw = (float) renderingCanvas.getWidth(), sh = (float) renderingCanvas.getHeight();
    float x, y;
    if(camera.isOrtho()) {
      x =  (((float) e.getX()) - sw / 2) / sw * 7;
      y = -(((float) e.getY()) - sh / 2) / sh * 7;
    }
    else {
      x = (((float)  e.getX()) - sw / 2f) * (p.getZ() * (float) Math.tan((double) camera.getFOV())) / sh; // sic !
      y = (((float) -e.getY()) + sh / 2f) * (p.getZ() * (float) Math.tan((double) camera.getFOV())) / sh;
    }
    
    Position p2 = new Point(x, y, p.getZ(), camera);
    cursor.move(e, p2);
    
    int modifiers = e.getModifiers();
    
    if(cursor.isPressed()) unlock();
    else render();
    
    x =  (((float) e.getX()) - sw / 2) / sw * 7;
    y = -(((float) e.getY()) - sh / 2) / sh * 7;
    ax = x; ay = y; lastMouseEvent = e;
  }
  /**
   * Called when the mouse is dragged, and move the 3D cursor according to the new mouse location.
   * May be overriden.
   * @param e the mouse event
   */
  protected void mouseDragged(MouseEvent e) {
    int modifiers = e.getModifiers();
    
    //if((modifiers & MouseEvent.BUTTON3_MASK) == MouseEvent.BUTTON3_MASK) {
    if(mouseButton(e) == 3) {
      cursor.setMagnetic(false);
    }
    else {
      if(e.isControlDown()) cursor.setMagnetic(!magnetic);
      else cursor.setMagnetic(magnetic);
    }
    
    if(mouseButton(e) == 1) lock();
    //if((modifiers & MouseEvent.BUTTON1_MASK) == MouseEvent.BUTTON1_MASK) lock();
    
    Position p  = cursor.clone(camera);
    float sw = (float) renderingCanvas.getWidth(), sh = (float) renderingCanvas.getHeight();
    float x, y;
    if(camera.isOrtho()) {
      x =  (((float) e.getX()) - sw / 2) / sw * 7;
      y = -(((float) e.getY()) - sh / 2) / sh * 7;
    }
    else {
      x = (((float)  e.getX()) - sw / 2f) * (p.getZ() * (float) Math.tan((double) camera.getFOV())) / sh; // sic !
      y = (((float) -e.getY()) + sh / 2f) * (p.getZ() * (float) Math.tan((double) camera.getFOV())) / sh;
    }
    
    Position p2 = new Point(x, y, p.getZ(), camera);
    cursor.move(e, p2);
    
    //if((modifiers & MouseEvent.BUTTON3_MASK) == MouseEvent.BUTTON3_MASK) {
    if(mouseButton(e) == 3) {
      x =  (((float) e.getX()) - sw / 2) / sw * 7;
      y = -(((float) e.getY()) - sh / 2) / sh * 7;
      p = cursor.clone(scene);
      toRotate.rotateLateral ((x - ax) * 60f);
      toRotate.rotateVertical((ay - y) * 60f);
      cursor.move(p);
    }
    //if((modifiers & MouseEvent.BUTTON1_MASK) == MouseEvent.BUTTON1_MASK) unlock();
    if(mouseButton(e) == 1) unlock();
    else render();
    
    x =  (((float) e.getX()) - sw / 2) / sw * 7;
    y = -(((float) e.getY()) - sh / 2) / sh * 7;
    ax = x; ay = y; lastMouseEvent = e;
  }
  /**
   * Called when the mouse is pressed, and press the 3D cursor.
   * May be overriden.
   * @param e the mouse event
   */
  protected void mousePressed(MouseEvent e) {
    //if((e.getModifiers() & MouseEvent.BUTTON1_MASK) == MouseEvent.BUTTON1_MASK) cursor.press(e);
    if(mouseButton(e) == 1) cursor.press(e);
    render();
  }
  /**
   * Called when the mouse is clicked.
   * May be overriden.
   * @param e the mouse event
   */
  protected void mouseClicked(MouseEvent e) {
    //if((e.getModifiers() & MouseEvent.BUTTON2_MASK) == MouseEvent.BUTTON2_MASK) cursor.hold(e, pseudoZVector());
    if(mouseButton(e) == 2) cursor.hold(e, pseudoZVector());
    render();
  }
  /**
   * Called when the mouse is released, and release the 3D cursor.
   * May be overriden.
   * @param e the mouse event
   */
  protected void mouseReleased(MouseEvent e) {
    //if((e.getModifiers() & MouseEvent.BUTTON1_MASK) == MouseEvent.BUTTON1_MASK) cursor.release(e);
    if(mouseButton(e) == 1) cursor.release(e);
    render();
  }
  /**
   * Called when a key is pressed.
   * May be overriden.
   * @param e the mouse event
   */
  public void keyPressed(KeyEvent e) {
    Position p;
    Vector v;
    switch(e.getKeyCode()) {
    case KeyEvent.VK_NUMPAD0: resetView(); break;
    case KeyEvent.VK_NUMPAD5: frontView(); break;
    case KeyEvent.VK_NUMPAD4: leftView (); break;
    case KeyEvent.VK_NUMPAD6: rightView(); break;
    case KeyEvent.VK_NUMPAD8: upperView(); break;
    case KeyEvent.VK_NUMPAD2: lowerView(); break;
    case KeyEvent.VK_LEFT:
      v = new Vector(-.5f, 0f, 0f, camera);
      //p = cursor.clone(cursor.getRootParent());
      //camera.setX(camera.getX() - .5f);
      camera.addVector(v);
      //p     .setX(p     .getX() - .5f);
      //cursor.move(p);
      cursor.addVector(v);
      render();
      break;
    case KeyEvent.VK_RIGHT:
      v = new Vector(.5f, 0f, 0f, camera);
      camera.addVector(v);
      cursor.addVector(v);
      render();
      break;
    case KeyEvent.VK_UP:
      v = new Vector(0f, .5f, 0f, camera);
      camera.addVector(v);
      cursor.addVector(v);
      render();
      break;
    case KeyEvent.VK_DOWN:
      v = new Vector(0f, -.5f, 0f, camera);
      camera.addVector(v);
      cursor.addVector(v);
      render();
      break;
    case KeyEvent.VK_ADD:
      if(camera.isOrtho()) {
        camera.setFOV(camera.getFOV() - 5f);
      }
      else {
        v = new Vector(0f, 0f, -.5f, camera);
        camera.addVector(v);
        cursor.addVector(v);
        setCursorVisible();
      }
      render();
      break;
    case KeyEvent.VK_SUBTRACT:
      if(camera.isOrtho()) {
        camera.setFOV(camera.getFOV() + 5f);
      }
      else {
        v = new Vector(0f, 0f, .5f, camera);
        camera.addVector(v);
        cursor.addVector(v);
        setCursorVisible();
      }
      render();
      break;
    case KeyEvent.VK_A:
    case KeyEvent.VK_PAGE_UP:
      v = pseudoZVector();
      v.oppose();
      if(cursor.isPressed()) lock();
      cursor.move(lastMouseEvent, v);
      if(cursor.isPressed()) unlock();
      render();
      break;
    case KeyEvent.VK_Q:
    case KeyEvent.VK_PAGE_DOWN:
      v = pseudoZVector();
      if(cursor.isPressed()) lock();
      cursor.move(lastMouseEvent, v);
      if(cursor.isPressed()) unlock();
      render();
      break;
    case KeyEvent.VK_Z:
      for(int i = 0; i < 15; i++) renderingCanvas.display(); // immediate rendering.
      break;
    case KeyEvent.VK_W:
      for(int i = 0; i < 15; i++) { lock(); unlock(); }
      break;
    case KeyEvent.VK_C:
      camera.setOrtho(!camera.isOrtho());
      cameraOrtho.setState (camera.isOrtho());
      if(!camera.isOrtho()) camera.setFOV(90f);
      render();
      break;
    }
  }
  private class CursorMover extends MouseAdapter implements MouseMotionListener, KeyListener {
    public CursorMover(Component c) {
      this.c = c;
      c.addMouseListener(this);
      c.addMouseMotionListener(this);
      c.addKeyListener(this);
    }
    private Component c;
    public void mouseMoved   (MouseEvent e) { GUIEditor.this.mouseMoved   (e); }
    public void mouseDragged (MouseEvent e) { GUIEditor.this.mouseDragged (e); }
    public void mousePressed (MouseEvent e) { GUIEditor.this.mousePressed (e); }
    public void mouseReleased(MouseEvent e) { GUIEditor.this.mouseReleased(e); }
    public void mouseClicked (MouseEvent e) { GUIEditor.this.mouseClicked (e); }
    public void keyPressed   (KeyEvent   e) { GUIEditor.this.keyPressed   (e); }
    public void keyReleased  (KeyEvent   e) {  }
    public void keyTyped     (KeyEvent   e) {  }
  }
  
  protected Vector pseudoZVector() {
    Vector v;
    if(camera.isOrtho()) {
      v = new Vector(0f, 0f, -step, camera);
    }
    else{
      v = new Vector(cursor, camera);
      v.setCoordSyst(camera);
      v.scale(step / v.length());
      if(cursor.clone(camera).getZ() > 0f) v.oppose();
      if(v.isNullVector()) v.move(0f, 0f, -step);
    }
    return v;
  }
  public void setCursorVisible() {
    Position p = cursor.clone(camera);
    if(p.getZ() > -.2f) {
      p.setZ(-.2f);
      cursor.move(p);
    }
  }
  
  /** The rendering canvas. */
  protected RenderingCanvas renderingCanvas;
  /** The world, in which you must add the element you want to edit. */
  protected World3D         world  = new World3D();
  public World3D getEditionWorld() { return world; }
  /** The scene. */
  protected Environment3D   scene  = new Environment3D();
  /** The camera. */
  protected Camera3D        camera = new Camera3D();
  /** The 3D cursor. */
  protected Cursor3D        cursor;

  protected int lockLevel;
  /**
   * Locks the refreshing of the 3D view.
   */
  public synchronized void lock() { lockLevel++; }
  /**
   * Unlocks the refreshing of the 3D view.
   */
  public synchronized void unlock() {
    if(lockLevel <= 0) return;
    if(--lockLevel == 0) render();
  }
  /**
   * Checks if the refreshing of the 3D view is locked.
   * @return true if locked
   */
  public synchronized boolean isLocked() { return lockLevel > 0; }
  
  public void render() {
    //renderingCanvas.display();
    renderingCanvas.render();
  }
  
  private Label statusBar = new Label();
  /**
   * Shows the given text into the satus bar.
   * @param s the text
   */
  protected void aff(String s) {
    System.out.println(s);
    statusBar.setText(s);
  }
}
