/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.editor;

import opale.soya.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import opale.soya.util.*;
import java.util.*;
import java.lang.Math;

/**
 * A collection of handles. Contains usefull method for manipulating handles by group.
 * 
 * @author Artiste on the Web
 */

public class Handles extends opale.soya.util.Bag {
  /**
   * Creates a new handles.
   */
  public Handles() {  }
  
  /**
   * Removes all the handles that edit the given object.
   * @param o the object for which you want to remove all the handles.
   */
  public void removeHandlesFor(Object o) {
    for(Iterator i = c.iterator(); i.hasNext(); ) {
      Handle3D h = (Handle3D) i.next();
      if(h.getEditedObject() == o) {
        i.remove();
        fireRemove(h);
      }
    }
  }
  /**
   * Removes all the handles that edit an object in the given collection.
   * @param os the collection of objects for which you want to remove all the handles.
   */
  public void removeHandlesFor(java.util.Collection os) {
    for(Iterator i = c.iterator(); i.hasNext(); ) {
      Handle3D h = (Handle3D) i.next();
      for(Iterator j = os.iterator(); j.hasNext(); ) {
        if(h.getEditedObject() == j.next()) {
          i.remove();
          fireRemove(h);
        }
      }
    }
  }
  
  /**
   * Gets the nearest handle of the given position.
   * @return the handle
   */
  public Handle3D getNearestHandle(Position p) {
    Handle3D best = null, h;
    float bestLength = Float.POSITIVE_INFINITY, length;
    for(Iterator i = iterator(); i.hasNext(); ) {
      h = (Handle3D) i.next();
      length = p.squareDistanceTo(h);
      if(length < bestLength) {
        best = h;
        bestLength = length;
      }
    }
    return best;
  }
  /**
   * Gets the nearest selected handle of the given position.
   * @return the handle
   */
  public Handle3D getNearestSelectedHandle(Position p) {
    Handle3D best = null, h;
    float bestLength = Float.POSITIVE_INFINITY, length;
    for(Iterator i = iterator(); i.hasNext(); ) {
      h = (Handle3D) i.next();
      if(h.isSelected()) {
        length = p.squareDistanceTo(h);
        if(length < bestLength) {
          best = h;
          bestLength = length;
        }
      }
    }
    return best;
  }
  /**
   * Gets the nearest unselected handle of the given position.
   * @return the handle
   */
  public Handle3D getNearestUnselectedHandle(Position p) {
    Handle3D best = null, h;
    float bestLength = Float.POSITIVE_INFINITY, length;
    for(Iterator i = iterator(); i.hasNext(); ) {
      h = (Handle3D) i.next();
      if(!h.isSelected()) {
        length = p.squareDistanceTo(h);
        if(length < bestLength) {
          best = h;
          bestLength = length;
        }
      }
    }
    return best;
  }
  /**
   * Gets the nearest clicked handle of the given position.
   * @return the handle
   */
  public Handle3D getNearestClickedHandle(Position p) {
    Handle3D best = null, h;
    float bestLength = Float.POSITIVE_INFINITY, length;
    for(Iterator i = iterator(); i.hasNext(); ) {
      h = (Handle3D) i.next();
      if(h.isClicked()) {
        length = p.squareDistanceTo(h);
        if(length < bestLength) {
          best = h;
          bestLength = length;
        }
      }
    }
    return best;
  }
  /**
   * Gets the nearest unclicked handle of the given position.
   * @return the handle
   */
  public Handle3D getNearestUnclickedHandle(Position p) {
    Handle3D best = null, h;
    float bestLength = Float.POSITIVE_INFINITY, length;
    for(Iterator i = iterator(); i.hasNext(); ) {
      h = (Handle3D) i.next();
      if(!h.isClicked()) {
        length = p.squareDistanceTo(h);
        if(length < bestLength) {
          best = h;
          bestLength = length;
        }
      }
    }
    return best;
  }
  /**
   * Gets the all the clicked handles.
   * @return the clicked handles
   */
  public java.util.Collection getClickedHandles() {
    java.util.Collection c = new java.util.Vector();
    for(Iterator i = iterator(); i.hasNext(); ) {
      Handle3D h = (Handle3D) i.next();
      if(h.isClicked()) c.add(h);
    }
    return c;
  }
  /**
   * Gets the all the selected handles.
   * @return the selected handles
   */
  public java.util.Collection getSelectedHandles() {
    java.util.Collection c = new java.util.Vector();
    for(Iterator i = iterator(); i.hasNext(); ) {
      Handle3D h = (Handle3D) i.next();
      if(h.isSelected()) c.add(h);
    }
    return c;
  }
  /**
   * Gets the all the objects that edit the clicked handles.
   * @return the objects
   */
  public java.util.Collection getClickedObjects() {
    java.util.Collection c = new java.util.Vector();
    for(Iterator i = iterator(); i.hasNext(); ) {
      Handle3D h = (Handle3D) i.next();
      if(h.isClicked()) c.add(h.getEditedObject());
    }
    return c;
  }
  /**
   * Gets the all the objects that are edited by selected handles.
   * @return the objects
   */
  public java.util.Collection getSelectedObjects() {
    java.util.Collection c = new java.util.Vector();
    for(Iterator i = iterator(); i.hasNext(); ) {
      Handle3D h = (Handle3D) i.next();
      if(h.isSelected()) c.add(h.getEditedObject());
    }
    return c;
  }

  /**
   * Gets the all the handles that edit the given object.
   * @return the handles
   */
  public java.util.Collection getHandlesFor(Object editedObject) {
    java.util.Collection c = new java.util.Vector();
    for(Iterator i = iterator(); i.hasNext(); ) {
      Handle3D h = (Handle3D) i.next();
      if(h.getEditedObject() == editedObject) c.add(h);
    }
    return c;
  }

  /**
   * Adds all the handles into the given world.
   * @param w the world
   */
  public void addIntoWorld(World3D w) {
    for(Iterator i = iterator(); i.hasNext(); ) w.add((Element3D) i.next());
  }
  
  private boolean visible = true;
  private java.util.Collection visibleIsForObjects;
  private void setCurrentVisibilityToHandle(Handle3D h) {
    if(visibleIsForObjects == null) h.setVisible(visible);
    else {
      Object testedObject = h.getEditedObject();
      for(Iterator j = visibleIsForObjects.iterator(); j.hasNext(); ) {
        if(testedObject == j.next()) {
          h.setVisible(visible);
          return;
        }
      }
      h.setVisible(!visible); // not found in the given collection forObjects.
    }
  }
  /**
   * Sets the visibility for all the handles, including handles that will be added later.
   * @param b the visibility
   */
  public void setVisible(boolean b) {
    visible = b;
    visibleIsForObjects = null;
    for(Iterator i = iterator(); i.hasNext(); ) ((Volume3D) i.next()).setVisible(b);
  }
  /**
   * Sets visible all the handles that edit one object in the given collection.
   * @param b the visibility for handles that edit one object in the given collection; the other
   * handles will have an opposite visibility ( !b )
   */
  public void setVisibleHandlesFor(boolean b, java.util.Collection forObjects) {
    visible = b;
    visibleIsForObjects = forObjects;
    firstFor:
    for(Iterator i = iterator(); i.hasNext(); ) {
      Handle3D h = (Handle3D) i.next();
      Object testedObject = h.getEditedObject();
      for(Iterator j = forObjects.iterator(); j.hasNext(); ) {
        if(testedObject == j.next()) {
          h.setVisible(b);
          continue firstFor;
        }
      }
      h.setVisible(!b); // not found in the given collection forObjects.
    }
  }

  private float scal = 1f;
  /**
   * Gets the scale of these handles.
   * @return the scale
   */
  public float getScale() { return scal; }
  /**
   * Sets the scale of these handles.
   * @param f the new scale
   */
  public void setScale(float f) {
    scal = f;
    for(Iterator i = iterator(); i.hasNext(); ) setScaleToHandle((Handle3D) i.next());
  }
  /**
   * Scales the given handle.
   * @param h the handle
   */
  protected void setScaleToHandle(Handle3D h) {
    h.scale(scal / h.getXFactor(), scal / h.getYFactor(), scal / h.getZFactor());
    //h.scale(scal);
    h.setStep(scal);
  }

  /**
   * Adds the given handle.
   * @param h the handle
   */
  public boolean add(Handle3D h) {
    boolean b = super.add(h);
    setScaleToHandle(h);
    setCurrentVisibilityToHandle(h);
    return b;
  }
  /**
   * Adds all handles in the given collection.
   * @param h the handles
   */
  public boolean addAll(java.util.Collection c) {
    boolean b = super.addAll(c);
    for(Iterator i = c.iterator(); i.hasNext(); ) {
      Handle3D h = (Handle3D) i.next();
      setScaleToHandle(h);
      setCurrentVisibilityToHandle(h);
    }
    return b;
  }
  /** Not supported. Handles can contain only Handle3D. */
  public boolean add(Object o) { throw new UnsupportedOperationException("Only handle can be added or removed."); }
  /**
   * Removes the given handle.
   * @param h the handle
   */
  public boolean remove(Handle3D h) { return super.remove(h); }
  /** Not supported. Handles can contain only Handle3D. */
  public boolean remove(Object o) { throw new UnsupportedOperationException("Only handle can be added or removed."); }
}
