/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.editor;

import opale.soya.*;
import java.beans.*;
import java.io.*;
import java.awt.*;

/**
 * Abstract editor for all object property. Allow to set the property to a new object, or to
 * edit the current object.
 * 
 * For example, see the texture property of the soya.soya2d.Material class, and the
 * soya.soya2d.TextureEditor class.
 * 
 * @author Artiste on the Web
 */

public abstract class SubPropertyEditor extends PropertyEditorSupport {
  /**
   * Creates a new sub-property editor.
   * @param avalaibleClassNames an array of class name avalaible for this property.
   */
  public SubPropertyEditor(String[] avalaibleClassNames, boolean allowNull,boolean allowOther) {
    super();
    int nb = avalaibleClassNames.length;
    if(allowNull ) nb++;
    if(allowOther) nb++;
    tags = new String[nb];
    System.arraycopy(avalaibleClassNames, 0, tags, 0, avalaibleClassNames.length);
    nb = avalaibleClassNames.length;
    if(allowNull ) tags[nb++] = NULL;
    if(allowOther) tags[nb++] = OTHER;
  }

  private static final String NULL  = "(null)"    ;
  private static final String OTHER = "(other...)";

  /**
   * Gets the property's value as text. The text is the name of the object's class.
   * @return the property's value as text
   */
  public synchronized String getAsText() {
    Object o = getValue();
    if(o == null) return NULL;
    else return o.getClass().getName();
  }
	
  /**
   * Sets the property's value as text. The text must be a class name. The property will be set
   * to a new instance of this class.
   * @param the property's new value class name
   */
  public synchronized void setAsText(String t) throws IllegalArgumentException {
    if(NULL.equals(t)) setValue(null);
    else {
      if(OTHER.equals(t)) {
        try { setValue(ClassChooser.question(getFrame()).newInstance()); }
        catch(Exception e) {  e.printStackTrace(); throw new IllegalArgumentException("Can't create object from this class."); }
      }
      else {
        try { setValue(Class.forName(t).newInstance()); }
        catch(Exception e) {  e.printStackTrace(); throw new IllegalArgumentException("Can't create object from class : " + t); }
      }
    }
  }
  private java.awt.Frame getFrame() {
    Component c = getCustomEditor();
    while(true) {
      c = c.getParent();
      if(c instanceof java.awt.Frame) return (java.awt.Frame) c;
    }
  }

  public synchronized void setValue(Object o) { super.setValue(o); }

  /**
   * Edits the sub-property as a bean.
   */
  public synchronized void editSubProperty() throws IntrospectionException, InstantiationException, IllegalAccessException {
    Object o = getValue();
    if(o != null) Editor.edit(o);
  }
  
  private String[] tags;
  /**
   * Gets the tags. The tags is an array of class name avalaible for this property.
   * @return the tags
   */
  public String[] getTags() { return tags; }

  /**
   * Custom edition is supported.
   * @return true
   */
  public boolean supportsCustomEditor() { return true; }
  private Component customEditor;
  /**
   * Recovers the custom editor.
   * @return the custom editor
   */
  public Component getCustomEditor() {
    if(customEditor == null) customEditor = new PropertySubProperty(this);
    return customEditor;
  }
}
