/*
 * Soya3D
 * Copyright (C) 1999  Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya2d;

import opale.soya.*;
import gl4java.*;
import java.util.*;
import java.awt.*;
import java.awt.image.*;

/**
 * A abstract class for all textures that have a mask (in addition to their image).
 *
 * Such a texture is defined by 2 images : the image itself and the mask. The mask is a 
 * monochrome image (if not, only the red part is read and used) that represents the alpha
 * channel : a white pixel means that the equivalent pixel in the first image is opaque, and
 * a black pixel means transparency. Gray pixels correspond to semi-transparency.
 *
 * Notice that both images must have the same size.
 *
 * @author Artiste on the Web
 */

public abstract class MaskedTexture extends Texture {
  /**
   * Creates a new texture.
   */
  public MaskedTexture() {  }
  /**
   * Creates a new texture from the given image and mask.
   * @param image the image
   * @param mask  the mask
   */
  public MaskedTexture(java.awt.Image image, java.awt.Image mask) throws InterruptedException {
    setImage(image);
    setMask (mask );
  }
  /**
   * Creates a new texture from the given image and mask file name.
   * @param imageFileName the image file name
   * @param maskFileName  the mask file name
   */
  public MaskedTexture(String imageFileName, String maskFileName) throws InterruptedException {
    setImageFileName(imageFileName);
    setMaskFileName (maskFileName );
  }

  protected String maskFileName = "";
  /**
   * Gets the mask file name, or null if it has no attached file name. It is not the file
   * where this object is stored, but the file name of the mask file (.gif,...) from
   * witch it was created.
   * @return the mask file name.
   */
  public String getMaskFileName() { return maskFileName; }
  /**
   * Sets the mask file name. The mask will be loaded into this texture.
   * @param fileName the mask file name.
   */
  public void setMaskFileName(String fileName) throws InterruptedException {
    maskFileName = fileName;
    update();
    firePropertyChange("maskFileName");
  }
  /**
   * Gets an awt.image that correspond to this opengl mask. The result id null if no file
   * name is associated with this mask.
   * @return the mask image or null
   */
  public java.awt.Image getMask() { return (new Canvas()).getToolkit().getImage(maskFileName); }
  /**
   * Defines the opengl mask from an awt.image.
   * If the image is loaded from a file, use rather setMaskFileName(String) .
   * @param an awt.image
   */
  public synchronized void setMask(java.awt.Image i) throws InterruptedException {
    MediaTracker mt = new MediaTracker(new Canvas());
    mt.addImage(i, 0);
    mt.waitForAll();
    int width  = i.getWidth (null);
    int height = i.getHeight(null);
    if((width != getWidth()) || (height != getHeight())) throw new BadImageException("Mask and image should have the same dimension. The image should also be defined before the mask. Mask : " + width + " * " + height + "; image : " + getWidth() + " * " + getHeight());
    byte[] maskData = dataFromImageRGBA(i);
    byte[] data = getImageData();
    for(int k = 0; k < getWidth() * getHeight(); k++) data[k * 4 + 3] = maskData[k * 4];
    firePropertyChange("maskFileName");
    reBuild();
  }
  
  protected byte[] computeImageToData(java.awt.Image i) throws InterruptedException {
    return dataFromImageRGBA(i);
  }
  
  // Overrides :
  public void update() throws InterruptedException {
    setImage(getImage());
    java.awt.Image mask = getMask();
    if(mask != null) setMask(mask);
  }
}
