/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya2d;

import opale.soya.*;
import opale.soya.util.*;
import gl4java.*;
import java.util.*;
import java.io.*;
import java.lang.ref.*;
import java.beans.*;

/**
 * A material contains all the properties that describe how to draw a surface : color,
 * specular, shininess, texture,...
 *
 * Materials implement NamedObject and StoredInPathObject, so they can have a name and they
 * can be stored into a path (defined by the Material.path field). The material file
 * extension is ".material" .
 * 
 * @author Artiste on the Web
 */

public class Material extends AbstractBean implements StoredInPathObject {
  private static final long serialVersionUID = 1543539363062794588l;

  /**
   * Creates a new material.
   */
  public Material() { super(); }
  /**
   * Creates a new textured material.
   * @param t the texture
   */
  public Material(Texture t) {
    texture = t;
    shininess = 30;
    defineUseAlpha();
  }
  /**
   * Creates a new colored material.
   * @param newColor the color
   */
  public Material(float[] newColor) { this((Texture) null, 0, newColor, newColor); }
  /**
   * Creates a new colored material.
   * @param newShininess the shininess (should range between 0f and 1f).
   * @param newColor the color
   */
  public Material(float newShininess, float[] newColor) { this((Texture) null, newShininess, newColor, newColor); }
  /**
   * Creates a new colored material, with different diffuse and specular color.
   * @param newShininess the shininess (should range between 0f and 1f).
   * @param newDiffuse the diffuse color
   * @param newSpecular the specular color
   */
  public Material(float newShininess, float[] newDiffuse, float[] newSpecular) { this((Texture) null, newShininess, newDiffuse, newSpecular); }
  /**
   * Creates a new textured and colored material.
   * @param t the texture
   * @param newShininess the shininess (should range between 0f and 1f).
   * @param newDiffuse the diffuse color
   * @param newSpecular the specular color
   */
  public Material(Texture t, float newShininess, float[] newDiffuse, float[] newSpecular) {
    texture = t;
    shininess = newShininess;
    System.arraycopy(newDiffuse, 0, diffuse, 0, 4);
    System.arraycopy(newSpecular, 0, specular, 0, 4);
    defineUseAlpha();
  }
  /**
   * Creates a new material.
   * @param n the material name
   */
  public Material(String n) {
    this();
    setName(n);
  }
  /**
   * Creates a new textured material.
   * @param n the material name
   * @param t the texture
   */
  public Material(String n, Texture t) {
    this(t);
    setName(n);
  }
  /**
   * Creates a new colored material.
   * @param n the material name
   * @param newColor the color
   */
  public Material(String n, float[] newColor) {
    this(newColor);
    setName(n);
  }
  /**
   * Creates a new colored material.
   * @param n the material name
   * @param newShininess the shininess (should range between 0f and 1f).
   * @param newColor the color
   */
  public Material(String n, float newShininess, float[] newColor) {
    this(newShininess, newColor);
    setName(n);
  }
  /**
   * Creates a new colored material, with different diffuse and specular color.
   * @param n the material name
   * @param newShininess the shininess (should range between 0f and 1f).
   * @param newDiffuse the diffuse color
   * @param newSpecular the specular color
   */
  public Material(String n, float newShininess, float[] newDiffuse, float[] newSpecular) {
    this(newShininess, newDiffuse, newSpecular);
    setName(n);
  }
  /**
   * Creates a new textured and colored material.
   * @param n the material name
   * @param t the texture
   * @param newShininess the shininess (should range between 0f and 1f).
   * @param newDiffuse the diffuse color
   * @param newSpecular the specular color
   */
  public Material(String n, Texture t, float newShininess, float[] newDiffuse, float[] newSpecular) {
    this(t, newShininess, newDiffuse, newSpecular);
    setName(n);
  }
  
  public Object clone() {
    return new Material(name, texture, shininess, diffuse, specular);
  }
  
  // stuff for serialization : allows other classes to serialize Material object.
  /**
   * Writes this material into the given output stream. If it has no name, it will be fully
   * written into the stream, else just the name will be written. It allow you to use the same
   * material in different files.
   * Use read(ObjectOutputStream) to read a material written by write(ObjectOutputStream).
   * @see opale.soya.soya2d.Material#read
   * @param s the ObjectOutputStream
   */
  public void write(ObjectOutputStream s) throws IOException {
    if(hasName()) {
      s.writeInt(OBJECT_SAVED_IN_PATH);
      s.writeUTF(name);
    }
    else {
      if(this == WHITE_MATERIAL) s.writeInt(NO_OBJECT_SAVED);
      else {
        s.writeInt(OBJECT_SAVED_HERE);
        s.writeObject(this);
      }
    }
  }
  /**
   * Writes the given material into the given output stream. If it has no name, it will be
   * fully written into the stream, else just the name will be written. It allow you to use the
   * same material in different files.
   * Use read(ObjectOutputStream) to read a material written by write(ObjectOutputStream).
   * @see opale.soya.soya2d.Material#read
   * @param s the ObjectOutputStream
   * @param m the material (may be null, which is considered as the same than NO_MATERIAL)
   */
  public static void write(ObjectOutputStream s, Material m) throws IOException {
    if((m == null) || (m != WHITE_MATERIAL)) s.writeInt(NO_OBJECT_SAVED);
    else m.write(s);
  }
  /**
   * Reads a material from the given output stream.
   * The material must have been written by one of the write method.
   * If just the name of the material was written, it can work only if the material path
   * contains a material file with the same name : the material will be recovered exactely as
   * the get method does.
   * @see opale.soya.soya2d.Material#write
   * @see opale.soya.soya2d.Material#get
   * @param s the ObjectOutputStream
   * @return the material
   */
  public static Material read(ObjectInputStream s) throws IOException, ClassNotFoundException {
    switch(s.readInt()) {
    case NO_OBJECT_SAVED: return WHITE_MATERIAL;
    case OBJECT_SAVED_IN_PATH: return get(s.readUTF());
    case OBJECT_SAVED_HERE: return (Material) s.readObject();
    }
    return WHITE_MATERIAL;
  }

  private void readObject(ObjectInputStream s) throws IOException, ClassNotFoundException { // Initialization.
    s.defaultReadObject();
    softReferenceToThis = new SoftReference(this);
    if(hasName()) materials.add(softReferenceToThis);
  }

  /**
   * The material path. Material files are named acoording to the material name and
   * with the ".material" extension, and they should be all located in the material
   * path. You can use the get(String) method to load a material of the given name; it
   * must be saved in the material path.
   * You can use the load(String) method to load a non-named material, from any path.
   * The path must ends with "/" .
   */
  public static String path;
  /**
   * Set the materials path.
   * @see opale.soya.soya2d.Material#path
   * @param p the new path
   */
  public static void setPath(String p) {
    if(p.endsWith("/")) path = p;
    else path = p + "/";
  }
  /**
   * Saves the material. The file name will be getName() + ".material", and it will be saved
   * in the material path.
   */
  public synchronized void save() throws java.io.IOException { save(path + name + ".material"); }
  /**
   * Saves the material with the given file name.
   * @param fileName the file name
   */
  public synchronized void save(String fileName) throws java.io.IOException {
    try {
      ObjectOutputStream o = new ObjectOutputStream(new BufferedOutputStream(new FileOutputStream(fileName)));
      o.writeObject(this);
      o.flush();
      o.close();
    }
    catch(Exception e) { throw (java.io.IOException) e; }
  }
  /**
   * Saves the given material. The file name will be m.getName() + ".material", and it will be saved
   * in the material path. 
   * @param m the material
   */
  public static void save(Material m) throws java.io.IOException { m.save(); }
  /**
   * Saves the given material with the given file name.
   * @param m the material
   * @param fileName the file name
   */
  public static void save(Material m, String fileName) throws java.io.IOException { m.save(fileName); }
  /**
   * Loads and returns the given material file.
   * @param fileName the material file name
   * @return the material
   */
  public static Material load(String fileName) throws java.io.IOException, java.lang.ClassNotFoundException {
    Material m;
    ObjectInputStream i = new ObjectInputStream(new BufferedInputStream(new FileInputStream(fileName)));
    m = (Material) i.readObject();
    i.close();
    return m;
  }
  private static final java.util.Set materials = new java.util.HashSet();
  /**
   * Gets the material from its name. If a material with this name already exist, it will be
   * returned; else it will be search and load in the material path.
   * @param name the material name
   * @return the material
   */
  public static Material get(String name) throws java.io.IOException, java.lang.ClassNotFoundException {
    Material m;
    for(Iterator i = materials.iterator(); i.hasNext(); ) {
      m = (Material) ((SoftReference) i.next()).get();
      if(m == null) i.remove();
      else {
        if(m.name.equals(name)) return m;
      }
    }
    return load(path + name + ".material");
  }


  /**
   * The white color.
   */
  public static final float[] WHITE_COLOR = { 1f, 1f, 1f, 1f };
  /**
   * The black color.
   */
  public static final float[] BLACK_COLOR = { 0f, 0f, 0f, 1f };
  /**
   * The transparent color.
   */
  public static final float[] TRANSPARENT_COLOR = { 0f, 0f, 0f, 0f };
  /**
   * The absence of color.
   */
  public static final float[] NO_COLOR = { 1f, 1f, 1f, -1f };

  private String name;
  public synchronized boolean hasName() { return name != null && !"".equals(name); }
  public String getName() { return name; }
  public synchronized void setName(String newName) {
    String oldName = name;
    boolean oldHasName = hasName();
    name = newName;
    if(hasName() && !oldHasName) materials.add(softReferenceToThis);
    else {
      if(!hasName() && oldHasName) materials.remove(softReferenceToThis);
    }
    firePropertyChange("name", oldName, name);
  }
  private transient SoftReference softReferenceToThis = new SoftReference(this);

  private Texture texture;
  /**
   * Gets the texture of this material. Default is null.
   * @return the texture
   */
  public Texture getTexture() { return texture; }
  /**
   * Sets the texture of this material.
   * @param t the new texture
   */
  public synchronized void setTexture(Texture t) {
    if((texture != null) && (changeListeners != null)) {
      synchronized(changeListeners) {
        for(Iterator i = changeListeners.iterator(); i.hasNext(); ) texture.removePropertyChangeListener((PropertyChangeListener) i.next());
      }
    }
    texture = t;
    if((texture != null) && (changeListeners != null)) {
      synchronized(changeListeners) {
        for(Iterator i = changeListeners.iterator(); i.hasNext(); ) texture.addPropertyChangeListener   ((PropertyChangeListener) i.next());
      }
    }
    defineUseAlpha();
    firePropertyChange("texture");
  }
  
  private float reflection; // TODO
  private int spc;          // TODO
  
  private float shininess = 1f;
  /**
   * Gets the shininess of this material. Shininess ranges from 0f (metallic) to 1f (plastic).
   * Default is 1f.
   * @return the shininess
   */
  public float getShininess() { return shininess; }
  /**
   * Sets the shininess of this material. Shininess ranges from 0f (metallic) to 1f (plastic).
   * @param f the new shininess
   */
  public void setShininess(float f) {
    shininess = f;
    firePropertyChange("shininess");
  }
  
  private final float[] diffuse = (float[]) WHITE_COLOR.clone();
  /**
   * Gets the diffuse color. Default is WHITE_COLOR.
   * @return the diffuse color
   */
  public float[] getDiffuse() { return diffuse; }
  /**
   * Sets the diffuse color with implicit alpha = 1f .
   * @param red   the red component
   * @param green the green component
   * @param blue  the blue component
   */
  public synchronized void setDiffuse(float red, float green, float blue) { setDiffuse(red, green, blue, 1f); }
  /**
   * Sets the diffuse color.
   * @param red   the red component
   * @param green the green component
   * @param blue  the blue component
   * @param alpha the alpha component
   */
  public synchronized void setDiffuse(float red, float green, float blue, float alpha) {
    diffuse[0] = red;
    diffuse[1] = green;
    diffuse[2] = blue;
    diffuse[3] = alpha;
    defineUseAlpha();
    firePropertyChange("diffuse");
  }
  /**
   * Sets the diffuse color.
   * @param i the new color (a float[4])
   */
  public synchronized void setDiffuse(float[] i) {
    System.arraycopy(i, 0, diffuse, 0, 4);
    defineUseAlpha();
    firePropertyChange("diffuse");
  }
  /**
   * Gets the diffuse color's red component.
   * @return the red component
   */
  public synchronized float getDiffuseRed() { return diffuse[0]; }
  /**
   * Sets the diffuse color's red component.
   * @param f the new red component
   */
  public synchronized void setDiffuseRed(float f) {
    diffuse[0] = f;
    firePropertyChange("diffuse");
  }
  /**
   * Gets the diffuse color's green component.
   * @return the green component
   */
  public synchronized float getDiffuseGreen() { return diffuse[1]; }
  /**
   * Sets the diffuse color's green component.
   * @param f the new green component
   */
  public synchronized void setDiffuseGreen(float f) {
    diffuse[1] = f;
    firePropertyChange("diffuse");
  }
  /**
   * Gets the diffuse color's blue component.
   * @return the blue component
   */
  public synchronized float getDiffuseBlue() { return diffuse[2]; }
  /**
   * Sets the diffuse color's blue component.
   * @param f the new blue component
   */
  public synchronized void setDiffuseBlue(float f) {
    diffuse[2] = f;
    firePropertyChange("diffuse");
  }
  /**
   * Gets the diffuse color's alpha component.
   * @return the alpha component
   */
  public synchronized float getDiffuseAlpha() { return diffuse[3]; }
  /**
   * Sets the diffuse color's alpha component.
   * @param f the new alpha component
   */
  public synchronized void setDiffuseAlpha(float f) {
    diffuse[3] = f;
    defineUseAlpha();
    firePropertyChange("diffuse");
  }
  
  private final float[] specular = (float[]) WHITE_COLOR.clone();
  /**
   * Gets the specular color. Default is WHITE_COLOR.
   * @return the specular color
   */
  public float[] getSpecular() { return diffuse; }
  /**
   * Sets the specular color with implicit alpha = 1f .
   * @param red   the red component
   * @param green the green component
   * @param blue  the blue component
   */
  public synchronized void setSpecular(float red, float green, float blue) {
    setSpecular(red, green, blue, 1f);
  }
  /**
   * Sets the specular color.
   * @param red   the red component
   * @param green the green component
   * @param blue  the blue component
   * @param alpha the alpha component
   */
  public synchronized void setSpecular(float red, float green, float blue, float alpha) {
    specular[0] = red;
    specular[1] = green;
    specular[2] = blue;
    specular[3] = alpha;
    defineUseAlpha();
    firePropertyChange("specular");
  }
  /**
   * Sets the specular color.
   * @param i the new color (a float[4])
   */
  public synchronized void setSpecular(float[] i) {
    System.arraycopy(i, 0, specular, 0, 4);
    defineUseAlpha();
    firePropertyChange("specular");
  }
  /**
   * Gets the specular color's red component.
   * @return the red component
   */
  public synchronized float getSpecularRed() { return specular[0]; }
  /**
   * Sets the specular color's red component.
   * @param f the new red component
   */
  public synchronized void setSpecularRed(float f) {
    specular[0] = f;
    firePropertyChange("specular");
  }
  /**
   * Gets the specular color's green component.
   * @return the green component
   */
  public synchronized float getSpecularGreen() { return specular[1]; }
  /**
   * Sets the specular color's green component.
   * @param f the new green component
   */
  public synchronized void setSpecularGreen(float f) {
    specular[1] = f;
    firePropertyChange("specular");
  }
  /**
   * Gets the specular color's blue component.
   * @return the blue component
   */
  public synchronized float getSpecularBlue() { return specular[2]; }
  /**
   * Sets the specular color's blue component.
   * @param f the new blue component
   */
  public synchronized void setSpecularBlue(float f) {
    specular[2] = f;
    firePropertyChange("specular");
  }
  /**
   * Gets the specular color's alpha component.
   * @return the alpha component
   */
  public synchronized float getSpecularAlpha() { return specular[3]; }
  /**
   * Sets the specular color's alpha component.
   * @param f the new alpha component
   */
  public synchronized void setSpecularAlpha(float f) {
    specular[3] = f;
    defineUseAlpha();
    firePropertyChange("specular");
  }

  private boolean useAlpha;
  /**
   * Cheks if the material use alpha blending. Alpha blending is always rendered at the end.
   * @return true if use alpha blending
   */
  public boolean getUseAlpha() { return useAlpha; }
  private synchronized void defineUseAlpha() {
    if(overlayed) {
      useAlpha = true;
      firePropertyChange("useAlpha");
      return;
    }
    if(texture != null) {
      if(texture.getUseAlpha()) {
        useAlpha = true;
        firePropertyChange("useAlpha");
        return;
      }
    }
    if((diffuse[3] != 1) || (specular[3] != 1)) useAlpha = true;
    else useAlpha = false;

    firePropertyChange("useAlpha");
  }
  
  private boolean wireframed;
  /**
   * Checks if this material is wireframed. Default is false.
   * Warning : wireframed faces are not taken into account for raypicking.
   * @return true if wireframed
   */
  public boolean isWireframed() { return wireframed; }
  /**
   * Sets if this material is wireframed.
   * @param b true if wireframed
   */
  public void setWireframed(boolean b) {
    wireframed = b;
    firePropertyChange("wireframed");
  }
  
  private boolean overlayed;
  /**
   * Checks if this material is drawn overlayed. An overlayed face is drawn over any other,
   * even if it is farther from the eye.
   * Default is false.
   * @return true if overlayed
   */
  public boolean isOverlayed() { return overlayed; }
  /**
   * Sets if this material is drawn overlayed.
   * @param b true if overlayed
   */
  public void setOverlayed(boolean b) {
    overlayed = b;
    defineUseAlpha();
    firePropertyChange("overlayed");
  }
  
  private boolean separateSpecular;
  /**
   * Checks if this material uses a separate specular. The separate specular can improve or
   * enforce the specular on textured material.
   * @return true if the specular is computed separately
   */
  public boolean isSeparateSpecular() { return separateSpecular; }
  /**
   * Sets if this material uses a separate specular.
   * @param b true to compute the specular separately
   */
  public void setSeparateSpecular(boolean b) {
    separateSpecular = b;
    firePropertyChange("separateSpecular");
  }
  
  public static final int BLENDING_TRANSPARENCY          = 0;
  public static final int BLENDING_ADDITIVE_TRANSPARENCY = 1;
  protected int blending = BLENDING_TRANSPARENCY;
  /**
   * Gets the blending operation. This property is useless if the material do not use alpha
   * channel.
   * If the value is BLENDING_TRANSPARENCY (default value), the alpha channel is used for
   * normal transparency. If it is BLENDING_ADDITIVE_TRANSPARENCY, the color of the material
   * (including texture) is added; this is often used for special effect.
   * @return a BLENDING_* constant
   */
  public int getBlending() { return blending; }
  /**
   * Sets the blending operation.
   * @param b a BLENDING_* constant
   */
  public void setBlending(int b) {
    blending = b;
    firePropertyChange("blending");
  }
  
  /*
  public static final int LOGIC_OP_NONE          = 0;
  public static final int LOGIC_OP_COPY          = GLEnum.GL_COPY         ;
  public static final int LOGIC_OP_AND           = GLEnum.GL_AND          ;
  public static final int LOGIC_OP_OR            = GLEnum.GL_OR           ;
  public static final int LOGIC_OP_INVERT        = GLEnum.GL_INVERT       ;
  public static final int LOGIC_OP_COPY_INVERTED = GLEnum.GL_COPY_INVERTED;
  public static final int LOGIC_OP_NAND          = GLEnum.GL_NAND         ;
  public static final int LOGIC_OP_NOR           = GLEnum.GL_NOR          ;
  public static final int LOGIC_OP_XOR           = GLEnum.GL_XOR          ;
  public static final int LOGIC_OP_EQUIV         = GLEnum.GL_EQUIV        ;
  public static final int LOGIC_OP_AND_INVERTED  = GLEnum.GL_AND_INVERTED ;
  public static final int LOGIC_OP_OR_INVERTED   = GLEnum.GL_OR_INVERTED  ;
  public static final int LOGIC_OP_AND_REVERSE   = GLEnum.GL_AND_REVERSE  ;
  public static final int LOGIC_OP_OR_REVERSE    = GLEnum.GL_OR_REVERSE   ;
  protected int logicOp;
  public int getLogicOp() { return logicOp; }
  public void setLogicOp(int i) {
    logicOp = i;
    firePropertyChange("logicOp");
  }
  */
  
  /**
   * Make this material the current material. Future openGL drawing will use its properties
   * (texture, colors, shininess,...).
   * @param gl  the gl
   * @param glu the glu
   */
  public synchronized void makeCurrent(GLFunc gl, GLUFunc glu) {
    if(texture != null) texture.makeCurrent(gl, glu);
    else {
      if(gl.glIsEnabled(GLEnum.GL_TEXTURE_2D)) gl.glDisable(GLEnum.GL_TEXTURE_2D);
    }
    if(separateSpecular) {
      gl.glLightModeli(gl.GL_LIGHT_MODEL_COLOR_CONTROL, gl.GL_SEPARATE_SPECULAR_COLOR);
      //gl.glGetIntegerv(gl.GL_LIGHT_MODEL_COLOR_CONTROL, oldSeparateSpecular);
    }
    /*
    if(separateSpecular) gl.glLightModeli(gl.GL_LIGHT_MODEL_COLOR_CONTROL, gl.GL_SEPARATE_SPECULAR_COLOR);
    else gl.glLightModeli(gl.GL_LIGHT_MODEL_COLOR_CONTROL, gl.GL_SINGLE_COLOR);
    */
    makeColorCurrent(gl, glu);
    if(overlayed)  gl.glDisable    (GLEnum.GL_DEPTH_TEST);
    if(wireframed) gl.glPolygonMode(GLEnum.GL_FRONT_AND_BACK, GLEnum.GL_LINE);
    /*
    if(logicOp != LOGIC_OP_NONE) {
      gl.glEnable(GLEnum.GL_COLOR_LOGIC_OP);
      gl.glLogicOp(logicOp);
    }
    */
    if(blending == BLENDING_ADDITIVE_TRANSPARENCY)
      gl.glBlendFunc(GLEnum.GL_SRC_ALPHA, GLEnum.GL_ONE);
  }
  /**
   * Make this material's color property current. Future openGL drawing will use these
   * colors.
   * @param gl  the gl
   * @param glu the glu
   */
  public synchronized void makeColorCurrent(GLFunc gl, GLUFunc glu) { // Not to use material properties do not cause a significant performance boost.
    gl.glMateriali(GLEnum.GL_FRONT_AND_BACK, GLEnum.GL_SHININESS, (int) (shininess * 128));
    gl.glMaterialfv(GLEnum.GL_FRONT_AND_BACK, GLEnum.GL_SPECULAR, specular);
    gl.glMaterialfv(GLEnum.GL_FRONT_AND_BACK, GLEnum.GL_AMBIENT_AND_DIFFUSE, diffuse);
  }
  /**
   * Release the current material. Future openGL drawing will no longer use its properties.
   * @param gl  the gl
   * @param glu the glu
   */
  public synchronized void unmakeCurrent(GLFunc gl, GLUFunc glu) {
    if(texture != null) texture.unmakeCurrent(gl, glu);
    if(overlayed )       gl.glEnable     (GLEnum.GL_DEPTH_TEST);
    if(wireframed)       gl.glPolygonMode(GLEnum.GL_FRONT_AND_BACK, GLEnum.GL_FILL);
    if(separateSpecular)
      gl.glLightModeli(gl.GL_LIGHT_MODEL_COLOR_CONTROL, gl.GL_SINGLE_COLOR);
    //if(separateSpecular) gl.glLightModeli(gl.GL_LIGHT_MODEL_COLOR_CONTROL, gl.GL_SINGLE_COLOR);
    /*
    if(logicOp != LOGIC_OP_NONE) {
      gl.glDisable(GLEnum.GL_COLOR_LOGIC_OP);
    }
    */
    if(blending == BLENDING_ADDITIVE_TRANSPARENCY)
      gl.glBlendFunc(GLEnum.GL_SRC_ALPHA, GLEnum.GL_ONE_MINUS_SRC_ALPHA);
  }
  
  // Event : Overrides :
  public void addPropertyChangeListener   (PropertyChangeListener l) {
    super.addPropertyChangeListener   (l);
    if(texture != null) texture.addPropertyChangeListener   (l);
  }
  public void removePropertyChangeListener(PropertyChangeListener l) {
    super.removePropertyChangeListener(l);
    if(texture != null) texture.removePropertyChangeListener(l);
  }
  
  public static final Material WHITE_MATERIAL = new Material();
}
