/*
 * Soya3D
 * Copyright (C) 1999  Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.awt.*;
import gl4java.*;
import java.io.*;
import java.lang.reflect.*;

/**
 * A camera is a graphical element that can be used as a renderer.
 * 
 * A camera can render on a renderingSurface.
 * 
 * @see opale.soya.awt.RenderingSurface
 * @see opale.soya.soya3d.Renderer
 * 
 * @author Artiste on the Web
 */

public class Camera3D extends GraphicalElement3D implements Renderer {
  public Camera3D()               {
    super()       ; 
    //setRotationType(Orientation.ROTATION_TYPE_INTERNAL);
  }
  public Camera3D(String newName) {
    super(newName);
    //setRotationType(Orientation.ROTATION_TYPE_INTERNAL);
  }
  
  /**
   * Clones this camera. Warning : the rendering surface is not cloned.
   * @see opale.soya.soya3d.GraphicalElement3D#clone
   * @see opale.soya.soya3d.Element3D#clone
   * @return the clone
   */
  public synchronized Object clone() {
    Camera3D c = (Camera3D) super.clone();
    c.zPreWriting = zPreWriting;
    c.dontClearFrameBuffer = dontClearFrameBuffer;
    try { c.setDrawablesCollectorClass(fc.getClass()); } // Can't occur because the class has
    catch(Exception e) { e.printStackTrace(); }        // already been used successfully in this.
    c.front = front;
    c.back = back;
    c.fov = fov;
    c.ortho = ortho;
    return c;
  }
  
  // Dimension :
  public float getWidth () { return 0f; }
  public float getHeight() { return 0f; }
  public float getDepth () { return 0f; }
  public DimensionWrapper wrapper() { return new Box(new Point(0f, 0f, 0f, this), new Point(0f, 0f, 0f, this)); }
  
  /**
   * If true, active z pre-writing. May speed up the rendering if a lot of face are
   * overlapped; can also slow down... A good idea is to try it, the best is to let the
   * choice to the final user...
   * Default is false.
   */
  public boolean zPreWriting;
  /**
   * If true, the frameBuffer is not cleared. In this case, rendering can be quicker if your
   * scene covers all the surface. It can also be usefull into a rendering overlay (see
   * soya.awt.RenderingOverlay).
   * Default is false.
   */
  public boolean dontClearFrameBuffer;
  
  /**
   * Computes the model view matrix.
   * @return the model view matrix
   */
  public float[] modelMatrix() {
    float[] mat = (float[]) getInvertedRootMatrix().clone();
    mat[12] = 0f;
    mat[13] = 0f;
    mat[14] = 0f;
    Position p = clone(getRootParent());
    mat = Matrix.matrixMultiply(mat, Matrix.matrixTranslate(-p.getX(), -p.getY(), -p.getZ()));
    return mat;
  }
  
  private DrawablesCollector fc = new DrawablesCollectorByMaterial(this);        // You can try those other 
  //private DrawablesCollector fc = new DrawablesCollectorByOrder(this);        // fragments collectors, for me the 
  //private DrawablesCollector fc = new DrawablesCollectorImmediateDraw(this); // "by material" one is the quickiest.
  //private DrawablesCollector fc = new DrawablesCollectorByDistance(this);
  /**
   * Gets the class of drawables collector used by this camera. The drawables collector is
   * responsible for the drawables collect and the drawables' draw order; it can perform
   * many kind of optimization.
   * Default is DrawablesCollectorByMaterial.class .
   * @see opale.soya.soya3d.DrawablesCollector
   * @see opale.soya.soya3d.DrawablesCollectorByMaterial
   * @return the class of drawables collector
   */
  public Class getDrawablesCollectorClass() { return fc.getClass(); }
  /**
   * Sets the class of drawables collector used by this camera.
   * @param c the new class of drawables collector.
   */
  public void setDrawablesCollectorClass(Class c) throws NoSuchMethodException, SecurityException, InstantiationException, IllegalAccessException, IllegalArgumentException, InvocationTargetException {
    Class [] paramsClasses = { Renderer.class };
    Object[] params        = { this           };
    Constructor constructor = c.getConstructor(paramsClasses);
    fc = (DrawablesCollector) constructor.newInstance(params);
  }
  
  public final ConfigurationChanger getConfigurationChanger() { return fc; }
  
  public void render() { // Collect the fragments and call the draw mehod.
    if(glj == null || gl == null) {
      if(rs != null) { rs.setRenderer(this); }
      if(glj == null || gl == null) return;
      //if(glj == null || gl == null) throw new UnsupportedOperationException("No GL Context.");
    }
    synchronized(fc) { // Only one rendering at a time.
      getDrawablesCollected();
      draw();
    }
    computeFPS();
  }
  /**
   * Calls the fragments collect, and return the fragments collector. This is used before
   * rendering.
   * @return the fragment collector
   */
  protected DrawablesCollector getDrawablesCollected() {// Collect fragments.
    World3D w = getRootParent();
    
    synchronized(fc) {
      fc.reset();
      nextLightID = GLEnum.GL_LIGHT0;
      nbRegistredLights = 0;
      
      gl.glMatrixMode(GLEnum.GL_MODELVIEW); // Should be already set.
      gl.glLoadIdentity();
      
      float[] mat = modelMatrix();
      w.fillCollector(fc, (Renderer) this, mat);
    }
    areDynamicLightsVisible = true;
    for(int i = nextLightID; i <= LAST_LIGHT_ID; i++) {
      gl.glDisable(i); // Disable unused lights.
      //registeredLights[i - GLEnum.GL_LIGHT0] = null;
    }
    return fc;
  }
  private void draw() { // draw the fragments
    Environment3D r = getEnvironment();
    int width = rs.getSurfaceWidth(), height = rs.getSurfaceHeight();
    synchronized(glj) {
      // Define environmental properties and clear.
      if(r == null) {
        Environment3D.makeDefaultEnvironmentCurrent(gl, glu);
        if(!dontClearFrameBuffer) Environment3D.defaultClear(gl, glu, rs);
        else Environment3D.clearZBuffer(gl, glu, rs);
      }
      else {
        r.makeEnvironmentCurrent(gl, glu);
        if(!dontClearFrameBuffer) r.clear(gl, glu, rs);
        else Environment3D.clearZBuffer(gl, glu, rs);
      }
      
      // Set the projection matrix.
      gl.glMatrixMode(GLEnum.GL_PROJECTION);
      gl.glLoadIdentity();
      if(ortho) {
        float l = fov / 75f, ratio = ((float) height) / ((float) width);
        gl.glOrtho((double) -l, (double) l, (double) (-l * ratio), (double) (l * ratio), -10000d, 10000d);
      }
      else glu.gluPerspective(fov, ((float) width) / ((float) height), front, back);
      gl.glMatrixMode(GLEnum.GL_MODELVIEW);
      
      // Ready... Draw !
      if(zPreWriting) {
        gl.glDisable(GLEnum.GL_LIGHTING);
        gl.glShadeModel(GLEnum.GL_FLAT);
        gl.glDrawBuffer(GLEnum.GL_NONE);
        //gl.glColorMask(false, false, false, false);
        fc.drawZOnly();
        fc.drawAlphaZOnly();
        //gl.glColorMask(true, true, true, true);
        gl.glDrawBuffer(GLEnum.GL_BACK);
        gl.glShadeModel(GLEnum.GL_SMOOTH);
        gl.glEnable(GLEnum.GL_LIGHTING);
        
        gl.glDepthFunc(GLEnum.GL_EQUAL);
        gl.glDepthMask(false);
        fc.draw();
        gl.glEnable(GLEnum.GL_BLEND);
        gl.glDepthMask(false);
        fc.drawAlpha();
        gl.glDisable(GLEnum.GL_BLEND);
        gl.glDepthMask(true);
        gl.glDepthFunc(GLEnum.GL_LESS);
      }
      else {
        fc.draw();
        gl.glEnable(GLEnum.GL_BLEND);
        gl.glDepthMask(false);
        fc.drawAlpha();
        gl.glDisable(GLEnum.GL_BLEND);
        gl.glDepthMask(true);
      }

      // Remove environmental properties.
      if(r == null) Environment3D.unmakeDefaultEnvironmentCurrent(gl, glu);
      else r.unmakeEnvironmentCurrent(gl, glu);
    }
  }


  private float front = 0.01f, back = 1000, fov = 90f;
  /**
   * Gets the front plane of the viewing frustum. Any pixels whose Z coordinates (in the
   * camera local coordinates system) is lower than the front is not drawn.
   * Default is 0.01f . It must be > 0f and < this.getBack() .
   * @return the front
   */
  public float getFront() { return front; }
  /**
   * Sets the front plane of the viewing frustum.
   * @param f the new front
   */
  public void setFront(float f) {
    if(f <= 0    ) throw new UnsupportedOperationException("Front must be > 0"   );
    if(f >= back ) throw new UnsupportedOperationException("Front must be < back");
    synchronized(this) { front = f; }
    firePropertyChange("front");
  }
  /**
   * Gets the back plane of the viewing frustum. Any pixels whose Z coordinates (in the
   * camera local coordinates system) is higher than the back is not drawn.
   * Default is 100. It must be > this.getFront() .
   * @return the front
   */
  public float getBack() { return back; }
  /**
   * Sets the back plane of the viewing frustum.
   * @param f the new back
   */
  public void setBack(float f) {
    if(f <= front) throw new UnsupportedOperationException("Back must be > front");
    synchronized(this) { back = f; }
    firePropertyChange("back");
  }
  /**
   * Gets the FOV (Field Of Vision); you can change it in order to make a zoom.
   * Default is 90f .
   * @return the FOV in degrees
   */
  public float getFOV() { return fov; }
  /**
   * Sets the FOV (Field Of Vision).
   * @param f the new FOV in degrees
   */
  public void setFOV(float f) {
    if(f <= 0) throw new UnsupportedOperationException("FOV must be > 0");
    synchronized(this) { fov = f; }
    firePropertyChange("FOV");
  }
  
  private boolean ortho;
  /**
   * Checks if this camera is ortho. An "ortho camera" draws with isometric perspective.
   * Default is false.
   * @return true if ortho
   */
  public boolean isOrtho() { return ortho; }
  /**
   * Sets if this camera is ortho.
   * @param b true for ortho
   */
  public void setOrtho(boolean b) {
    synchronized(this) { ortho = b; }
    firePropertyChange("ortho");
  }
  
  // Renderer :
  protected RenderingSurface rs;
  public RenderingSurface getRenderingSurface() { return rs; }
  public void setRenderingSurface(RenderingSurface r) {
    synchronized(this) { 
      if(r.getRenderer() != this) {
        r.setRenderer(this);
        return;
      }
      rs = r;
      if(rs == null) setGLContext(null);
      else setGLContext(rs.getGLContext());
    }
    firePropertyChange("renderingSurface");
  }
  protected transient GLContext glj;
  protected transient GLFunc  gl ;
  protected transient GLUFunc glu;
  public GLContext getGLContext() { return glj; }
  private void setGLContext(GLContext c) {
    glj = c;
    if(glj == null) {
      gl  = null;
      glu = null;
      fc.setGLandGLU(null, null);
    }
    else {
      gl  = glj.getGLFunc ();
      glu = glj.getGLUFunc();
      fc.setGLandGLU(gl, glu);
    }
  }
  public GLFunc  getGLFunc () { return gl ; }
  public GLUFunc getGLUFunc() { return glu; }
  
  public Position getPosition() { return (Position) this; }
  /*
  public float getDistanceTo() {
    float[] m = new float[16];
    gl.glGetFloatv(GLEnum.GL_MODELVIEW_MATRIX, m);
    return (float) Math.sqrt(Math.pow((double) m[12], 2) + Math.pow((double) m[13], 2) + Math.pow((double) m[14], 2));
  }
  */
  
  private transient int nextLightID;
  protected static final int LAST_LIGHT_ID = GLEnum.GL_LIGHT7;
  /*
  protected transient Light3D[] registeredLights = new Light3D[LAST_LIGHT_ID - GLEnum.GL_LIGHT0 + 1];
  public int newLightID(Light3D s) {
    if(nextLightID <= LAST_LIGHT_ID) {
      registeredLights[nextLightID - GLEnum.GL_LIGHT0] = s;
      return nextLightID++;
    }
    else return 0;
  }
  */
  protected transient boolean[] areRegisteredLightsDynamic = new boolean[LAST_LIGHT_ID - GLEnum.GL_LIGHT0 + 1];
  protected int nbRegistredLights;
  public synchronized int newLightID(boolean dynamic) {
    if(nextLightID <= LAST_LIGHT_ID) {
      areRegisteredLightsDynamic[nbRegistredLights++] = dynamic;
      return nextLightID++;
    }
    else return 0;
  }
  protected boolean areDynamicLightsVisible = true;
  public void setDynamicLightsVisible(boolean b) {
    if(areDynamicLightsVisible != b) {
      areDynamicLightsVisible = b;
      for(int i = 0; i < nextLightID - GLEnum.GL_LIGHT0; i++) {
        if(areRegisteredLightsDynamic[i]) {
          if(b) gl.glEnable(GLEnum.GL_LIGHT0 + i);
          else gl.glDisable(GLEnum.GL_LIGHT0 + i);
        }
      }
    }
  }
  
  // FPS :
  private float theoricalFPS = 20f, fps, fpsFactor;
  /**
   * Gets the theorical FPS. the speed of the scene will be the same as if it was rendered
   * at this FPS value.
   * Default is 20 FPS
   * @return the theorical FPS
   */
  public float getTheoricalFPS() { return theoricalFPS; }
  /**
   * Sets the theorical FPS.
   * @param f the new theorical FPS
   */
  public void setTheoricalFPS(float f) {
    theoricalFPS = f;
    firePropertyChange("theoricalFPS");
  }
  protected long[] lastTimes;
  protected void computeFPS() {
    if(lastTimes == null) {
      lastTimes = new long[10];
      lastTimes[9] = System.currentTimeMillis();
      for(int j = 8; j >= 0; j--) lastTimes[j] = lastTimes[j + 1] - (long) (1000f / theoricalFPS);
    }
    int max = lastTimes.length - 1;
    System.arraycopy(lastTimes, 1, lastTimes, 0, max);
    lastTimes[max] = System.currentTimeMillis();
    fps = 1000f * ((float) lastTimes.length) / ((float) (lastTimes[max] - lastTimes[0]));
    fpsFactor = theoricalFPS / fps;
  }
  public void resetFPSComputation() {
    lastTimes = null;
  }
  /**
   * Gets the current FPS.
   * @return the fps
   */
  public float fps()       { return fps      ; }
  /**
   * Gets the current FPS factor : the ratio between the theorical FPS and the real one.
   * @return the fps factor
   */
  public float fpsFactor() { return fpsFactor; }
}
