/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d;

import opale.soya.*;
import opale.soya.awt.*;
import opale.soya.soya2d.*;
import gl4java.*;
import java.io.*;

/**
 * An environment is a world that have also environmental properties, like fog, background
 * color or image, ...
 * 
 * Environments are often used as "root world".
 * 
 * @author Artiste on the Web
 */

public class Environment3D extends World3D {
  private static final long serialVersionUID = 8995190792621426992l;
  /**
   * Clones this environment. Warning : background image is not cloned in depth.
   * @see opale.soya.soya3d.World3D#clone
   * @see opale.soya.soya3d.Volume3D#clone
   * @see opale.soya.soya3d.GraphicalElement3D#clone
   * @see opale.soya.soya3d.Element3D#clone
   * @return the clone
   */
  public synchronized Object clone() {
    Environment3D e = (Environment3D) super.clone();
    System.arraycopy(backgroundColor, 0, e.backgroundColor, 0, 4);
    System.arraycopy(ambient, 0, e.ambient, 0, 4);
    e.backgroundImage = backgroundImage;
    e.fogEnabled = fogEnabled;
    e.fogType    = fogType   ;
    e.fogStart   = fogStart  ;
    e.fogEnd     = fogEnd    ;
    e.fogDensity = fogDensity;
    return e;
  }
  
  /** Constant for linear fog */
  public static final int FOG_TYPE_LINEAR = GLEnum.GL_LINEAR;
  /** Constant for exponentiel fog */
  public static final int FOG_TYPE_EXPONENTIEL = GLEnum.GL_EXP;
  /** Constant for exponentiel squarre fog */
  public static final int FOG_TYPE_EXPONENTIEL_2 = GLEnum.GL_EXP2;
  
  /**
   * Makes current the default environmental properties in the GL.
   * @param gl  the gl
   * @param glu the glu
   */
  protected static void makeDefaultEnvironmentCurrent(GLFunc gl, GLUFunc glu) { // Properties used if no Environment3D.
    gl.glClearColor(0f, 0f, 0f, 0f);
    gl.glLightModelfv(GLEnum.GL_LIGHT_MODEL_AMBIENT, Material.BLACK_COLOR);
    gl.glDisable(GLEnum.GL_FOG);
  }
  /**
   * Unmakes current the default environmental properties in the GL.
   * @param gl  the gl
   * @param glu the glu
   */
  protected static void unmakeDefaultEnvironmentCurrent(GLFunc gl, GLUFunc glu) {
  }
  /**
   * Clears the GL with the default clearing method.
   * @param gl  the gl
   * @param glu the glu
   * @param rs the rendering surface
   */
  protected static void defaultClear(GLFunc gl, GLUFunc glu, RenderingSurface rs) {
    if((rs instanceof RenderingSurfacePart) && (((RenderingSurfacePart) rs).getRenderingMultipleSurface().getNumberOfComponents() > 1)) {
      // Not the whole surface, so cannot clear !
      Quality.makeOverlaysCurrent(gl, glu, rs); // Disable useless state (lighting, ...);
      gl.glColor4f(0f, 0f, 0f, 1f);
      gl.glBegin(GLEnum.GL_QUADS);
      gl.glVertex3f(0f           , 0f            , 0f);
      gl.glVertex3f(rs.getSurfaceWidth(), 0f            , 0f);
      gl.glVertex3f(rs.getSurfaceWidth(), rs.getSurfaceHeight(), 0f);
      gl.glVertex3f(0f           , rs.getSurfaceHeight(), 0f);
      gl.glEnd();
      Quality.unmakeOverlaysCurrent(gl, glu, rs);
      gl.glClear(GLEnum.GL_DEPTH_BUFFER_BIT);
    }
    else {
      gl.glClear(GLEnum.GL_DEPTH_BUFFER_BIT | GLEnum.GL_COLOR_BUFFER_BIT);
    }
  }
  /**
   * Clears the GL z buffer.
   * @param gl  the gl
   * @param glu the glu
   * @param rs the rendering surface
   */
  protected static void clearZBuffer(GLFunc gl, GLUFunc glu, RenderingSurface rs) {
    gl.glClear(GLEnum.GL_DEPTH_BUFFER_BIT);
  }
  
  /**
   * Makes current in the GL this environment's environmental properties.
   * @param gl  the gl
   * @param glu the glu
   */
  protected synchronized void makeEnvironmentCurrent(GLFunc gl, GLUFunc glu) {
    gl.glClearColor(backgroundColor[0], backgroundColor[1], backgroundColor[2], backgroundColor[3]);
    gl.glLightModelfv(GLEnum.GL_LIGHT_MODEL_AMBIENT, ambient);
    makeFogCurrent(gl, glu);
  }
  /**
   * Unmakes current in the GL this environment's environmental properties.
   * @param gl  the gl
   * @param glu the glu
   */
  protected void unmakeEnvironmentCurrent(GLFunc gl, GLUFunc glu) {
  }
  
  /**
   * Clears the GL with this environment's clearing method.
   * @param gl  the gl
   * @param glu the glu
   * @param rs the rendering surface
   */
  protected synchronized void clear(GLFunc gl, GLUFunc glu, RenderingSurface rs) {
    if(backgroundImage == null) clearWithColor(gl, glu, rs);
    else {
      if(backgroundImage.isTransparent()) clearWithColor(gl, glu, rs);
      Quality.makeOverlaysCurrent(gl, glu, rs); // Disable useless state (lighting, ...);
      backgroundImage.draw(gl, glu, 0, rs.getSurfaceHeight(), rs.getSurfaceWidth(), -rs.getSurfaceHeight()); // Optimizable (use a pre-scaled image).
      Quality.unmakeOverlaysCurrent(gl, glu, rs);
    }
  }
  protected synchronized void clearWithColor(GLFunc gl, GLUFunc glu, RenderingSurface rs) {
    if((rs instanceof RenderingSurfacePart) && (((RenderingSurfacePart) rs).getRenderingMultipleSurface().getNumberOfComponents() > 1)) {
      // Not the whole surface, so cannot clear !
      Quality.makeOverlaysCurrent(gl, glu, rs); // Disable useless state (lighting, ...);
      gl.glColor4fv(backgroundColor);
      gl.glBegin(GLEnum.GL_QUADS);
      gl.glVertex3f(0f           , 0f            , 0f);
      gl.glVertex3f(rs.getSurfaceWidth(), 0f            , 0f);
      gl.glVertex3f(rs.getSurfaceWidth(), rs.getSurfaceHeight(), 0f);
      gl.glVertex3f(0f           , rs.getSurfaceHeight(), 0f);
      gl.glEnd();
      Quality.unmakeOverlaysCurrent(gl, glu, rs);
      gl.glClear(GLEnum.GL_DEPTH_BUFFER_BIT);
    }
    else {
      gl.glClear(GLEnum.GL_DEPTH_BUFFER_BIT | GLEnum.GL_COLOR_BUFFER_BIT);
    }
  }
  
  protected final float[] backgroundColor = (float[]) Material.BLACK_COLOR.clone();
  /**
   * Gets the background color. It's the color used for clearing the rendering surface.
   * Default is Material.BLACK_COLOR .
   * @return the background color in a float[4]
   */
  public float[] getBackgroundColor() { return backgroundColor; }
  /**
   * Sets the background color.
   * @param red   the new red   component
   * @param green the new green component
   * @param blue  the new blue  component
   */
  public void setBackgroundColor(float red, float green, float blue) { setBackgroundColor(red, green, blue, 1f); }
  /**
   * Sets the background color.
   * @param red   the new red   component
   * @param green the new green component
   * @param blue  the new blue  component
   * @param alpha the new alpha component
   */
  public void setBackgroundColor(float red, float green, float blue, float alpha) {
    synchronized(this) {
      backgroundColor[0] = red;
      backgroundColor[1] = green;
      backgroundColor[2] = blue;
      backgroundColor[3] = alpha;
    }
    firePropertyChange("backgroundColor");
  }
  /**
   * Sets the background color.
   * @param c the new color in a float[4]
   */
  public void setBackgroundColor(float[] c) {
    synchronized(this) { System.arraycopy(c, 0, backgroundColor, 0, 4); }
    firePropertyChange("backgroundColor");
  }
  /**
   * Gets the background color red component.
   * @return the background color red component
   */
  public float getBackgroundRed  () { return backgroundColor[0]; }
  /**
   * Gets the background color green component.
   * @return the background color green component
   */
  public float getBackgroundGreen() { return backgroundColor[1]; }
  /**
   * Gets the background color blue component.
   * @return the background color blue component
   */
  public float getBackgroundBlue () { return backgroundColor[2]; }
  /**
   * Gets the background color alpha component.
   * @return the background color alpha component
   */
  public float getBackgroundAlpha() { return backgroundColor[3]; }
  /**
   * Sets the background color red component.
   * @param f the new background color red component
   */
  public void setBackgroundRed  (float f) {
    synchronized(this) { backgroundColor[0] = f; }
    firePropertyChange("backgroundColor");
  }
  /**
   * Sets the background color green component.
   * @param f the new background color green component
   */
  public void setBackgroundGreen(float f) {
    synchronized(this) { backgroundColor[1] = f; }
    firePropertyChange("backgroundColor");
  }
  /**
   * Sets the background color blue component.
   * @param f the new background color blue component
   */
  public void setBackgroundBlue (float f) {
    synchronized(this) { backgroundColor[2] = f; }
    firePropertyChange("backgroundColor");
  }
  /**
   * Sets the background color alpha component.
   * @param f the new background color alpha component
   */
  public void setBackgroundAlpha(float f) {
    synchronized(this) { backgroundColor[3] = f; }
    firePropertyChange("backgroundColor");
  }

  protected Image backgroundImage;
  /**
   * Gets the background image. This image is drawn in order to clear the rendering
   * surface, the background color is not used, except if the image is transparent.
   * Default is null .
   * @return the background image
   */
  public Image getBackgroundImage() { return backgroundImage; }
  /**
   * Sets the background image.
   * @param i the new background image
   */
  public void setBackgroundImage(Image i) {
    synchronized(this) { backgroundImage = i; }
    firePropertyChange("backgroundImage");
  }

  private boolean fogEnabled = false;
  /**
   * Checks if this environment is fogged.
   * @return true if the fog is enabled
   */
  public boolean isFogEnabled() { return fogEnabled; }
  /**
   * Sets if this environment is fogged.
   * @param true to set the fog enabled
   */
  public void setFogEnabled(boolean b) {
    synchronized(this) { fogEnabled = b; }
    firePropertyChange("fogEnabled");
  }
  
  private int fogType = FOG_TYPE_LINEAR;
  /**
   * Gets the fog type.
   * Default is FOG_TYPE_LINEAR .
   * @return a FOG_TYPE_* constant
   */
  public int getFogType() { return fogType; }
  /**
   * Sets the fog type.
   * @param a FOG_TYPE_* constant
   */
  public void setFogType(int i) {
    synchronized(this) { fogType = i; }
    firePropertyChange("fogType");
  }
  
  private float fogStart;
  /**
   * Gets the fog start. It's the distance from the eye where the fog begins to apply. It
   * is used by linear fog type.
   * Default is 0f .
   * @return the fog start
   */
  public float getFogStart() { return fogStart; }
  /**
   * Sets the fog start.
   * @param f the new fog start
   */
  public void setFogStart(float f) {
    synchronized(this) { fogStart = f; }
    firePropertyChange("fogStart");
  }
  
  private float fogEnd = 10f;
  /**
   * Gets the fog end. It's the distance from the eye where the fog is total (nothing is
   * visible). It is used by linear fog type.
   * Default is 10f .
   * @return the fog end
   */
  public float getFogEnd() { return fogEnd; }
  /**
   * Sets the fog end.
   * @param f the new fog end
   */
  public void setFogEnd(float f) {
    synchronized(this) { fogEnd = f; }
    firePropertyChange("fogEnd");
  }
  
  private float fogDensity = 1f;
  /**
   * Gets the fog density. It is used by exponentiel and exponentiel squarre fog type.
   * Default is 1f .
   * @return the fog end
   */
  public float getFogDensity() { return fogDensity; }
  /**
   * Sets the fog density.
   * @param f the new fog density
   */
  public void setFogDensity(float f) {
    synchronized(this) { fogDensity = f; }
    firePropertyChange("fogDensity");
  }
  
  /**
   * Makes current in the GL this environment's fog.
   * @param gl  the gl
   * @param glu the glu
   */
  protected synchronized void makeFogCurrent(GLFunc gl, GLUFunc glu) {
    if(fogEnabled) {
      gl.glEnable(GLEnum.GL_FOG);
      gl.glFogf (GLEnum.GL_FOG_MODE, fogType);
      gl.glFogf (GLEnum.GL_FOG_START, fogStart);
      gl.glFogf (GLEnum.GL_FOG_END, fogEnd);
      gl.glFogf (GLEnum.GL_FOG_DENSITY, fogDensity);
      gl.glFogfv(GLEnum.GL_FOG_COLOR, backgroundColor);
    }
    else gl.glDisable(GLEnum.GL_FOG);
  }
  
  private float[] ambient = (float[]) Material.BLACK_COLOR.clone();
  /**
   * Gets the ambient lighting color.
   * Default is Material.BLACK_COLOR .
   * @return the ambient color in a float[4]
   */
  public float[] getAmbientColor() { return ambient; }
  /**
   * Sets the ambient lighting color.
   * @param red   the new red   component
   * @param green the new green component
   * @param blue  the new blue  component
   */
  public void setAmbientColor(float red, float green, float blue) { setAmbientColor(red, green, blue, 1f); }
  /**
   * Sets the ambient lighting color.
   * @param red   the new red   component
   * @param green the new green component
   * @param blue  the new blue  component
   * @param alpha the new alpha component
   */
  public void setAmbientColor(float red, float green, float blue, float alpha) {
    synchronized(this) { 
      ambient[0] = red;
      ambient[1] = green;
      ambient[2] = blue;
      ambient[3] = alpha;
    }
    firePropertyChange("ambientColor");
  }
  /**
   * Sets the ambient lighting color.
   * @param i the new color in a float[4]
   */
  public void setAmbientColor(float[] i) {
    synchronized(this) { System.arraycopy(i, 0, ambient, 0, 4); }
    firePropertyChange("ambientColor");
  }
  
  /**
   * Gets the ambient color red component.
   * @return the ambient color red component
   */
  public float getAmbientRed  () { return ambient[0]; }
  /**
   * Gets the ambient color green component.
   * @return the ambient color green component
   */
  public float getAmbientGreen() { return ambient[1]; }
  /**
   * Gets the ambient color blue component.
   * @return the ambient color blue component
   */
  public float getAmbientBlue () { return ambient[2]; }
  /**
   * Gets the ambient color alpha component.
   * @return the ambient color alpha component
   */
  public float getAmbientAlpha() { return ambient[3]; }
  /**
   * Sets the ambient color red component.
   * @param f the new ambient color red component
   */
  public void setAmbientRed  (float f) {
    synchronized(this) { ambient[0] = f; }
    firePropertyChange("ambientColor");
  }
  /**
   * Sets the ambient color green component.
   * @param f the new ambient color green component
   */
  public void setAmbientGreen(float f) {
    synchronized(this) { ambient[1] = f; }
    firePropertyChange("ambientColor");
  }
  /**
   * Sets the ambient color blue component.
   * @param f the new ambient color blue component
   */
  public void setAmbientBlue (float f) {
    synchronized(this) { ambient[2] = f; }
    firePropertyChange("ambientColor");
  }
  /**
   * Sets the ambient color alpha component.
   * @param f the new ambient color alpha component
   */
  public void setAmbientAlpha(float f) {
    synchronized(this) { ambient[3] = f; }
    firePropertyChange("ambientColor");
  }
}
