/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.model.*;
import gl4java.*;
import java.util.*;
import java.io.*;
import java.beans.*;


/**
 * Abstract class for graphical 3D elements that can act either as fragments or as 3D
 * elements, and whose drawing doesn't change at each rendering.
 * 
 * This class is design to be extended by many other element / fragment. You just need to
 * implements a few methods :
 *  - getNaturalWidth(), getNaturalHeight(), getNaturalDepth() : returns the natural
 * dimensions : without any scaling, ... just the size of what you draw in the fixedDraw
 * method.
 *  - wrapper() : returns a dimension wrapper (like a Box).
 *  - getMaterial() : returns the material; you should add setMaterial(Material m) if needed.
 *  - fixedDraw(GLFunc gl, GLUFunc glu) : this is your drawing method.
 *  - the static method required for all the fragment classes, as the following (just
 * recopy it) :
 * public static Fragment[] fragmentsFromShapeElements(Shape newShape, FragmentedShapeElement[] ses) {
 *   return FixedFragmentElement3D.fragmentsFromShapeElements(newShape, ses);
 * }
 * 
 * You may want to override the following methods :
 *  - getUseAlpha() : if your fragment element use alpha blending, or if you want it to be drawn
 * at the end, in a second pass. The default implementation returns true if and only if the
 * material (returned by getMaterial()) use alpha.
 *  - fillCollector(FragmentsCollector f, Renderer r, float[] mat) : to perform special optimization
 * (= do not collect the fragment element if not needed).
 *  - reBuild() : if you want to do something each time the fragment element is modified. Do
 * not forget to call the super implementation of this method.
 * 
 * Add also your own properties and method; whenever the fragment element is changed, you
 * must call the reBuild() method.
 * 
 * The drawing is encapsulated into an openGL call list for a higher speed; do not assume
 * the drawing method to be called once each rendering. If you need that, extends rather
 * FragmentElement3D.
 * 
 * @see opale.soya.soya3d.model.Fragment
 * @see opale.soya.soya3d.model.FragmentedShapeElement
 * @see opale.soya.soya3d.Box
 * @see opale.soya.soya3d.FragmentElement3D
 * @author Artiste on the Web
 */

public abstract class FixedFragmentElement3D extends FragmentElement3D {
  /**
   * Creates a new fixed fragment element.
   */
  public FixedFragmentElement3D() {  }
  /**
   * Creates a new fixed fragment element, with the given name.
   */
  public FixedFragmentElement3D(String newName) { super(newName); }
  
  /**
   * Clones this fixed fragment element.
   * @return the clone.
   */
  public synchronized Object clone() {
    FragmentElement3D o = (FragmentElement3D) super.clone();
    return o;
  }
  
  public void finalize() { Soya.releaseCallListID(id); }
  
  // FragmentsCollectorFiller : Overrides :
  public void fillCollector(DrawablesCollector f, Renderer r, float[] mat) { // Optimizable
    if(visible) {
      if(!isBuilt) build(r.getGLFunc(), r.getGLUFunc());
      super.fillCollector(f, r, mat);
    }
  }

  // Fragment :
  /*
  public static Fragment[] fragmentsFromShapeElements(Shape newShape, FragmentedShapeElement[] ses) {
    return FragmentElement3D.fragmentsFromShapeElements(newShape, ses);
  }
  */
  
  private transient boolean isBuilt;
  /**
   * This method must be called when a property has changed.
   */
  public synchronized void reBuild() { isBuilt = false; }
  public synchronized void build(GLFunc gl, GLUFunc glu) {
    if(id == 0) id = gl.glGenLists(1);
    gl.glNewList(id, GLEnum.GL_COMPILE);
    fixedDraw(gl, glu);
    gl.glEndList();
    
    isBuilt = true;
  }
  /**
   * Draws the fragment element.
   * @param gl  the gl
   * @param glu the glu
   */
  public abstract void fixedDraw(GLFunc gl, GLUFunc glu);
  
  private transient int id;
  /**
   * Calls the call list.
   * @param r the renderer
   * @param gl  the gl
   * @param glu the glu
   */
  public final void draw(Renderer r, GLFunc gl, GLUFunc glu) {
    gl.glCallList(id);
  }
  
  public synchronized String propertiesString() {
    return super.propertiesString();
  }
}
