/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya3d.event.*;
import opale.soya.soya3d.model.*;
import opale.soya.soya3d.animation.*;
import gl4java.*;
import java.io.*;
import java.beans.*;

/**
 * Abstract class for all graphical 3D elements.
 * 
 * @author Artiste on the Web
 */

public abstract class GraphicalElement3D extends Element3D implements Position, Orientation, Dimension, CoordSyst, Animable, Transformable {
  private static final long serialVersionUID = -4323941202487688963l;
  
  public GraphicalElement3D()               { super(       ); Matrix.matrixIdentity(m); }
  public GraphicalElement3D(String newName) { super(newName); Matrix.matrixIdentity(m); }
  
  // Maybe we can add more constructors...
  
  protected SpecialEffect specialEffect;
  public SpecialEffect getSpecialEffect() { return specialEffect; }
  public void setSpecialEffect(SpecialEffect s) {
    specialEffect = s;
    firePropertyChange("specialEffect");
  }
  
  protected boolean visible = true;
  /**
   * Checks if this graphical element is visible. Non-visible elements are never drawn.
   * Default is true.
   * @return true if visible.
   */
  public boolean isVisible() { return visible; }
  /**
   * Sets if this graphical element is visible.
   * @param newVisible true for visible.
   */
  public void setVisible(boolean newVisible) {
    synchronized(this) { visible = newVisible; }
    firePropertyChange("visible");
  }
  
  /*
   * Gets the radius of this graphical element. The element is inside a sphere, defined by
   * the origin of the graphical element as center, and this radius.
   * The radius is unterstood as being in parent system coordinates.
   * This default implementation returns Float.POSITIVE_INFINITY.
   * @return the radius of this graphical element
   */
  //public float getRadius() { return Float.POSITIVE_INFINITY; }
  /*
   * Called when the radius must be recomputed.
   * Default implementation is no-op.
   */
  //protected void computeRadius() {  }
  
  // Overrides :
  /**
   * Clones this graphical 3D element. Does not clone animation properties.
   * @see opale.soya.soya3d.Element3D#clone
   * @return the clone
   */
  public synchronized Object clone() {
    GraphicalElement3D ge = (GraphicalElement3D) super.clone();
    System.arraycopy(m, 0, ge.m, 0, 16);
    ge.factorX = factorX;
    ge.factorY = factorY;
    ge.factorZ = factorZ;
    ge.incline = incline;
    ge.rotationType = rotationType;
    ge.leftHanded = leftHanded;
    ge.visible = visible;
    return ge;
  }
  
  public synchronized String propertiesString() {
    String s = super.propertiesString();
    s = s + "position : " + Float.toString(m[12]) + ", "
                          + Float.toString(m[13]) + ", "
                          + Float.toString(m[14]) + "\n";
    if(rotationType == Orientation.ROTATION_TYPE_INTERNAL) s = s + "rotationType : internal\n";
    else s = s + "rotationType : external\n";
    if(parent != null) {
      Position t = target();
      t.setCoordSyst(parent);
      s = s + "target : " + Float.toString(t.getX()) + ", "
                          + Float.toString(t.getY()) + ", "
                          + Float.toString(t.getZ()) + "\n";
    }
    s = s + "dimension : " + Float.toString(getWidth ()) + ", "
                           + Float.toString(getHeight()) + ", "
                           + Float.toString(getDepth ()) + "\n";
    s = s + "visible : " + visible + "\n";
    return s;
  }
  
  public void unlock() {
    synchronized(this) { 
      if(lockLevel == 0) return;
      lockLevel--;
    }
    if(lockLevel == 0) fireAnimationChange(); // Position, orientation and dimension may have changed.
  }
  protected void added(World3D in) {
    super.added(in);
    invalidatePreBuiltMatrixes();
  }
  protected void removed(World3D from) {
    super.removed(from);
    invalidatePreBuiltMatrixes();
    rootM = m; // No parent, so the matrix is also the root matrix.
  }
  
  // CoordSyst :
  public CoordSyst getRootCoordSyst() { return getRootParent(); }
  public CoordSyst getCoordSyst() { return parent; }
  public synchronized void setCoordSyst(CoordSyst newCoordSyst) { 
    if(newCoordSyst instanceof World3D) {
      Position p = clone(newCoordSyst);
      remove();
      ((World3D) newCoordSyst).add(this);
      move(p);
    }
    else throw new UnsupportedOperationException("setCoordSyst on a 3D element is only possible with a World3D as argument.");
  }
  protected final float[] m = new float[16];
  public float[] getMatrix() { return m; }
  protected transient float[] rootM;
  protected transient float[] invertedRootM;
  /**
   * Invalidates the pre built matrixes. They will be re-built when needed.
   * This method must be called if you change the matrix m (the method that change m
   * indirectly, like move, rotate, scale... do that).
   */
  protected synchronized void invalidatePreBuiltMatrixes() { // Invalid the pre-calculated matrix.
    rootM         = null; // Will be recalculed when needed.
    invertedRootM = null;
  }
  public float[] getRootMatrix() { // Optimizable by bypassing the matrix of the root world (which doesn't change anything : it's a noop identity matrix).
    World3D parent2;
    synchronized(this) {
      if(rootM != null) return rootM;
      if(parent == null) {
        rootM = m;
        return rootM;
      }
      parent2 = parent;
    }
    float[] n = parent2.getRootMatrix();
    rootM = Matrix.matrixMultiply(n, m);
    return rootM;
  }
  private void buildRootMatrix() {
    World3D parent2;
    synchronized(this) {
      if(rootM  != null) return; // Already done.
      if(parent == null) {
        rootM = m;
        return;
      }
      parent2 = parent;
    }
    rootM = Matrix.matrixMultiply(parent2.getRootMatrix(), m);
  }
  public float[] getInvertedRootMatrix() {
    buildInvertedRootMatrix();
    return invertedRootM;
  }
  private void buildInvertedRootMatrix() { // Also call buildRootMatrix().
    if(invertedRootM != null) return;
    buildRootMatrix();
    invertedRootM = Matrix.matrixInvert(rootM);
  }
  public void convertPointTo(float[] p) { // Optimizable
    buildInvertedRootMatrix();
    System.arraycopy(Matrix.pointMultiplyByMatrix(invertedRootM, p), 0, p, 0, 3);
  }
  public void convertPointFrom(float[] p) { // Optimizable
    buildRootMatrix();
    System.arraycopy(Matrix.pointMultiplyByMatrix(rootM, p), 0, p, 0, 3);
  }
  public void convertVectorTo(float[] p) { // Optimizable
    buildInvertedRootMatrix();
    System.arraycopy(Matrix.vectorMultiplyByMatrix(invertedRootM, p), 0, p, 0, 3);
  }
  public void convertVectorFrom(float[] p) { // Optimizable
    buildRootMatrix();
    System.arraycopy(Matrix.vectorMultiplyByMatrix(rootM, p), 0, p, 0, 3);
  }
  public void printMatrix() {
    buildInvertedRootMatrix();
    System.out.println("matrix : " + Matrix.matrixToString(m));
    System.out.println("root matrix : " + Matrix.matrixToString(rootM));
    System.out.println("inverted root matrix : " + Matrix.matrixToString(invertedRootM));
  }
  public void printRealMatrix() {
    invalidatePreBuiltMatrixes();
    buildInvertedRootMatrix();
    System.out.println("matrix : " + Matrix.matrixToString(m));
    System.out.println("root matrix : " + Matrix.matrixToString(rootM));
    System.out.println("inverted root matrix : " + Matrix.matrixToString(invertedRootM));
  }
  public Vector x () { return new Vector(1f, 0f, 0f, this); }
  public Vector y () { return new Vector(0f, 1f, 0f, this); }
  public Vector z () { return new Vector(0f, 0f, 1f, this); }
  public Position origin () { return new Point(0f, 0f, 0f, this); }
  public boolean isLeftHanded () { return  leftHanded; }
  public boolean isRightHanded() { return !leftHanded; }

  // Rotable :
  public void rotate(Position origin, Vector vector, float angle) {
    origin = convertToCoordSyst(origin);
    Position p = new Point(origin);
    p.addVector((Vector) convertToCoordSyst(vector));
    float[] o = { origin.getX(), origin.getY(), origin.getZ() };
    float[] ps = { p.getX(), p.getY(), p.getZ() };
    rotate(Matrix.matrixRotate(angle, o, ps));
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeMoveOrientateEvent(this));
  }
  public void rotate(Position p1, Position p2, float angle) {
    p1 = convertToCoordSyst(p1);
    p2 = convertToCoordSyst(p2);
    float[] p1s = { p1.getX(), p1.getY(), p1.getZ() };
    float[] p2s = { p2.getX(), p2.getY(), p2.getZ() };
    rotate(Matrix.matrixRotate(angle, p1s, p2s));
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeMoveOrientateEvent(this));
  }
  
  // Position :
  public float getX() { return m[12]; }
  public void setX(float newCoord) {
    addVector(newCoord - m[12], 0, 0);
    fireMove("x");
  }
  public float getY() { return m[13]; }
  public void setY(float newCoord) {
    addVector(0, newCoord - m[13], 0);
    fireMove("y");
  }
  public float getZ() { return m[14]; }
  public void setZ(float newCoord) {
    addVector(0, 0, newCoord - m[14]);
    fireMove("z");
  }
  public synchronized void minimize(Position p) {
    if((p.getCoordSyst() != parent) && (p.getCoordSyst() != null) && (parent != null)) clone(p.getCoordSyst()).minimize(p);
    else {
      if(p.getX() > m[12]) p.setX(m[12]);
      if(p.getY() > m[13]) p.setY(m[13]);
      if(p.getZ() > m[14]) p.setZ(m[14]);
    }
  }
  public synchronized void maximize(Position p) {
    if((p.getCoordSyst() != parent) && (p.getCoordSyst() != null) && (parent != null)) clone(p.getCoordSyst()).maximize(p);
    else {
      if(p.getX() < m[12]) p.setX(m[12]);
      if(p.getY() < m[13]) p.setY(m[13]);
      if(p.getZ() < m[14]) p.setZ(m[14]);
    }
  }
  public void move(float newX, float newY, float newZ) {
    addVector(newX - m[12], newY - m[13], newZ - m[14]);
    fireMove();
  }
  public void move(float[] pxyz) { move(pxyz[0], pxyz[1], pxyz[2]); }
  public void move(Position p) {
    p = convertToCoordSyst(p);
    move(p.getX(), p.getY(), p.getZ());
  }
  public void addVector(Vector v) {
    v = (Vector) convertToCoordSyst(v);
    addVector(v.getX(), v.getY(), v.getZ());
  }
  public void addVector(float x, float y, float z) {
    synchronized(this) { System.arraycopy(Matrix.matrixTranslate(m, x, y, z), 0, m, 0, 16); } // Optimizable
		invalidatePreBuiltMatrixes();
  }
  public float distanceTo(Position p) {
    p = convertToCoordSyst(p);
    synchronized(this) { return (float) Math.sqrt(Matrix.pow2(m[12] - p.getX()) + Matrix.pow2(m[13] - p.getY()) + Matrix.pow2(m[14] - p.getZ())); }
  }
  public float squareDistanceTo(Position p) {
    p = convertToCoordSyst(p);
    synchronized(this) { return (float) Matrix.pow2(m[12] - p.getX()) + Matrix.pow2(m[13] - p.getY()) + Matrix.pow2(m[14] - p.getZ()); }
  }
  /**
   * Converts the given position to the coordinates system of this graphical element. If they
   * are already defined in the same coordinates system, or if either of their coordinates
   * systems is null, the point is returned "as is".
   * @param p the position
   * @return a position in a coodinates system compatible with the one of this graphical
   * element (either null or this.getCoordSyst())
   */
  protected Position convertToCoordSyst(Position p) {
    CoordSyst f1 = getCoordSyst(), f2 = p.getCoordSyst();
    if((f1 != f2) && (f1 != null) && (f2 != null)) return p.clone(f1);
    else return p; // No conversion required.
  }
  public Position clone(CoordSyst cs) {
    Position p = new Point(this);
    p.setCoordSyst(cs);
    return p;
  }
  
  // Transformable :
  public void transform(float[] mat) {
    synchronized(this) { System.arraycopy(Matrix.matrixMultiply(m, mat), 0, m, 0, 16); }
    fireAnimationChange();
  }

  // Orientation :
  public Position target() { return new Point(0f, 0f, -1f, this); }
  private float incline;
  public float getInclineValue() { return incline; }
  public void setInclineValue(float f) {
    rotateInternal(Matrix.matrixRotateIncline (f - incline));
    incline = f;
  }
  private int rotationType = Orientation.ROTATION_TYPE_INTERNAL;
  public int getRotationType() { return rotationType; }
  public void setRotationType(int newRotationType) {
    synchronized(this) { rotationType = newRotationType; }
    firePropertyChange("rotationType");
  }
  public void resetOrientation() {
    synchronized(this) {
      m[ 0] = factorX;
      m[ 1] = 0f;
      m[ 2] = 0f;
      m[ 3] = 0f;
      m[ 4] = 0f;
      m[ 5] = factorY;
      m[ 6] = 0f;
      m[ 7] = 0f;
      m[ 8] = 0f;
      m[ 9] = 0f;
      m[10] = factorZ;
      m[11] = 0f;
      m[15] = 1f;
      incline = 0;
    }
    invalidatePreBuiltMatrixes();
    fireOrientate();
  }
  public void rotateLateral (float angle) {
    if(rotationType == Orientation.ROTATION_TYPE_EXTERNAL) rotate(Matrix.matrixRotateLateral (angle));
    else rotateInternal(Matrix.matrixRotateLateral (angle));
  }
  public void rotateVertical(float angle) {
    if(rotationType == Orientation.ROTATION_TYPE_EXTERNAL) rotate(Matrix.matrixRotateVertical (angle));
    else rotateInternal(Matrix.matrixRotateVertical(angle));
  }
  public void rotateIncline (float angle) {
    if(rotationType == Orientation.ROTATION_TYPE_EXTERNAL) rotate(Matrix.matrixRotateIncline (angle));
    else rotateInternal(Matrix.matrixRotateIncline (angle));
    synchronized(this) { incline = incline + angle; }
  }
  public void rotate(Vector axe, float angle) {
    CoordSyst f = axe.getCoordSyst();
    synchronized(this) {
      if((f != this) && (f != null)) {
        axe = new Vector(axe);
        axe.setCoordSyst(this);
      }
    }
    rotate(axe.getX(), axe.getY(), axe.getZ(), angle);
  }
  public void rotate(float ax, float ay, float az, float angle) {
    rotate(Matrix.matrixRotate(angle, ax, ay, az));
  }
  private void rotate(float[] rotationMatrix) { // Optimizable (avoid having to save the translation by performing all calcul only in the 3*3 part of the 4*4 matrix).
    synchronized(this) {
      float[] t = { m[12], m[13], m[14] }; // Save translation.
      float[] temp = null;
      m[12] = 0f;
      m[13] = 0f;
      m[14] = 0f;
      temp = Matrix.matrixScale(Matrix.matrixMultiply(rotationMatrix, Matrix.matrixScale(m, 1f / factorX, 1f / factorY, 1f / factorZ)), factorX, factorY, factorZ);
      System.arraycopy(temp, 0, m, 0, 12);
      System.arraycopy(t, 0, m, 12, 3);
    }
    invalidatePreBuiltMatrixes();
    fireOrientate();
  }
  private void rotateInternal(float[] rotationMatrix) { // Optimizable (avoid having to save the translation by performing all calcul only in the 3*3 part of the 4*4 matrix).
    synchronized(this) {
      float[] t = { m[12], m[13], m[14] }; // Save translation.
      float[] temp = null;
      m[12] = 0f;
      m[13] = 0f;
      m[14] = 0f;
      //temp = Matrix.matrixMultiply(m, rotationMatrix);
      temp = Matrix.matrixScale(Matrix.matrixMultiply(Matrix.matrixScale(m, 1f / factorX, 1f / factorY, 1f / factorZ), rotationMatrix), factorX, factorY, factorZ);
      System.arraycopy(temp, 0, m, 0, 12);
      System.arraycopy(t, 0, m, 12, 3);
    }
    invalidatePreBuiltMatrixes();
    fireOrientate();
  }
  public void lookAt(Vector v) {
    Point p = new Point(this);
    p.addVector(v);
    lookAt(p);
  }
  public void lookAt(Position p) { // Optimizable (a lot!).
    lock(); // Avoid firing many events.
    float aincline = incline;
    setInclineValue(0f);
    
    Position p2 = null, p3;
    Vector v1, v2, axe;
    float angle;
    synchronized(this) { 
      if(p.getCoordSyst() == parent) p2 = p;
      else p2 = p.clone(parent);
      p3 = p2.clone(this);
      
      // v1 : current orienation; v2 : desired.
      v1 = new Vector(0f, 0f, -1f);
      v2 = new Vector(p3.getX(), 0f, p3.getZ());
      axe = v1.crossProduct(v2);
      if(axe.isNullVector()) axe.move(1f, 0f, 0f); // Arbitrary values to avoid error. axe == vector null if angle is PI or 0...
      angle = v1.angle(v2);
      
      if(axe.getY() < 0) angle = -angle;
    }
    rotate(Matrix.matrixRotateLateral(angle));
    
    synchronized(this) { 
      p3 = p2.clone(this);
      
      v1.move(0f, 0f, -1f);
      v2.move(0f, p3.getY(), p3.getZ());
      axe = v2.crossProduct(v1);
      if(axe.isNullVector()) axe.move(1f, 0f, 0f); // Arbitrary values to avoid error.
      angle = v2.angle(v1);
      
      if(axe.getX() > 0) angle = -angle;
    }
    rotateInternal(Matrix.matrixRotateVertical(angle));
    
    setInclineValue(aincline);
    unlock();
    fireOrientate("target");
  }
  
  // For bean edition :
  public float getTargetX() { return target().clone(parent).getX() - m[12]; }
  public float getTargetY() { return target().clone(parent).getY() - m[13]; }
  public float getTargetZ() { return target().clone(parent).getZ() - m[14]; }
  public void setTargetX(float f) {
    Position p;
    synchronized(this) { 
      p = target().clone(parent);
      p.setX(f + m[12]);
    }
    lookAt(p);
  }
  public void setTargetY(float f) {
    Position p;
    synchronized(this) { 
      p = target().clone(parent);
      p.setY(f + m[13]);
    }
    lookAt(p);
  }
  public void setTargetZ(float f) {
    Position p;
    synchronized(this) { 
      p = target().clone(parent);
      p.setZ(f + m[14]);
    }
    lookAt(p);
  }
  
  // Dimension :
  protected boolean leftHanded;
  private synchronized void defineLeftHanded() { leftHanded = (factorX * factorY * factorZ) < 0; } // Left handed if an even number of dimension are negative.
  /**
   * The relative dimension : the product of all scale factors that have been apllied.
   * Default is (1, 1, 1) (=No scaling).
   */
  protected float factorX = 1f, factorY = 1f, factorZ = 1f;
  public float getXFactor() { return factorX; }
  public float getYFactor() { return factorY; }
  public float getZFactor() { return factorZ; }
  public void setXFactor(float f) { scale(f / factorX, 1f, 1f); }
  public void setYFactor(float f) { scale(1f, f / factorY, 1f); }
  public void setZFactor(float f) { scale(1f, 1f, f / factorZ); }
  public void setWidth(float newDim) {
    float width = getWidth();
    if(width != 0)  scale(newDim / width,  1f, 1f);
  }
  public void setHeight(float newDim) {
    float height = getHeight();
    if(height != 0) scale(1f, newDim / height, 1f);
  }
  public void setDepth(float newDim) {
    float depth = getDepth();
    if(depth != 0) scale(1f, 1f, newDim / depth);
  }
  public void setDims(float w, float h, float d) {
    scale(w / getWidth(), h / getHeight(), d / getDepth());
  }
  public void scale(float f) { scale(f, f, f); }
  public void scale(float fx, float fy, float fz) {
    float[] temp = Matrix.matrixScale(m, fx, fy, fz);
    synchronized(this) {
      factorX = factorX * fx;
      factorY = factorY * fy;
      factorZ = factorZ * fz;
      System.arraycopy(temp, 0, m, 0, 16);
      defineLeftHanded();
    }
    invalidatePreBuiltMatrixes();
    fireResize();
  }
  public abstract float getWidth ();   // Returns the corresponding dimensions.
  public abstract float getHeight();  // Must be overriden by the child class.
  public abstract float getDepth (); // For example see Volume3D.
  
  // Animable :
  protected Movement movement;
  public Movement getMovement() { return movement; }
  public synchronized void setMovement(Movement m) { movement = m; }
  public float getAnimationTime() { return parent.getAnimationTime(); }
  public void setAnimationTime(float time) {
    if(Float.isNaN(time) || (movement == null)) return;
    setState(movement.getState(time));
    fireAnimationChange();
  }
  public void setState(InterpolatedState s) {
    if(s == null) return;
    synchronized(this) {
      System.arraycopy(s.matrix, 0, m, 0, 16);
      incline = s.inclineValue;
      factorX = s.width ;
      factorY = s.height;
      factorZ = s.depth ;
      defineLeftHanded();
    }
    invalidatePreBuiltMatrixes();
  }
  
  // Events :
  public void fireMove() {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeMoveEvent(this));
  }
  public void fireMove(String propertyName) {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeMoveEvent(this, propertyName));
  }
  public void fireMove(String propertyName, Object oldValue, Object newValue) {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeMoveEvent(this, propertyName, oldValue, newValue));
  }
  public void fireOrientate() {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeOrientateEvent(this));
  }
  public void fireOrientate(String propertyName) {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeOrientateEvent(this, propertyName));
  }
  public void fireOrientate(String propertyName, Object oldValue, Object newValue) {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeOrientateEvent(this, propertyName, oldValue, newValue));
  }
  public void fireResize() {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeResizeEvent(this));
  }
  public void fireResize(String propertyName) {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeResizeEvent(this, propertyName));
  }
  public void fireResize(String propertyName, Object oldValue, Object newValue) {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeResizeEvent(this, propertyName, oldValue, newValue));
  }
  public void fireAnimationChange() {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeAnimationEvent(this));
  }
}
