/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.font.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import opale.soya.soya3d.model.*;
import gl4java.*;

/**
 * A label is a 2D text drawn at a 3D position.
 * 
 * @author Artiste on the Web
 */

public class Label3D extends FixedFragmentElement3D implements Colored {
  /**
   * Creates a new label.
   */
  public Label3D() { super(); }
  /**
   * Creates a new label, with the given text.
   * @param text the text
   */
  public Label3D(String text) {
    this();
    setText(text);
  }
  
  /**
   * clone this label.
   * Warning : the font is not cloned but just referenced;
   * @return a clone
   */
  public Object clone() {
    Label3D t = null;
    try { t = (Label3D) getClass().newInstance(); }
    catch(Exception e) { System.out.println("Cannot create a new object : " + getClass().getName()); return null; }
    t.font      = font     ;
    t.overlayed = overlayed;
    t.lit       = lit      ;
    t.setColor(color);
    return t;
  }
  
  // Dimension :
  /** not supported. */
  public float getNaturalWidth () { return 0f; }
  /** not supported. */
  public float getNaturalHeight() { return 0f; }
  public float getNaturalDepth () { return 0f; }
  public DimensionWrapper wrapper() { return new Box(origin(), origin()); }
  
  public Material getMaterial() { return Material.WHITE_MATERIAL; }
  
  private boolean overlayed;
  /**
   * Checks if this label is overlayed. An overlayed label is drawn over any other fragment,
   * even if it is'nt the closest from the viewpoint.
   * Default is false.
   * @return true if overlayed
   */
  public boolean isOverlayed() { return overlayed; }
  /**
   * Sets if this label is overlayed.
   * @param b true for overlayed
   */
  public synchronized void setOverlayed(boolean b) {
    overlayed = b;
    reBuild();
    firePropertyChange("overlayed");
  }
  public boolean getUseAlpha() {
    if(overlayed) return true;
    return color[3] !=1;
  }

  private boolean lit = true;
  /**
   * Checks if this text is lit. Set it to false if you want this text to be always visible
   * (even if there is no light).
   * Default value is true.
   * @return true if lit
   */
  public boolean isLit() { return lit; }
  /**
   * Sets if this text is lit.
   * @param b true if lit
   */
  public synchronized void setLit(boolean b) {
    lit = b;
    reBuild();
    firePropertyChange("lit");
  }

  private String text;
  /**
   * Gets the text of this label.
   * Default is null .
   * @return the text
   */
  public String getText() { return text; }
  /**
   * Sets the text of this label.
   * @param s the new text
   */
  public synchronized void setText(String s) {
    text = s;
    reBuild();
    firePropertyChange("text");
  }
  
  private Font2D font = DefaultFont2D.HELVETICA_10;
  /**
   * Gets the font of this label.
   * Default is DefaultFont2D.HELVETICA_10 .
   * @return the font
   */
  public Font2D getFont() { return font; }
  /**
   * Sets the font of this label.
   * @param f the font
   */
  public synchronized void setFont(Font2D f) {
    font = f;
    reBuild();
    firePropertyChange("font");
  }
  
  private final float[] color = (float[]) Material.WHITE_COLOR.clone();
  public boolean getUseColor() { return true; }
  public float[] getColor() {	return color; }
  public synchronized void setColor(float red, float green, float blue) { setColor(red, green, blue, 1f); }
  public synchronized void setColor(float red, float green, float blue, float alpha) {
    color[0] = red;
    color[1] = green;
    color[2] = blue;
    color[3] = alpha;
    reBuild();
    firePropertyChange("color");
  }
  public synchronized void setColor(float[] c) {
    System.arraycopy(c, 0, color, 0, 4);
    reBuild();
    firePropertyChange("color");
  }
  public float getRed  () { return color[0]; }
  public float getGreen() { return color[1]; }
  public float getBlue () { return color[2]; }
  public float getAlpha() { return color[3]; }
  public synchronized void setRed  (float f) {
    color[0] = f;
    reBuild();
    firePropertyChange("color");
  }
  public synchronized void setGreen(float f) {
    color[1] = f;
    reBuild();
    firePropertyChange("color");
  }
  public synchronized void setBlue (float f) {
    color[2] = f;
    reBuild();
    firePropertyChange("color");
  }
  public synchronized void setAlpha(float f) {
    color[3] = f;
    reBuild();
    firePropertyChange("color");
  }
  
  /**
   * Draws the text, using the given gl and glu.
   * @param gl  the gl
   * @param glu the gl
   */
  public synchronized void fixedDraw(GLFunc gl, GLUFunc glu) {
    if(text != null) {
      if(overlayed) gl.glDisable(GLEnum.GL_DEPTH_TEST);
      if(!lit)      gl.glDisable(GLEnum.GL_LIGHTING  );
      gl.glMaterialfv(GLEnum.GL_FRONT_AND_BACK, GLEnum.GL_AMBIENT_AND_DIFFUSE, color);
      font.draw(gl, glu, text, 0f, 0f, 0f);
      gl.glMaterialfv(GLEnum.GL_FRONT_AND_BACK, GLEnum.GL_AMBIENT_AND_DIFFUSE, Material.WHITE_COLOR); // Must be reseted to default for Material.NO_MATERIAL.
      if(!lit)      gl.glEnable (GLEnum.GL_LIGHTING  );
      if(overlayed) gl.glEnable (GLEnum.GL_DEPTH_TEST);
    }
  }
}
