/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya3d.event.*;
import java.io.*;
import java.beans.*;

/**
 * A point is a basic position.
 * 
 * @author Artiste on the Web
 */

public class Point extends AbstractBean implements Position {
  /**
   * Creates a new point.
   */
  public Point() {  }
  /**
   * Creates a new point.
   * @param newX the x coordinate
   * @param newY the y coordinate
   * @param newZ the z coordinate
   */
  public Point(float newX, float newY, float newZ) { this(newX, newY, newZ, null); }
  /**
   * Creates a new point.
   * @param newX the x coordinate
   * @param newY the y coordinate
   * @param newZ the z coordinate
   * @param newCoordSyst the coordinates system where the vector is defined
   */
  public Point(float newX, float newY, float newZ, CoordSyst newCoordSyst) {
    x = newX;
    y = newY;
    z = newZ;
    coordSyst = newCoordSyst;
  }
  /**
   * Creates a new point.
   * @param p the position
   */
  public Point(Position p) {
    x = p.getX();
    y = p.getY();
    z = p.getZ();
    coordSyst = p.getCoordSyst();
  }
  public Object clone() { return new Point(this); }

  private CoordSyst coordSyst;
  public CoordSyst getCoordSyst() { return coordSyst; }
  public void setCoordSyst(CoordSyst newCoordSyst) {
    synchronized(this) {
      if((coordSyst != null) && (newCoordSyst != null)) {
        mutedTransform(coordSyst.getRootMatrix());
        mutedTransform(newCoordSyst.getInvertedRootMatrix());
      }
      coordSyst = newCoordSyst;
    }
    fireMove("coordSyst");
  }

  public synchronized Position clone(CoordSyst f) {
    Position p = new Point(this);
    p.setCoordSyst(f);
    return p;
  }

  private float x, y, z;
  public float getX() { return x; }
  public void setX(float newCoord) {
    x = newCoord;
    fireMove("x");
  }
  public float getY() { return y; }
  public void setY(float newCoord) {
    y = newCoord;
    fireMove("y");
  }
  public float getZ() { return z; }
  public void setZ(float newCoord) {
    z = newCoord;
    fireMove("z");
  }
  public void move(float newX, float newY, float newZ) {
    x = newX;
    y = newY;
    z = newZ;
    fireMove();
  }
  public void move(Position p) {
    p = convertToCoordSyst(p);
    x = p.getX();
    y = p.getY();
    z = p.getZ();
    fireMove();
  }
  public void addVector(Vector v) {
    v = (Vector) convertToCoordSyst(v);
    x = x + v.getX();
    y = y + v.getY();
    z = z + v.getZ();
    fireMove();
  }
  public void addVector(float vx, float vy, float vz) {
    x = x + vx;
    y = y + vy;
    z = z + vz;
    fireMove();
  }

  public float distanceTo(Position p) {
    p = convertToCoordSyst(p);
    return (float) Math.sqrt(Matrix.pow2(x - p.getX()) + Matrix.pow2(y - p.getY()) + Matrix.pow2(z - p.getZ()));
  }
  public float squareDistanceTo(Position p) {
    p = convertToCoordSyst(p);
    return (float) Matrix.pow2(x - p.getX()) + Matrix.pow2(y - p.getY()) + Matrix.pow2(z - p.getZ());
  }

  protected Position convertToCoordSyst(Position p) {
    CoordSyst f = p.getCoordSyst();
    if((f != coordSyst) && (f != null) && (coordSyst != null)) return p.clone(coordSyst);
    else return p;
  }
  
  public void minimize(Position p) {
    if((p.getCoordSyst() != coordSyst) && (p.getCoordSyst() != null) && (coordSyst != null)) clone(p.getCoordSyst()).minimize(p);
    else {
      if(p.getX() > x) p.setX(x);
      if(p.getY() > y) p.setY(y);
      if(p.getZ() > z) p.setZ(z);
    }
  }
  public void maximize(Position p) {
    if((p.getCoordSyst() != coordSyst) && (p.getCoordSyst() != null) && (coordSyst != null)) clone(p.getCoordSyst()).maximize(p);
    else {
      if(p.getX() < x) p.setX(x);
      if(p.getY() < y) p.setY(y);
      if(p.getZ() < z) p.setZ(z);
    }
  }
  
  public void define(Position p) {
    x = p.getX();
    y = p.getY();
    z = p.getZ();
    coordSyst = p.getCoordSyst();
    fireMove();
  }
  
  public String toString() {
    return "Point { " + Float.toString(x) + ", " + Float.toString(y) + ", " + Float.toString(z) + " }";
  }
  
  // Rotable :
  public void rotate(Position origin, Vector vector, float angle) {
    // TODO Pac, je compte sur toi :-)
  }
  public void rotate(Position p1, Position p2, float angle) {
    // TODO Pac, je compte sur toi :-)
  }
  
  // Transformable :
  public void transform(float[] m) {
    mutedTransform(m);
    fireMove();
  }
  private void mutedTransform(float[] m) {
    float ax = x, ay = y, az = z;
    x = ax * m[0] + ay * m[4] + az * m[ 8] + m[12];
    y = ax * m[1] + ay * m[5] + az * m[ 9] + m[13];
    z = ax * m[2] + ay * m[6] + az * m[10] + m[14];
  }
  public void scale(float f) {
    x = f * x;
    y = f * y;
    z = f * z;
    fireMove();
  }
  public void scale(float fx, float fy, float fz) {
    x = fx * x;
    y = fy * y;
    z = fz * z;
    fireMove();
  }
  
  // Overrides :
  public boolean equals(Object o) {
    if(o instanceof Position) {
      Position p = convertToCoordSyst((Position) o);
      if(Matrix.pow2(x - p.getX()) + Matrix.pow2(y - p.getY()) + Matrix.pow2(z - p.getZ()) < 0.00001f) return true;
    }
    return false;
  }
  
  // Event :
  public void fireMove() {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeMoveEvent(this));
  }
  public void fireMove(String propertyName) {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeMoveEvent(this, propertyName));
  }
  public void fireMove(String propertyName, Object oldValue, Object newValue) {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeMoveEvent(this, propertyName, oldValue, newValue));
  }
}
