/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d;

import opale.soya.*;
import opale.soya.util.*;
import java.beans.*;

/**
 * Interface for any object that has a position in 3D.
 * A position correspond to 3 float coordinates AND a reference to the coordinates system in
 * witch those coordinates are defined.
 * 
 * @author Artiste on the web
 */

public interface Position extends Bean, Rotable, Transformable {
  /**
   * Returns the coordinates system in witch the position's coordinates are defined.
   * @return the coordinates system
   */
  public CoordSyst getCoordSyst();
  /**
   * Sets the coordinates system in witch the position's coordinates are defined.
   * The coordinates and converted to the new coordinates system. If you don't want a
   * conversion, use setCoordSyst(null); setCoordSyst(newCoordSyst); .
   * @param newCoordSyst the new coordinates system
   */
  public void setCoordSyst(CoordSyst newCoordSyst);
  
  /**
   * Returns a point clone of this position, defined into the given frame.
   * If required, a frame coordinates convertion is performed.
   * @return the point clone
   */
  public Position clone(CoordSyst f);
  
  /**
   * Returns the x coordinate.
   * @return the x coordinate
   */
  public float getX();
  /**
   * Sets the x coordinate.
   * @param newCoord the new x coordinate
   */
  public void setX(float newCoord);
  
  /**
   * Returns the y coordinate.
   * @return the y coordinate
   */
  public float getY();
  /**
   * Sets the y coordinate.
   * @param newCoord the new y coordinate
   */
  public void setY(float newCoord);
  
  /**
   * Returns the z coordinate.
   * @return the z coordinate
   */
  public float getZ();
  /**
   * Sets the z coordinate.
   * @param newCoord the new z coordinate
   */
  public void setZ(float newCoord);
  
  /**
   * Moves the position.
   * Assumes that the given coordinates are defined in getFrame() frame-coordinate.
   * @param newX the new x coordinate
   * @param newY the new y coordinate
   * @param newZ the new z coordinate
   */
  public void move(float newX, float newY, float newZ);
  /**
   * Moves the position.
   * If required, a frame coordinates convertion is performed.
   * @param p the new position
   */
  public void move(Position p);
  
  /**
   * Moves this position by the given vector. For point, it's a translation; for vector it's
   * means vectorial addition.
   * If required, a frame coordinates convertion is performed.
   * @param v the vector
   */
  public void addVector(Vector v);
  /**
   * Moves this position by the given vector. For point, it's a translation; for vector it's
   * means vectorial addition.
   * @param vx the vector x coordinate
   * @param vy the vector y coordinate
   * @param vz the vector z coordinate
   */
  public void addVector(float vx, float vy, float vz);
  
  /**
   * Minimizes the given position by the coordinates of the position.
   * @param p the position
   */
  public void minimize(Position p);
  /**
   * Maximizes the given position by the coordinates of the position.
   * @param p the position
   */
  public void maximize(Position p);
  
  /**
   * Returns the distance between this position and another.
   * If required, a frame coordinates convertion is performed.
   * @param p the other position
   * @return the distance
  */
  public float distanceTo(Position p);
  /**
   * Returns the square distance between this position and another.
   * If required, a frame coordinates convertion is performed.
   * This method may be quicker than distanceTo(Position). The square distance can be used
   * to compare distance.
   * @param p the other position
   * @return the square distance
   */
  public float squareDistanceTo(Position p);
  
  /**
   * Fires a move property change event. Such an event is automatically fired when the
   * position is moved; you shoud not have to use this method.
   * @see opale.soya.soya3d.event.MoveEvent
   */
  public void fireMove();
  /**
   * Fires a move property change event for the given property name. Such an event is
   * automatically fired when the position is moved; you shoud not have to use this method.
   * @see opale.soya.soya3d.event.MoveEvent
   */
  public void fireMove(String propertyName);
  /**
   * Fires a move property change event for the given property name, and the given old and
   * new values. Such an event is automatically fired when the position is moved; you shoud
   * not have to use this method.
   * @see opale.soya.soya3d.event.MoveEvent
   */
  public void fireMove(String propertyName, Object oldValue, Object newValue);
}
