/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya3d.event.*;
import gl4java.*;
import java.io.*;
import java.beans.*;
import java.lang.Math;

/**
 * A vector is a vector-like position.
 * 
 * @author Artiste on the Web
 */

public class Vector extends AbstractBean implements Position {
  /**
   * Creates a new vector.
   */
  public Vector() { super(); }
  /**
   * Creates a new vector.
   * @param newX the x coordinate
   * @param newY the y coordinate
   * @param newZ the z coordinate
   */
  public Vector(float newX, float newY, float newZ) { this(newX, newY, newZ, null); }
  /**
   * Creates a new vector.
   * @param newX the x coordinate
   * @param newY the y coordinate
   * @param newZ the z coordinate
   * @param newCoordSyst the coordinates system where the vector is defined
   */
  public Vector(float newX, float newY, float newZ, CoordSyst newCoordSyst) {
    x = newX;
    y = newY;
    z = newZ;
    coordSyst = newCoordSyst;
  }
  /**
   * Creates a new vector.
   * @param p the vector position
   */
  public Vector(Position p) {
    x = p.getX();
    y = p.getY();
    z = p.getZ();
    coordSyst = p.getCoordSyst();
  }
  /**
   * Creates a new vector from 2 points.
   * @param start the start point
   * @param end the end point
   */
  public Vector(Position start, Position end) {
    CoordSyst f = end.getCoordSyst();
    if(start.getCoordSyst() != f) start = start.clone(f);
    x = end.getX() - start.getX();
    y = end.getY() - start.getY();
    z = end.getZ() - start.getZ();
    coordSyst = f;
  }
  public Object clone() {
    return new Vector(x, y, z, coordSyst);
    //return new Vector(this);
  }
  
  // Position :
  private CoordSyst coordSyst;
  public CoordSyst getCoordSyst() { return coordSyst; }
  public void setCoordSyst(CoordSyst newCoordSyst) {
    if((coordSyst != null) && (newCoordSyst != null)) {
      mutedTransform(coordSyst.getRootMatrix());
      mutedTransform(newCoordSyst.getInvertedRootMatrix());
    }
    coordSyst = newCoordSyst;
    fireMove("coordSyst");
  }

  public Position clone(CoordSyst f) {
    Position p = new Vector(this);
    p.setCoordSyst(f);
    return p;
  }

  private float x, y, z;
  public float getX() { return x; }
  public void setX(float newCoord) {
    x = newCoord;
    fireMove("x");
  }
  public float getY() { return y; }
  public void setY(float newCoord) {
    y = newCoord;
    fireMove("y");
  }
  public float getZ() { return z; }
  public void setZ(float newCoord) {
    z = newCoord;
    fireMove("z");
  }
  public void move(float newX, float newY, float newZ) {
    x = newX;
    y = newY;
    z = newZ;
    fireMove();
  }
  public void move(Position p) {
    p = convertToCoordSyst(p);
    x = p.getX();
    y = p.getY();
    z = p.getZ();
    fireMove();
  }
  public void addVector(Vector v) {
    v = (Vector) convertToCoordSyst(v);
    x = x + v.getX();
    y = y + v.getY();
    z = z + v.getZ();
    fireMove();
  }
  public void addVector(float vx, float vy, float vz) {
    x = x + vx;
    y = y + vy;
    z = z + vz;
    fireMove();
  }

  public void minimize(Position p) {
    if((p.getCoordSyst() != coordSyst) && (p.getCoordSyst() != null) && (coordSyst != null)) clone(p.getCoordSyst()).minimize(p);
    else {
      if(x < 0f) p.setX(p.getX() + x);
      if(y < 0f) p.setY(p.getY() + y);
      if(z < 0f) p.setZ(p.getZ() + z);
    }
  }
  public void maximize(Position p) {
    if((p.getCoordSyst() != coordSyst) && (p.getCoordSyst() != null) && (coordSyst != null)) clone(p.getCoordSyst()).maximize(p);
    else {
      if(x > 0f) p.setX(p.getX() + x);
      if(y > 0f) p.setY(p.getY() + y);
      if(z > 0f) p.setZ(p.getZ() + z);
    }
  }

  public float distanceTo(Position p) {
    p = convertToCoordSyst(p);
    return (float) Math.sqrt(Matrix.pow2(x - p.getX()) + Matrix.pow2(y - p.getY()) + Matrix.pow2(z - p.getZ()));
  }
  public float squareDistanceTo(Position p) {
    p = convertToCoordSyst(p);
    return (float) Matrix.pow2(x - p.getX()) + Matrix.pow2(y - p.getY()) + Matrix.pow2(z - p.getZ());
  }

  /**
   * Gets the length of this vector.
   * return the lenght
   */
  public float length() {
    return (float) Math.sqrt(x * x + y * y + z * z);
  }
  /**
   * Checks if this vector is null.
   * return true if null
   */
  public boolean isNullVector() { return length() == 0; }
  /**
   * Checks if this vector is normalized.
   * return true if normalized
   */
  public boolean isNormalized() { return Math.abs(length() - 1f) < .00001f; }
  /**
   * Normalizes this vector.
   */
  public void normalize() {
    float f = length();
    x = x / f;
    y = y / f;
    z = z / f;
    fireMove();
  }
  /**
   * Return a normalized clone of this vector.
   * @return a normalized clone
   */
  public Vector cloneNormalized() {
    Vector v = new Vector(this);
    v.normalize();
    return v;
  }
  /**
   * Oppose this vector (multiplies all coordinates by -1).
   */
  public void oppose() {
    x = -x;
    y = -y;
    z = -z;
    fireMove();
  }
  /**
   * Returns the dot product of this vector by the given vector.
   * @param v the other vector
   * @return the dot product
   */
  public float dotProduct(Vector v) {
    v = convertToCoordSyst(v);
    return x * v.x + y * v.y + z * v.z;
  }
  /**
   * Returns the cross product of this vector by the given vector.
   * @param v the other vector
   * @return the cross product
   */
  public Vector crossProduct(Vector v) {
    v = convertToCoordSyst(v);
    return new Vector(y * v.z - z * v.y,
                      z * v.x - x * v.z,
                      x * v.y - y * v.x, coordSyst);
  }
  /**
   * Returns the absolute angle between this vector and the given one.
   * @param v the other vector
   * @return the angle in degree
   */
  public float angle(Vector v) {
    v = convertToCoordSyst(v);
    float s = length() * v.length();
    if(s == 0f) return 0f;
    float f = dotProduct(v) / s;
    if(f >=  1f) return   0f;
    if(f <= -1f) return 180f;
    return (float) ((Math.atan(-f / Math.sqrt(1f - f * f)) + Matrix.PI / 2f) * 180f / Matrix.PI);
  }
  /**
   * Rotates this vector around the given axis.
   * @param axe the axe
   * @param angle th angle
   */
  public void rotate(Vector axe, float angle) {
    axe = convertToCoordSyst(axe);
    float[] m = Matrix.matrixRotate(angle, axe.x, axe.y, axe.z);
		
    move(x * m[0] + y * m[4] + z * m[ 8],
         x * m[1] + y * m[5] + z * m[ 9],
         x * m[2] + y * m[6] + z * m[10]);
  }
  
  // Rotable :
  public void rotate(Position origin, Vector vector, float angle) {
    // TODO Pac, je compte sur toi :-)
  }
  public void rotate(Position p1, Position p2, float angle) {
    // TODO Pac, je compte sur toi :-)
  }
  
  // Transformable :
  public void transform(float[] m) {
    mutedTransform(m);
    fireMove();
  }
  private void mutedTransform(float[] m) {
    float ax = x, ay = y, az = z;
    x = ax * m[0] + ay * m[4] + az * m[ 8];
    y = ax * m[1] + ay * m[5] + az * m[ 9];
    z = ax * m[2] + ay * m[6] + az * m[10];
  }
  public void scale(float f) {
    x = f * x;
    y = f * y;
    z = f * z;
    fireMove();
  }
  public void scale(float fx, float fy, float fz) {
    x = fx * x;
    y = fy * y;
    z = fz * z;
    fireMove();
  }
  
  private Position convertToCoordSyst(Position v) {
    CoordSyst f = v.getCoordSyst();
    if((f != coordSyst) && (f != null) && (coordSyst != null)) return v.clone(coordSyst);
    else return v;
  }
  private Vector convertToCoordSyst(Vector v) {
    CoordSyst f = v.getCoordSyst();
    if((f != coordSyst) && (f != null) && (coordSyst != null)) return (Vector) v.clone(coordSyst);
    else return v;
  }

  public String toString() {
    return "Vector { " + Float.toString(x) + ", " + Float.toString(y) + ", " + Float.toString(z) + " }";
  }

  public void fireMove() {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeMoveEvent(this));
  }
  public void fireMove(String propertyName) {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeMoveEvent(this, propertyName));
  }
  public void fireMove(String propertyName, Object oldValue, Object newValue) {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeMoveEvent(this, propertyName, oldValue, newValue));
  }
}
