/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya3d.model.*;
import gl4java.*;
import java.util.Iterator;
import java.io.*;
import java.beans.*;

/**
 * A volume is a graphical 3D element that draws something at the screen. What is drawn is
 * not described by the volume itself but rather by its shape. The volume describe only
 * the position, orientation, dimension, ...
 * 
 * The same shape can be shared between many volume.
 *
 * @see opale.soya.soya3d.model.Shape
 * @see opale.soya.soya3d.model.FragmentedShape
 * @author Artiste on the Web
 */

public class Volume3D extends GraphicalElement3D implements Fragmentable, DrawablesCollectorFiller, RaypickingComplex {
  private static final long serialVersionUID = -2627584441031401789l;
  
  /**
   * Creates a new volume.
   */
  public Volume3D()                                    { super()          ;            }
  /**
   * Creates a new volume with the given shape.
   * @param s the shape
   */
  public Volume3D(Shape s)                             { super()          ; shape = s; }
  /**
   * Creates a new volume with the given name.
   * @param newName the name
   */
  public Volume3D(String newName)                      { super(newName)   ;            }
  /**
   * Creates a new volume with the given name and shape.
   * @param newName the name
   * @param s the shape
   */
  public Volume3D(String newName, Shape s)             { super(newName)   ; shape = s; }
  
  /**
   * Clones this volume. Its shape is not cloned; the original volume and its clone will
   * share the same shape.
   * @see opale.soya.soya3d.GraphicalElement3D#clone
   * @see opale.soya.soya3d.Element3D#clone
   * @return the clone
   */
  public synchronized Object clone() {
    Volume3D v = (Volume3D) super.clone();
    v.shape = shape;
    v.solid = solid;
    return v;
  }

  // Dimension :
  public synchronized float getWidth () {
    if(shape == null) return 0f;
    else return factorX * shape.getWidth ();
  }
  public synchronized float getHeight() {
    if(shape == null) return 0f;
    else return factorY * shape.getHeight();
  }
  public synchronized float getDepth () {
    if(shape == null) return 0f;
    else return factorZ * shape.getDepth ();
  }
  public synchronized DimensionWrapper wrapper() {
    if(shape == null) {
      Position p = new Point(0f, 0f, 0f, this);
      return new Box(p, p);
    }
    else {
      /* old and odd code...???
      DimensionWrapper b = shape.wrapper();
      b.setCoordSyst(null);
      b.setCoordSyst(this);
      return b;
      */
      DimensionWrapper sdw = shape.wrapper();
      Position min = new Point(sdw.getMin());
      min.setCoordSyst(null);
      min.setCoordSyst(this);
      Position max = new Point(sdw.getMax());
      max.setCoordSyst(null);
      max.setCoordSyst(this);
      return new Box(min, max);
    }
  }
  
  // FragmentsCollectorFiller :
  public synchronized void fillCollector(DrawablesCollector f, Renderer r, float[] mat) {
    if(shape != null && visible) {
      mat = Matrix.matrixMultiply(mat, m);
      if(leftHanded) f.invertConfiguration();
      if(specialEffect == null) shape.fillCollector(f, r, mat);
      else {
        f.   addSpecialEffect(specialEffect);
        shape.fillCollector(f, r, mat);
        f.removeSpecialEffect(specialEffect);
      }
      if(leftHanded) f.invertConfiguration();
    }
  }
  
  protected transient Shape shape;
  /**
   * Gets the shape of this volume.
   * Default is null .
   * @return the shape
   */
  public Shape getShape() { return shape; }
  /**
   * Sets the shape of this volume.
   * @param newShape the new shape
   */
  public void setShape(Shape newShape) {
    synchronized(this) {
      if((shape != null) && (changeListeners != null)) {
        synchronized(changeListeners) {
          for(Iterator i = changeListeners.iterator(); i.hasNext(); ) shape.removePropertyChangeListener((PropertyChangeListener) i.next());
        }
      }
      shape = newShape;
      if((shape != null) && (changeListeners != null)) {
        synchronized(changeListeners) {
          for(Iterator i = changeListeners.iterator(); i.hasNext(); ) shape.addPropertyChangeListener   ((PropertyChangeListener) i.next());
        }
      }
    }
    fireResize("shape");
  }
  
  /**
   * Convert this element into fragments.
   * @return the fragments
   */
  public synchronized Fragment[] toFragments() {
    if(shape == null) return new Fragment[0];
    Fragment[] fs = shape.toFragments();
    for(int i = 0; i < fs.length; i++) fs[i].preTransform(m);
    return fs;
  }

  public synchronized String getDescription() {
    if((shape == null) || (!shape.hasName()))  return super.getDescription();
    else return super.getDescription() + " : " + shape.getName();
  }
  public synchronized String propertiesString() {
    String s = super.propertiesString();
    if(shape == null) s = s + "shape : null\n";
    else {
      if(shape.hasName()) s = s + "shape : " + shape.getName();
      else s = s + "shape : (no name)";
    }
    return s;
  }

  // Events : Override : also receive shape event.
  public void addPropertyChangeListener   (PropertyChangeListener l) {
    super.addPropertyChangeListener   (l);
    if(shape != null) shape.addPropertyChangeListener   (l); // when the shape will be changed.
  }
  public void removePropertyChangeListener(PropertyChangeListener l) {
    super.removePropertyChangeListener(l);
    if(shape != null) shape.removePropertyChangeListener(l);
  }


  // RaypickingComplex :
  protected boolean solid = true;
  /**
   * Checks if this volume is solid. A non-solid volume is not used for raypicking.
   * Default is true.
   * @return true if solid
   */
  public boolean isSolid() { return solid; }
  /**
   * Sets if this volume is solid.
   * @param b true if solid
   */
  public void setSolid(boolean b) {
    synchronized(this) { solid = b; }
    firePropertyChange("solid");
  }
  public synchronized boolean ensureRaypickIsUsefull(float distanceSquarred) {
    if(shape == null) return false;
    return shape.ensureRaypickIsUsefull(distanceSquarred / Math.max(Math.max(factorX, factorY), factorZ));
  }
  public synchronized Position raypick(Position origin, Vector direction, int sens, int intersection) {
    if((shape == null) || (!solid)) return null;
    else {
      Position p = shape.raypick(origin.clone(this), (Vector) direction.clone(this), sens, intersection);
      if(p != null) p.setCoordSyst(this);
      return p;
    }
  }
  public synchronized int raypickNumber(Position origin, Vector direction, int sens, int intersection, float maxLenght) {
    if((shape == null) || (!solid)) return 0;
    else return shape.raypickNumber(origin.clone(this), (Vector) direction.clone(this), sens, intersection, maxLenght);
  }
  public synchronized boolean raypickBoolean(Position origin, Vector direction, int sens, int intersection, float maxLenght) {
    if((shape == null) || (!solid)) return false;
    else return shape.raypickBoolean(origin.clone(this), (Vector) direction.clone(this), sens, intersection, maxLenght);
  }
  public synchronized RaypickingComplex.RaypickingComplexResult raypickComplex(Position origin, Vector direction, int sens, int intersection, int maxHits) {
    if((shape == null) || (!solid)) return new RaypickingComplex.RaypickingComplexResult(origin, direction, sens, intersection, maxHits);
    RaypickingComplex.RaypickingComplexResult rcr = new RaypickingComplex.RaypickingComplexResult(origin, direction, sens, intersection, maxHits);
    Position p = raypick(origin, direction, sens, intersection);
    if(p != null) rcr.hits.add(new RaypickingComplex.Hit(this, p));
    return rcr;
  }
  public synchronized Volume3D isInterior(Position p) {
    if(shape == null) return null;
    if(p.getCoordSyst() != this) p = p.clone(this);
    if(shape.isInterior(p)) return this;
    return null;
  }

  // Serializable :
  private void writeObject(ObjectOutputStream s) throws IOException {
    s.defaultWriteObject();
    Shape.write(s, shape);
  }
  private void readObject(ObjectInputStream s) throws IOException, ClassNotFoundException {
    s.defaultReadObject();
    shape = Shape.read(s);
  }
}
