/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya3d.model.*;
import opale.soya.soya3d.animation.*;
import gl4java.*;
import java.io.*;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Arrays;
import java.lang.ClassNotFoundException;
import java.beans.*;

/**
 * A world is a volume that can contains many other 3D elements, including other worlds.
 * You can think to a world as a container. The graphical elements in a world are defined
 * into the world local coordinates system.
 * 
 * You can also receive all the events from the elements of a world (including elements in a
 * child world).
 * 
 * A 3D element can be added only in one world at a time.
 * 
 * @see opale.soya.util.Collection
 * @see opale.soya.soya3d.Element3D
 * @author Artiste on the Web
 */

public class World3D extends Volume3D implements opale.soya.util.Collection {
  private static final long serialVersionUID = -3347233111791620480l;
  
  /**
   * Creates a new world.
   */
  public World3D()                                    { super()          ; }
  /**
   * Creates a new world with the given shape.
   * @param s the shape
   */
  public World3D(Shape s)                             { super(s)         ; }
  /**
   * Creates a new world with the given name.
   * @param newName the name
   */
  public World3D(String newName)                      { super(newName)   ; }
  /**
   * Creates a new world with the given name and shape.
   * @param s the shape
   * @param newName the name
   */
  public World3D(String newName, Shape s)             { super(newName, s); }
  
  /**
   * Clones this world. All its elements are cloned. Warning : animation, collection
   * listeners and filename are not cloned.
   * @see opale.soya.soya3d.Volume3D#clone
   * @see opale.soya.soya3d.GraphicalElement3D#clone
   * @see opale.soya.soya3d.Element3D#clone
   * @return the clone
   */
  public Object clone() {
    World3D w;
    synchronized(this) {
      w = (World3D) super.clone();
      w.animations = animations;
    }
    for(Iterator i = elements(); i.hasNext(); ) w.add(((Element3D) i.next()).clone());
    return w;
  }
  
  /**
   * Saves this world in a file. File extension should be .ser .
   * @param fileName the fileName
   */
  public void save(String fileName) throws IOException {
    lastFileName = fileName;
    try {
      ObjectOutputStream o = new ObjectOutputStream(new BufferedOutputStream(new FileOutputStream(fileName)));
      synchronized(this) {
        World3D parent2 = parent;
        parent = null; // Do not save the parent.
        o.writeObject(this);
        parent = parent2;
      }
      o.flush();
      o.close();
    }
    catch(Exception e) { throw (java.io.IOException) e; }
  }
  /**
   * Saves this world in its file (the file from where it was loaded, or the last file in
   * whitch it has been saved). Do noyhing if no file name is defined (you can chek with
   * hasFileName() if it has a file name).
   */
  public void save() throws IOException {
    if(lastFileName != null) save(lastFileName);
    else throw new IOException("No filename");
  }
  private transient String lastFileName;
  /**
   * Gets this world's file name (the file from where it was loaded, or the last file in
   * whitch it has been saved), or null if no file name is defined.
   * @return the file name
   */
  public String getFileName() { return lastFileName; }
  /**
   * Cheks if this world has a file name (the file from where it was loaded, or the last file in
   * whitch it has been saved).
   * @return true if there is a the file name
   */
  public boolean hasFileName() { return lastFileName != null; }
  /**
   * Sets this world's file name.
   * @param the new file name
   */
  public void setFileName(String s) {
    synchronized(this) { lastFileName = s; }
    firePropertyChange("fileName");
  }
  /**
   * Loads a world from a file. File extension should be .ser .
   * @param fileName the fileName
   * @return the world
   */
  public static World3D load(String fileName) throws java.io.IOException, java.lang.ClassNotFoundException {
    World3D  w;
    ObjectInputStream i = new ObjectInputStream(new BufferedInputStream(new FileInputStream(fileName)));
    w = (World3D) i.readObject();
    i.close();
    w.lastFileName = fileName;
    return w;
  }
  
  private transient Animations animations;
  /**
   * Gets the animations of this worlds. It is just a collection of animation, in order to
   * easier the animation storage.
   * The animations is written with the world if it has no name, else it is supposed to be
   * written into the animation path.
   * Default is null .
   * @return the animations
   */
  public Animations getAnimations() { return animations; }
  /**
   * Sets the animations of this worlds.
   * @param as the new animations
   */
  public void setAnimations(Animations as) {
    synchronized(this) { animations = as; }
    firePropertyChange("animations");
  }
  
  protected Animation animation;
  /**
   * Gets the current animation of this world.
   * Default is null .
   * @return the animation
   */
  public Animation getAnimation() { return animation; }
  public void setAnimationFromAnimations(String name) { setAnimation(animations.animation(name)); }
  /**
   * Sets the current animation of this world. There is no obligation for the given
   * animation to be contained into this world's animations.
   * The animation is defined as current but not played; the world or its childs are not
   * move or animated. To do that, you must call setAnimationTime in this world.
   * 
   * @see opale.soya.soya3d.Animable#getAnimationTime
   * @see opale.soya.soya3d.Animable#setAnimationTime
   * @return the animation
   */
  public void setAnimation(Animation a) { 
    setAnimationTime(Float.NaN);
    synchronized(this) {
      if(animation == a) return;
      animation = a;
    }
    if(animation == null) {
      setMovement(null);
      for(Iterator i = graphicalsRecursive(); i.hasNext(); ) ((GraphicalElement3D) i.next()).setMovement(null);
    }
    else {
      if(!animation.isEmpty()) {
        setMovement(animation.getMovement(getName())); // Get only by name here.
        GraphicalElement3D e;
        for(Iterator i = graphicalsRecursive(); i.hasNext(); ) {
          e = (GraphicalElement3D) i.next();
          e.setMovement(animation.getMovement(e.getName()));
        }
      }
    }
    firePropertyChange("animation");
  }
  
  // Overrides :
  public void setMovement(Movement m) {
    super.setMovement(m);
    synchronized(this) { animationTime = Float.NaN; }
  }
  private float animationTime;
  public float getAnimationTime() { return animationTime; }
  public void setAnimationTime(float time) {
    synchronized(this) {
      if(animationTime == time) return;
      animationTime = time;
    }
    super.setAnimationTime(animationTime);
    for(Iterator i = graphicals(); i.hasNext(); ) {
      ((GraphicalElement3D) i.next()).setAnimationTime(animationTime);
    }
  }
  
  // GraphicalElement3D : override :
  protected void invalidatePreBuiltMatrixes() {
    super.invalidatePreBuiltMatrixes();
    Object[] os;
    synchronized(c) {
      os = toArray();
      /*
      for(Iterator i = c.iterator(); i.hasNext(); ) {
        Object o = i.next();
        if(o instanceof GraphicalElement3D) ((GraphicalElement3D) o).invalidatePreBuiltMatrixes();
      }
      */
    }
    for(int i = 0; i < os.length; i++) {
      if(os[i] instanceof GraphicalElement3D) ((GraphicalElement3D) os[i]).invalidatePreBuiltMatrixes();
    }
  }

  // Dimension : override :
  public float getWidth () {
    DimensionWrapper w = wrapper();
    w.setCoordSyst(parent);
    return w.getWidth ();
  }
  public float getHeight() {
    DimensionWrapper w = wrapper();
    w.setCoordSyst(parent);
    return w.getHeight();
  }
  public float getDepth () {
    DimensionWrapper w = wrapper();
    w.setCoordSyst(parent);
    return w.getDepth ();
  }
  public DimensionWrapper wrapper() {
    if(shape == null) return new Box(graphicals(), this);
    else {
      Box b = new Box(graphicals(), this);
      b.addWrapper(super.wrapper());
      return b;
    }
  }
  public void setDims(float w, float h, float d) {
    DimensionWrapper wrap = wrapper();
    wrap.setCoordSyst(parent);
    scale(w / wrap.getWidth(), h / wrap.getHeight(), d / wrap.getDepth());
  }

  // DrawablesCollectorFiller : Override :
  public void fillCollector(DrawablesCollector f, Renderer r, float[] mat) {
    synchronized(this) {
      if(!visible) return;
      if(leftHanded) f.invertConfiguration();
      mat = Matrix.matrixMultiply(mat, m);
      if(specialEffect != null) f.addSpecialEffect(specialEffect);
      Shape shape = getShape();
      if(shape != null) shape.fillCollector(f, r, mat);
    }
    
    Object[] os;
    synchronized(c) {
      os = toArray();
      /*
      for(Iterator i = iterator(); i.hasNext(); ) {
        Object o = i.next();
        if(o instanceof DrawablesCollectorFiller) ((DrawablesCollectorFiller) o).fillCollector(f, r, mat);
      }
      */
    }
    
    for(int i = 0; i < os.length; i++) {
      if(os[i] instanceof DrawablesCollectorFiller) ((DrawablesCollectorFiller) os[i]).fillCollector(f, r, mat);
    }
    
    if(leftHanded) f.invertConfiguration();
    if(specialEffect != null) f.removeSpecialEffect(specialEffect);
  }

  // Collection :
  private final java.util.Collection c = new java.util.Vector();
  public boolean add(Element3D e) {
    boolean b;
    synchronized(c) {
      b = c.add(e);
      e.added(this);
      addition(e);
    }
    if(b) fireAdd(e);
    return b;
  }
  public boolean remove(Element3D e) {
    boolean b;
    synchronized(c) {
      b = c.remove(e);
      e.removed(this);
      removing(e);
    }
    if(b) fireRemove(e);
    return b;
  }
  public boolean add(Object e) { return add((Element3D) e); }
  public boolean addAll(java.util.Collection col) {
    boolean b;
    synchronized(c) {
      b = c.addAll(col);
      if(!b) return false;
      for(Iterator i = col.iterator(); i.hasNext(); ) {
        Element3D e = (Element3D) i.next();
        e.added(this);
        addition(e);
      }
    }
    for(Iterator i = col.iterator(); i.hasNext(); ) fireAdd((Element3D) i.next());
    return b;
  }
  public void clear() {
    java.util.Collection s = new java.util.Vector(c.size());
    synchronized(c) {
      s.addAll(c);
      c.clear();
      for(Iterator i = s.iterator(); i.hasNext(); ) {
        Element3D e = (Element3D) i.next();
        e.removed(this);
        removing(e);
      }
    }
    for(Iterator i = s.iterator(); i.hasNext(); ) fireRemove((Element3D) i.next());
  }
  public boolean contains(Object o) { return c.contains(o); }
  public boolean containsAll(java.util.Collection col) { return c.containsAll(col); }
  
  public boolean isEmpty()                       { return c.isEmpty(); }
  public Iterator iterator()                     { return c.iterator(); }
  public Iterator iteratorRecursive()            { return new Collection.RecursiveIterator(this); }
  public Iterator iterator(Class clazz)          { return new Collection.ClassSpecificIterator(this, clazz); }
  public Iterator iteratorRecursive(Class clazz) { return new Collection.RecursiveIterator(iterator(clazz)); }
  public Iterator elements()                     { return c.iterator(); }
  public Iterator elementsRecursive()            { return new Collection.RecursiveIterator(this); }
  
  /**
   * Gets the first element in this world with the given name.
   * @param name the name
   * @return the element
   */
  public Element3D element(String name) {
    Element3D e;
    synchronized(c) {
      for(Iterator i = elements(); i.hasNext(); ) {
        e = (Element3D) i.next();
        if(name.equals(e.getName())) return e;
      }
    }
    return null;
  }
  /**
   * Gets the first element in this world or its child-worlds with the given name.
   * @param name the name
   * @return the element
   */
  public Element3D elementRecursive(String name) {
    Element3D e;
    synchronized(c) {
      for(Iterator i = elementsRecursive(); i.hasNext(); ) {
        e = (Element3D) i.next();
        if(name.equals(e.getName())) return e;
      }
    }
    return null;
  }
  /**
   * Gets all the elements in this world with the given name.
   * @param name the name
   * @return an iterator that will iterates through the elements
   */
  public Iterator elements(String name) { return new ElementNameIterator(elements(), name); }
  /**
   * Gets all the elements in this world or its child-worlds with the given name.
   * @param name the name
   * @return an iterator that will iterates through the elements
   */
  public Iterator elementsRecursive(String name) { return new ElementNameIterator(elementsRecursive(), name); }
  private class ElementNameIterator implements Iterator {
    private ElementNameIterator(Iterator it, String n) {
      i = it;
      name = n;
    }
    private Iterator i;
    private String name;
    private Object myNext;

    public boolean hasNext() {
      while(true) {
        if(!i.hasNext()) return false;
        myNext = i.next();
        if(((Element3D) myNext).getName().equals(name)) return true;
      }
    }
    public Object next() { return myNext; }
    public void remove() { i.remove(); }
  }

  /**
   * Gets all the elements in this world with the given name.
   * @param name the name, can have generic regexp like "*"
   * @return an iterator that will iterates through the elements
   */
  public Iterator search(String name) { return new SearchNameIterator(elements(), name); }
  /**
   * Gets all the elements in this world or its child-worlds with the given name.
   * @param name the name, can have generic regexp like "*"
   * @return an iterator that will iterates through the elements
   */
  public Iterator searchRecursive(String name) { return new SearchNameIterator(elementsRecursive(), name); }
  private class SearchNameIterator implements Iterator {
    private SearchNameIterator(Iterator it, String n) {
      i = it;
      name = n;
    }
    private Iterator i;
    private String name;
    private Object myNext;

    public boolean hasNext() {
      String s;
      while(true) {
        if(!i.hasNext()) return false;
        myNext = i.next();
        s = ((Element3D) myNext).getName();
        if(s != null) {
          if(Soya.match(s, name)) return true;
        }
      }
    }
    public Object next() { return myNext; }
    public void remove() { i.remove(); }
  }

  public Iterator graphicals() {
    return new Collection.ClassSpecificIterator(this, opale.soya.soya3d.GraphicalElement3D.class);
  }
  public Iterator graphicalsRecursive() {
    return new Collection.ClassSpecificIterator(new Collection.RecursiveIterator(this), opale.soya.soya3d.GraphicalElement3D.class);
  }
  public Iterator volumes() {
    return new Collection.ClassSpecificIterator(this, opale.soya.soya3d.Volume3D.class);
  }
  public Iterator volumesRecursive() {
    return new Collection.ClassSpecificIterator(new Collection.RecursiveIterator(this), opale.soya.soya3d.Volume3D.class);
  }
  public Iterator worlds() {
    return new Collection.ClassSpecificIterator(this, opale.soya.soya3d.World3D.class);
  }
  public Iterator worldsRecursive() {
    return new Collection.ClassSpecificIterator(new Collection.RecursiveIterator(this), opale.soya.soya3d.World3D.class);
  }
  public Iterator environments() {
    return new Collection.ClassSpecificIterator(this, opale.soya.soya3d.Environment3D.class);
  }
  public Iterator environmentsRecursive() {
    return new Collection.ClassSpecificIterator(new Collection.RecursiveIterator(this), opale.soya.soya3d.Environment3D.class);
  }
  public Iterator lights() {
    return new Collection.ClassSpecificIterator(this, opale.soya.soya3d.Light3D.class);
  }
  public Iterator lightsRecursive() {
    return new Collection.ClassSpecificIterator(new Collection.RecursiveIterator(this), opale.soya.soya3d.Light3D.class);
  }
  public Iterator cameras() {
    return new Collection.ClassSpecificIterator(this, opale.soya.soya3d.Camera3D.class);
  }
  public Iterator camerasRecursive() {
    return new Collection.ClassSpecificIterator(new Collection.RecursiveIterator(this), opale.soya.soya3d.Camera3D.class);
  }
  public Iterator raypickingComplex() {
    return new Collection.ClassSpecificIterator(this, opale.soya.soya3d.RaypickingComplex.class);
  }
  public Iterator raypickingComplexRecursive() {
    return new Collection.ClassSpecificIterator(new Collection.RecursiveIterator(this), opale.soya.soya3d.RaypickingComplex.class);
  }
  public Iterator raypickings() {
    return new Collection.ClassSpecificIterator(this, opale.soya.soya3d.Raypicking.class);
  }
  public Iterator raypickingsRecursive() {
    return new Collection.ClassSpecificIterator(new Collection.RecursiveIterator(this), opale.soya.soya3d.Raypicking.class);
  }
  private Iterator drawablesCollectorFillers() {
    return new Collection.ClassSpecificIterator(this, opale.soya.soya3d.DrawablesCollectorFiller.class);
  }
  private Iterator drawablesCollectorFillersRecursive() {
    return new Collection.ClassSpecificIterator(new Collection.RecursiveIterator(this), opale.soya.soya3d.DrawablesCollectorFiller.class);
  }
  private Iterator fragmentables() {
    return new Collection.ClassSpecificIterator(this, opale.soya.soya3d.model.Fragmentable.class);
  }
  private Iterator fragmentablesRecursive() {
    return new Collection.ClassSpecificIterator(new Collection.RecursiveIterator(this), opale.soya.soya3d.model.Fragmentable.class);
  }
  private Iterator collections() {
    return new Collection.ClassSpecificIterator(this, opale.soya.util.Collection.class);
  }
  private Iterator collectionsRecursive() {
    return new Collection.ClassSpecificIterator(new Collection.RecursiveIterator(this), opale.soya.util.Collection.class);
  }
  public boolean remove(Object e) { return remove((Element3D) e); }
  public boolean removeAll(java.util.Collection col) {
    boolean b;
    synchronized(this) {
      b = c.removeAll(col);
      if(!b) return false;
      for(Iterator i = col.iterator(); i.hasNext(); ) {
        Element3D e = (Element3D) i.next();
        e.removed(this);
        removing(e);
      }
    }
    for(Iterator i = col.iterator(); i.hasNext(); ) fireRemove((Element3D) i.next());
    return b;
  }
  public boolean retainAll(java.util.Collection col) {
    java.util.Collection s = new java.util.Vector(col.size());
    boolean b;
    synchronized(this) {
      for(Iterator i = col.iterator(); i.hasNext(); ) {
        Object o = i.next();
        if(c.contains(o)) s.add(o);
      }
      b = c.retainAll(col);
      if(!b) return false;
      for(Iterator i = s.iterator(); i.hasNext(); ) {
        Element3D e = (Element3D) i.next();
        e.removed(this);
        removing(e);
      }
    }
    for(Iterator i = s.iterator(); i.hasNext(); ) fireRemove((Element3D) i.next());
    return b;
  }
  public int size() { return c.size(); }
  public Object[] toArray() { return c.toArray(); }
  public Object[] toArray(Object[] a) { return c.toArray(a); }

  // Collection events :
  private transient java.util.Collection collectionListeners;
  public void addCollectionListener(CollectionListener l) {
    if(collectionListeners == null) collectionListeners = new java.util.Vector();
    collectionListeners.add(l);
  }
  public void removeCollectionListener(CollectionListener l) {
    if(collectionListeners != null) collectionListeners.remove(l);
  }
  public java.util.Collection collectionListeners() { return collectionListeners; }
  public void fireAdd(Object element) {
    if(collectionListeners != null) {
      int size;
      Object[] ls;
      synchronized (collectionListeners) {
        size = collectionListeners.size();
        ls = collectionListeners.toArray();
      }
      AddEvent e = new AddEvent(this, element);
      for (int i = 0; i < size; i++) ((CollectionListener) ls[i]).added(e);
    }
  }
  public void fireRemove(Object element) {
    if(collectionListeners != null) {
      int size;
      Object[] ls;
      synchronized (collectionListeners) {
        size = collectionListeners.size();
        ls = collectionListeners.toArray();
      }
      RemoveEvent e = new RemoveEvent(this, element);
      for (int i = 0; i < size; i++) ((CollectionListener) ls[i]).removed(e);
    }
  }
  
  // Sub events :
  private transient java.util.Collection subPropertyChangeListeners;
  private transient java.util.Collection subCollectionListeners;
  
  private class SubCollectionListener implements CollectionListener, Serializable {
    private static final long serialVersionUID = 7877234852443438454l;
    public void added(AddEvent e) {
      Element3D el = (Element3D) e.getElement();
      if(el instanceof World3D) addition((World3D) el);
      else addition(el);
    }
    public void removed(RemoveEvent e) {
      Element3D el = (Element3D) e.getElement();
      if(el instanceof World3D) removing((World3D) el);
      else removing(el);
    }
  }
  private CollectionListener subCollectionListener = new SubCollectionListener();
  
  public void addition(Element3D e) {
    registerSubListeners(e);
    if(e instanceof World3D) {
      World3D w = (World3D) e;
      
      w.addCollectionListener(subCollectionListener);
      Element3D e2;
      synchronized(w.c) {
        for(Iterator i = w.elements(); i.hasNext(); ) {
          e2 = (Element3D) i.next();
          if(e2 instanceof World3D) addition((World3D) e2);
          else addition(e2);
        }
      }
    }
  }
  public void addition(World3D w) {
    addition((Element3D) w);
  }
  private void registerSubListeners(Element3D e) {
    if(subPropertyChangeListeners != null) {
      for(Iterator i = subPropertyChangeListeners.iterator(); i.hasNext(); ) e.addPropertyChangeListener((PropertyChangeListener) i.next());
    }
    if((subCollectionListeners != null) && (e instanceof Collection)) {
      Collection c = (Collection) e;
      for(Iterator i = subCollectionListeners.iterator(); i.hasNext(); ) c.addCollectionListener((CollectionListener) i.next());
    }
  }
  public void removing(Element3D e) {
    unregisterSubListeners(e);
  }
  public void removing(World3D w) {
    removing((Element3D) w);
    w.removeCollectionListener(subCollectionListener);
    Element3D e;
    synchronized(w.c) {
      for(Iterator i = w.elements(); i.hasNext(); ) {
        e = (Element3D) i.next();
        if(e instanceof World3D) removing((World3D) e);
        else removing(e);
      }
    }
  }
  private void unregisterSubListeners(Element3D e) {
    if(subPropertyChangeListeners != null) {
      for(Iterator i = subPropertyChangeListeners.iterator(); i.hasNext(); ) e.removePropertyChangeListener((PropertyChangeListener) i.next());
    }
    if((subCollectionListeners != null) && (e instanceof Collection)) {
      Collection c = (Collection) e;
      for(Iterator i = subCollectionListeners.iterator(); i.hasNext(); ) c.removeCollectionListener((CollectionListener) i.next());
    }
  }

  public void addSubPropertyChangeListener(PropertyChangeListener l) {
    if(subPropertyChangeListeners == null) subPropertyChangeListeners = new java.util.Vector();
    subPropertyChangeListeners.add(l);
    for(Iterator i = elementsRecursive(); i.hasNext(); ) ((Element3D) i.next()).addPropertyChangeListener(l);
  }
  public void removeSubPropertyChangeListener(PropertyChangeListener l) {
    if(subPropertyChangeListeners != null) {
      subPropertyChangeListeners.remove(l);
      for(Iterator i = elementsRecursive(); i.hasNext(); ) ((Element3D) i.next()).removePropertyChangeListener(l);
    }
  }
  public void addSubCollectionListener(CollectionListener l) {
    if(subCollectionListeners == null) subCollectionListeners = new java.util.Vector();
    subCollectionListeners.add(l);
    for(Iterator i = collectionsRecursive(); i.hasNext(); ) ((Collection) i.next()).addCollectionListener(l);
  }
  public void removeSubCollectionListener(CollectionListener l) {
    if(subCollectionListeners != null) {
      subCollectionListeners.remove(l);
      for(Iterator i = collectionsRecursive(); i.hasNext(); ) ((Collection) i.next()).addCollectionListener(l);
    }
  }

  // Element :
  public synchronized String getDescription() { return super.getDescription() + ", " + Integer.toString(c.size()) + " elements\n"; }
  public String propertiesString() {
    String s;
    synchronized(this) { s = super.propertiesString() + Integer.toString(c.size()) + " elements {\n"; }
    for(Iterator i = elements(); i.hasNext(); ) s = s + i.next().toString();
    s = s + "\n}";
    return s;
  }

  // Serializable :
  private void writeObject(ObjectOutputStream s) throws IOException {
    s.defaultWriteObject();
    Animations.write(s, animations);
    if(animation == null) s.writeInt(StoredInPathObject.NO_OBJECT_SAVED);
    else {
      if((animations != null) && (animation != null) && (animation.hasName()) && (animations.contains(animation))) {
        s.writeInt(StoredInPathObject.OBJECT_SAVED_IN_PATH); // Not in path here, but in animations, whitch may be saved in path or not...
        s.writeUTF(animation.getName());
      }
      else {
        s.writeInt(StoredInPathObject.OBJECT_SAVED_HERE);
        s.writeObject(animation);
      }
    }
  }
  private void readObject(ObjectInputStream s) throws java.io.IOException, ClassNotFoundException { // Initialization.
    s.defaultReadObject();
    animations = Animations.read(s);
    switch(s.readInt()) {
    case StoredInPathObject.OBJECT_SAVED_IN_PATH:
      animation = animations.animation(s.readUTF());
      break;
    case StoredInPathObject.OBJECT_SAVED_HERE:
      animation = (Animation) s.readObject();
      break;
    case StoredInPathObject.NO_OBJECT_SAVED:
    }
      
    for(Iterator i = worldsRecursive(); i.hasNext(); ) ((World3D) i.next()).addCollectionListener(subCollectionListener);
  }
  
  // Fragmentable : Override :
  public synchronized Fragment[] toFragments() {
    Fragment[] fs = super.toFragments(), fs2, oldFs = null;
    for(Iterator i = fragmentables(); i.hasNext(); ) {
      Fragmentable frag = (Fragmentable) i.next();
      fs2 = frag.toFragments();
      if(fs2.length > 0) {
        for(int j = 0; j < fs2.length; j++) fs2[j].preTransform(m);
        if(fs != null) {
          oldFs = fs;
          fs = new Fragment[oldFs.length + fs2.length];
          System.arraycopy(oldFs, 0, fs, 0           , oldFs.length);
          System.arraycopy(fs2  , 0, fs, oldFs.length,   fs2.length);
        }
        else fs = fs2;
      }
    }
    return fs;
  }
  /**
   * Converts this world into a shape. All the static lights that are inside the world are
   * applied as static lighting.
   * @return a shape
   */
  public synchronized Shape toShape() {
    java.util.Collection dynamicLights = new java.util.Vector();
    World3D root = getRootParent();
    if(root == null) root = this;
    for(Iterator i = root.lightsRecursive(); i.hasNext(); ) {
      Light3D l = (Light3D) i.next();
      if(l.isVisible() && l.isStatic()) dynamicLights.add(l);
    }
    FragmentedShape s = new FragmentedShape();
    s.lock();
    Fragment[] fs = toFragments();
    for(int i = 0; i < fs.length; i++) {
      if(fs[i] instanceof FaceVisibility) {
        FaceVisibility fv = (FaceVisibility) fs[i];
        switch(fv.getVisibility()) {
        case FaceVisibility.VISIBILITY_INTERIOR:
        case FaceVisibility.VISIBILITY_EXTERIOR:
          fv.setVisibility(FaceVisibility.VISIBILITY_NO_INTERIOR);
        }
      }
    }
    s.addAllFragments(fs);
    
    if(dynamicLights.size() > 0) {
      Light3D[] lights = (Light3D[]) dynamicLights.toArray(new Light3D[0]);
      if(parent == null) s.applyStaticLighting(lights, this  );
      else               s.applyStaticLighting(lights, parent);
    }
    s.unlock();
    return s;
  }

  public boolean ensureRaypickIsUsefull(float distanceSquarred) {
    Element3D e;
    float distanceSquarred2;
    synchronized(this) {
      distanceSquarred = distanceSquarred / Math.max(Math.max(factorX, factorY), factorZ);
      if(super.ensureRaypickIsUsefull(distanceSquarred)) return true;
    }
    for(Iterator i = elements(); i.hasNext(); ) {
      e = (Element3D) i.next();
      if(e instanceof Raypicking) {
        if(e instanceof GraphicalElement3D) distanceSquarred2 = distanceSquarred + squareDistanceTo((GraphicalElement3D) e);
        else distanceSquarred2 = distanceSquarred;
        if(((Raypicking) e).ensureRaypickIsUsefull(distanceSquarred2)) return true;
      }
    }
    return false;
  }
  
  // RaypickingComplex : Override :
  public Position raypick(Position origin, Vector direction, int sens, int intersection) {
    Position current, best = null;
    float currentDist, bestDist = Float.POSITIVE_INFINITY;
    synchronized(this) {
      if(!isSolid()) return null;
      if(shape != null) {
        best = super.raypick(origin, direction, sens, intersection); // shape.raypick(origin.clone(this), (Vector) direction.clone(this), sens, intersection);
        if(best != null) bestDist = origin.squareDistanceTo(best);
      }
    }
    for(Iterator i = raypickings(); i.hasNext(); ) {
      current = ((Raypicking) i.next()).raypick(origin, direction, sens, intersection);
      if(current != null) {
        currentDist = origin.squareDistanceTo(current);
        if(currentDist < bestDist) {
          best = current;
          bestDist = currentDist;
        }
      }
    }
    if(best != null) best.setCoordSyst(this);
    return best;
  }
  public int raypickNumber(Position origin, Vector direction, int sens, int intersection, float maxLenght) {
    int nb = 0;
    synchronized(this) {
      if(!isSolid()) return 0;
      if(shape != null) {
        nb = shape.raypickNumber(origin.clone(this), (Vector) direction.clone(this), sens, intersection, maxLenght);
      }
    }
    for(Iterator i = raypickings(); i.hasNext(); ) {
      nb = nb + ((Raypicking) i.next()).raypickNumber(origin, direction, sens, intersection, maxLenght);
    }
    return nb;
  }
  public boolean raypickBoolean(Position origin, Vector direction, int sens, int intersection, float maxLenght) {
    synchronized(this) {
      if(!isSolid()) return false;
      if(shape != null) {
        if(shape.raypickBoolean(origin.clone(this), (Vector) direction.clone(this), sens, intersection, maxLenght)) return true;
      }
    }
    for(Iterator i = raypickings(); i.hasNext(); ) {
      if(((Raypicking) i.next()).raypickBoolean(origin, direction, sens, intersection, maxLenght))
      return true;
    }
    return false;
  }
  public RaypickingComplex.RaypickingComplexResult raypickComplex(Position origin, Vector direction, int sens, int intersection, int maxHits) {
    RaypickingComplex.RaypickingComplexResult r, rcr = new RaypickingComplex.RaypickingComplexResult(origin, direction, sens, intersection, maxHits);
    synchronized(this) {
      if(!isSolid()) return rcr; // empty.
      if(shape != null) {
        Position p = shape.raypick(origin.clone(this), (Vector) direction.clone(this), sens, intersection);
        if(p != null) rcr.hits.add(new RaypickingComplex.Hit(this, p));
      }
    }
    for(Iterator i = raypickingComplex(); i.hasNext(); ) {
      if(rcr.hits.size() >= maxHits) break;

      r = ((RaypickingComplex) i.next()).raypickComplex(origin, direction, sens, intersection, maxHits);
      rcr.hits.addAll(r.hits);
    }
    return rcr;
  }
  public Volume3D isInterior(Position p) {
    Volume3D v;
    v = super.isInterior(p);
    if(v != null) return v;
    synchronized(c) {
      for(Iterator i = elements(); i.hasNext(); ) {
        Object o = i.next();
        if(o instanceof Volume3D) {
          v = ((Volume3D) o).isInterior(p);
          if(v != null) return v;
        }
      }
    }
    return null;
  }
}
