/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USAint
 */

package opale.soya.soya3d;

import opale.soya.*;
import opale.soya.util.Lockable;
import opale.soya.util.*;
import opale.soya.awt.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.Point;
import opale.soya.soya3d.event.*;
import opale.soya.soya3d.model.*;
import opale.soya.soya3d.animation.*;
import opale.soya.editor.*;
import java.beans.*;
import java.lang.reflect.*;
import java.awt.*;
import java.awt.event.*;
import java.util.Iterator;
import java.util.Enumeration;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.tree.*;

public class WorldGUIEditor extends GUIEditor { // implements TreeExpansionListener, TreeSelectionListener {
  public static void edit(World3D w, JTree tree, DefaultTreeModel treeModel) {
    WorldGUIEditor wge = new WorldGUIEditor(w);
    wge.setTree(tree, treeModel);
    wge.setVisible(true);
  }
  
  protected Menu selMenu;
  
  protected void aff(String s) { super.aff(s); }
  protected void setStep(float step) {
    super.setStep(step);
    handles.setScale(step);
  }
  public void render() { super.render(); }

  private World3D w;
  public World3D getWorld() { return w; }
  public WorldGUIEditor(World3D wo) {
    super(wo.getName() + " -- soya world GUI editor");
    
    w = wo;
    
    // Build menu:
    MenuItem file1 = new MenuItem("Save"); fileMenu.add(file1);
    file1.setShortcut(new MenuShortcut(KeyEvent.VK_S));
    file1.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        try {
          if(Editor.isSaveable(w)) {
            Editor.save(w, WorldGUIEditor.this);
            aff("Saved.");
          }
        }
        catch(Exception e2) { aff("Can't save world : " + w.getName() + " : " + e2); }
      }
    });
    MenuItem file2 = new MenuItem("Save as..."); fileMenu.add(file2);
    file2.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        try {
          if(Editor.isSaveable(w)) {
            Editor.saveAs(w, WorldGUIEditor.this);
            aff("Saved.");
          }
        }
        catch(Exception e2) { aff("Can't save world : " + w.getName() + " : " + e2); }
      }
    });
    
    MenuItem edit1 = new MenuItem("incline step..."); editMenu.add(edit1);
    edit1.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        float f = Float.parseFloat(TextChooser.question(WorldGUIEditor.this, "New incline step", Float.toString(Editor.getOptions().inclineStep)));
        if(f > 0f) Editor.getOptions().inclineStep = f;
      }
    });
    CheckboxMenuItem edit2 = new CheckboxMenuItem("Show ground", Editor.getOptions().showGround); editMenu.add(edit2);
    edit2.addItemListener(new ItemListener() {
      public void itemStateChanged(ItemEvent e) {
        Editor.getOptions().showGround = !Editor.getOptions().showGround;
        setShowGround(Editor.getOptions().showGround);
        render();
      }
    });
    
    Menu worldMenu = new Menu("World"); menuBar.add(worldMenu);
    MenuItem world1 = new MenuItem("edit..."); worldMenu.add(world1);
    world1.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        try { Editor.edit(w); }
        catch(Exception ex) { aff("Can't edit world!"); ex.printStackTrace(); }
      }
    });
    
    selMenu = new Menu("Selection"); menuBar.add(selMenu);
    MenuItem sel5 = new MenuItem("Edit selection..."); selMenu.add(sel5);
    sel5.setShortcut(new MenuShortcut(KeyEvent.VK_E));
    sel5.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) { editSelection(); }
    });
    MenuItem sel6 = new MenuItem("Delete selection..."); selMenu.add(sel6);
    sel6.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) { deleteSelection(); }
    });
    selMenu.add(new MenuItem("-"));
    MenuItem sel1 = new MenuItem("Mode selection"); selMenu.add(sel1);
    sel1.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) { setMode(MODE_SELECTION); }
    });
    MenuItem sel4 = new MenuItem("Hide handles"); selMenu.add(sel4);
    sel4.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) { setMode(MODE_NO_HANDLES); }
    });
    
    Menu asMenu = new Menu("Animations"); menuBar.add(asMenu);
    MenuItem as1 = new MenuItem("Edit..."); asMenu.add(as1);
    as1.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        Animations as = w.getAnimations();
        if(as != null) {
          try { Editor.edit(as, false, new AnimationsCustomizer(as, w)); }
          catch(Exception ex) { System.out.println("Can't edit animations!"); ex.printStackTrace(); }
        }
      }
    });
    
    // Build scene:
    world.add(w);
		
    // Build handles:
    handles = handlesFor(w);
    handles.addIntoWorld(world);
    for(Iterator i = handles.iterator(); i.hasNext(); ) ((Handle3D) i.next()).addPropertyChangeListener(listener);
    handles.addCollectionListener(new CollectionListener() { // Will automatically add new handles.
      public void added(AddEvent    e) {
        Handle3D h = (Handle3D) e.getElement();
        WorldGUIEditor.this.getEditionWorld().add   (h);
        h.addPropertyChangeListener   (listener);
      }
      public void removed(RemoveEvent e) {
        Handle3D h = (Handle3D) e.getElement();
        WorldGUIEditor.this.getEditionWorld().remove(h);
        h.removePropertyChangeListener(listener);
      }
    });
    
    w.addPropertyChangeListener   (w_propertyChangeListener);
    w.addSubPropertyChangeListener(w_propertyChangeListener);
    w.addCollectionListener       (w_collectionListener    );
    w.addSubCollectionListener    (w_collectionListener    );
    
    cursor.setHandles(handles);
    setMode(MODE_SELECTION);
    setStep(Editor.getOptions().magneticStep);
    aff("Welcome to soya GUI editor :-) Use mouse and page-up/down keys.");
  }
  public void dispose() {
    super.dispose();
    w.removePropertyChangeListener(w_propertyChangeListener);
    w.removeCollectionListener    (w_collectionListener    );
    w.remove();
  }
  PropertyChangeListener w_propertyChangeListener = new PropertyChangeListener() {
    public void propertyChange(PropertyChangeEvent e) {
      if((e instanceof MoveEvent) || (e instanceof OrientateEvent) || (e instanceof ResizeEvent)) {
        Object o = e.getSource();
        if(o instanceof World3D) {
          World3D w = (World3D) o;
          for(Iterator i = handles.iterator(); i.hasNext(); ) {
            Handle3D h = (Handle3D) i.next();
            if(((Element3D) h.getEditedObject()).isInside(w)) h.update();
          }
        }
      }
      render();
    }
  };
  CollectionListener w_collectionListener = new CollectionListener() {
    public void added(AddEvent e) {
      Element3D el = (Element3D) e.getElement();
      addElement(el);
    }
    public void removed(RemoveEvent e) { handles.removeAll(handlesFor((Element3D) e.getElement())); }
  };
  
  private Cursor3D getCursor3D() { return cursor; }
  private RenderingCanvas getRenderingCanvas() { return renderingCanvas; }
  
  protected int mode;
  public static final int MODE_SELECTION = 0;
  public static final int MODE_ADDITION = 1;
  public static final int MODE_NO_HANDLES = 2;
  public int getMode() { return mode; }
  public void setMode(int newMode) {
    switch(newMode) {
    case MODE_SELECTION:
      mode = newMode; aff("Selection mode : Select, move and rotate one or more elements.");
      handles.setVisible(true);
      break;
    case MODE_ADDITION:
      mode = newMode; aff("Addition mode : position the new element.");
      break;
    case MODE_NO_HANDLES:
      mode = newMode; aff("No handle mode : Useful for screenshot :-)");
      handles.setVisible(false);
      break;
    }
    render();
  }
  
  public void keyPressed(KeyEvent e) {
    lock();
    switch(e.getKeyCode()) {
    case KeyEvent.VK_BACK_SPACE: cancelAddition      (); break;
    case KeyEvent.VK_DELETE:     deleteSelection     (); break;
    case KeyEvent.VK_E:          editSelection       (); break;
    case KeyEvent.VK_END:
    case KeyEvent.VK_INSERT:     rotateIncline( Editor.getOptions().inclineStep); break;
    case KeyEvent.VK_HOME:       rotateIncline(-Editor.getOptions().inclineStep); break;
    case KeyEvent.VK_S:
      try {
        w.save();
        aff("Saved.");
      }
      catch(Exception e2) { aff("Can't save world : " + w.getName() + " : " + e2); e2.printStackTrace(); }
      break;
    default: super.keyPressed(e); break;
    }
    unlock();
  }
  private void deleteSelection() {
    for(Iterator i = cursor.getHandles().getClickedObjects().iterator(); i.hasNext(); ) {
      Object o = i.next();
      if(o instanceof Element3D) ((Element3D) o).remove();
    }
  }
  private void editSelection() {
    for(Iterator i = cursor.getHandles().getClickedObjects().iterator(); i.hasNext(); ) {
      try { Editor.edit(i.next(), false); }
      catch(Exception e) { aff("can't edit an element in the selection!"); e.printStackTrace(); }
    }
  }
  private void rotateIncline(float angle) {
    for(Iterator i = cursor.getHandles().getClickedObjects().iterator(); i.hasNext(); ) {
      Object o = i.next();
      if(o instanceof Orientation) ((Orientation) o).rotateIncline(angle);
    }
  }
  
  private Volume3D ground;
  public void setShowGround(boolean b) {
    System.out.println("showGround " + b);
    if(b) {
      if(ground == null) {
        try { ground = new Volume3D(opale.soya.soya3d.model.Shape.get("ground")); }
        catch(Exception e) { System.out.println("Can't find ground shape!"); e.printStackTrace(); return; }
        ground.rotateVertical(90f);
      }
      world.add(ground);
    }
    else {
      if(ground != null) ground.remove();
    }
  }
  
  protected Handles handles;
  protected Element3D lastAddedElement;
  protected void addElement(Element3D e) {
    if(mode == MODE_ADDITION) cancelAddition();
    setMode(MODE_ADDITION);
    lastAddedElement = e;
    
    Handles eHandles = handlesFor(e);
    handles.addAll(eHandles);
    if(e instanceof GraphicalElement3D) {
      for(Iterator i = eHandles.iterator(); i.hasNext(); ) {
        Handle3D h = (Handle3D) i.next();
        if(h instanceof AtomicHandle3D) {
          getCursor3D().press(new MouseEvent(getRenderingCanvas(), MouseEvent.MOUSE_PRESSED, System.currentTimeMillis(), MouseEvent.BUTTON1_MASK, 0, 0, 0, false), h);
          break;
        }
      }
    }
  }
  protected void cancelAddition() {
    if(mode == MODE_ADDITION) {
      setMode(MODE_SELECTION);
      aff("Addition canceled !");
    }
    else {
      if(lastAddedElement != null) {
        w.remove(lastAddedElement);
        lastAddedElement = null;
        aff("Last added element removed !");
      }
      else aff("Can't remove last added element ! No element or the last added one have already been removed !");
    }
  }
  
  protected Handles handlesFor(Element3D e) {
    Handles hs = new Handles();
    if(e instanceof GraphicalElement3D) {
      GraphicalElement3D ge = (GraphicalElement3D) e;
      hs.add(new AtomicHandle3D     (ge));
      hs.add(new DirectionalHandle3D(ge));
    }
    if(e instanceof World3D) {
      for(Iterator i = ((World3D) e).iterator(); i.hasNext(); ) hs.addAll(handlesFor((Element3D) i.next()));
    }
    return hs;
  }
  
  private class PropertyChangeAndTreeListener implements PropertyChangeListener, TreeExpansionListener, TreeSelectionListener {
    private boolean muted; // Avoid events sending events...
    public synchronized void propertyChange(PropertyChangeEvent e) {
      if(!muted) {
        muted = true;
        if("clicked".equals(e.getPropertyName())) {
          Handle3D h = (Handle3D) e.getSource();
          setSelectedForTreeNodeOfHandle(h, h.isClicked());
        }
        muted = false;
      }
    }
    public synchronized void valueChanged(TreeSelectionEvent e) {
      if(!muted) {
        muted = true;
        TreePath[] paths = e.getPaths();
        int nb = paths.length;
        for(int i = 0; i < nb; i++)
          setClickedForHandlesOfPath(paths[i], e.isAddedPath(paths[i]));
        muted = false;
        render();
      }
    }
    public synchronized void treeExpanded(TreeExpansionEvent e) {
      Element3D el = ((ElementTreeNode) e.getPath().getLastPathComponent()).getElement(); // Normally, only world can
      if(el instanceof World3D) setHandlesVisibilityForElementOf((World3D) el, true );   // be expanded/collapsed.
      render();
    }
    public synchronized void treeCollapsed(TreeExpansionEvent e) {
      Element3D el = ((ElementTreeNode) e.getPath().getLastPathComponent()).getElement();
      if(el instanceof World3D) setHandlesVisibilityForElementOf((World3D) el, false);
      render();
    }
  }
  private void setClickedForHandlesOfPath(TreePath p, boolean b) {
    for(Iterator i = handles.getHandlesFor(((ElementTreeNode) p.getLastPathComponent()).getElement()).iterator(); i.hasNext(); ) {
      if(b) ((Handle3D) i.next()).setClicked (true );
      else  ((Handle3D) i.next()).setSelected(false);
    }
  }
  private void setSelectedForTreeNodeOfHandle(Handle3D h, boolean b) {
    if(b) tree.addSelectionPath   (new TreePath(getTreeNodeForHandle(h).getPath()));
    else  tree.removeSelectionPath(new TreePath(getTreeNodeForHandle(h).getPath()));
  }
  private ElementTreeNode getTreeNodeForHandle(Handle3D h) {
    ElementTreeNode n = (ElementTreeNode) treeModel.getRoot();
    Element3D searchWhat = (Element3D) h.getEditedObject();
    if(n.getElement() == searchWhat) return n;
    return getTreeNodeForHandleSearch(n, searchWhat);
  }
  private ElementTreeNode getTreeNodeForHandleSearch(ElementTreeNode n, Element3D searchWhat) {
    for(Enumeration i = n.children(); i.hasMoreElements(); ) {
      ElementTreeNode n2 = (ElementTreeNode) i.nextElement();
      if(n2.getElement() == searchWhat) return n2;
      if(!n2.isLeaf()) {
        n2 = getTreeNodeForHandleSearch(n2, searchWhat);
        if(n2 != null) return n2;
      }
    }
    return null;
  }
  private void setHandlesVisibilityForElementOf(World3D w, boolean b) {
    synchronized(handles) {
      for(Iterator i = handles.iterator(); i.hasNext(); ) {
        Handle3D h = (Handle3D) i.next();
        if(((Element3D) h.getEditedObject()).isInside(w)) h.setVisible(b);
      }
    }
  }
  private PropertyChangeAndTreeListener listener = new PropertyChangeAndTreeListener();
  private JTree tree;
  private DefaultTreeModel treeModel;
  public void setTree(JTree tree, DefaultTreeModel treeModel) {
    this.tree = tree;
    this.treeModel = treeModel;
    tree.addTreeSelectionListener(listener);
    tree.addTreeExpansionListener(listener);
  }
}
