/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d.animation;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import java.io.*;
import java.beans.*;
import java.util.Iterator;

public class Movement extends Object implements Serializable, Cloneable {
  private static final long serialVersionUID = -3543706150331178640l;
  public Movement() {  }
  public Movement(String elementName) {
    setElementName(elementName);
  }
  
  public Object clone() {
    Movement m = new Movement();
    m.elementName = elementName;
    m.cyclicLap = cyclicLap;
    int nb = states.length;
    m.states = new MovementState[nb];
    for(int i = 0; i < nb; i++) m.states[i] = (MovementState) states[i].clone();
    return m;
  }
  
  private String elementName;
  public String getElementName() { return elementName; }
  public synchronized void setElementName(String name) { elementName = name; }

  private float cyclicLap;
  public float getCyclicLap() { return cyclicLap; }
  public boolean isCyclic() { return cyclicLap > 0; }
  public void setCyclicLap(float f) { cyclicLap = f; }

  private MovementState[] states = new MovementState[0];
  public synchronized InterpolatedState getState(float time) {
    int l = states.length - 1;
    if(l == -1) return null;
    if(time == states[0].time) return new InterpolatedState(states[0]);
    if(time < states[0].time) {
      if(isCyclic()) return new InterpolatedState((cyclicLap - (states[0].time - time)) / cyclicLap, states[l], states[0]);
      else return new InterpolatedState(states[0]);
    }
    if(time == states[l].time) return new InterpolatedState(states[l]);
    if(time > states[l].time) {
      if(isCyclic()) return new InterpolatedState((time - states[l].time) / cyclicLap, states[l], states[0]);
      else return new InterpolatedState(states[l]);
    }
    for(int i = 1; i <= l; i++) { // start with i == 1 because the first state was already checked.
      if(time <= states[i].time) return new InterpolatedState(states[i - 1], states[i], time);
    }
    return null;
  }
  public synchronized float[] times() {
    int l = states.length;
    float[] times = new float[l];
    for(int i = 0; i < l; i++) times[i] = states[i].time;
    return times;
  }
  public void addState(float time, GraphicalElement3D e) {
    addState(new MovementState(time, e));
  }
  public synchronized void addState(MovementState s) {
    int l = states.length;
    for(int i = 0; i < l; i++) {
      if(s.time < states[i].time) { // insert.
        MovementState[] states2 = new MovementState[l + 1];
        System.arraycopy(states, 0, states2, 0, i);
        states2[i] = s;
        System.arraycopy(states, i, states2, i + 1, l - i);
        states = states2;
        return;
      }
      if(s.time == states[i].time) { // replace.
        states[i] = s;
        return;
      }
    }
    // add at the end.
    MovementState[] states2 = new MovementState[l + 1];
    System.arraycopy(states, 0, states2, 0, l);
    states2[l] = s;
    states = states2;
  }
  public synchronized void removeState(float time) {
    for(int i = 0; i < states.length; i++) {
      if(time == states[i].time) {
        MovementState[] states2 = new MovementState[states.length - 1];
        System.arraycopy(states, 0, states2, 0, i);
        System.arraycopy(states, i + 1, states2, i, states.length - i - 1);
        states = states2;
        return;
      }
    }
  }
  
  public void scale(float t) {
    if(t == 0f) return;
    for(int i = 0; i < states.length; i++) states[i].time = states[i].time * t;
  }
  public void changeTime(float oldTime, float newTime) {
    for(int i = 0; i < states.length; i++) {
      if(states[i].time == oldTime) states[i].time = newTime;
    }
  }
  
  public String toString() {
    String s;
    s = "Movement for " + elementName + "\n";
    s = s + states.length + " states : {\n";
    for(int i = 0; i < states.length; i++) s = s + states[i].toString() + "\n";
    s = s + "}";
    return s;
  }
}
