/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d.fx.particle;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import opale.soya.soya3d.model.*;
import opale.soya.soya3d.fx.*;
import gl4java.*;
import java.io.*;
import java.beans.*;

/**
 * A bunch of moving particles, like a black hole.
 * 
 * In such a bunch, the particles are attracted by the center of the black hole.
 * 
 * @author Artiste on the Web
 */

public class BlackHole3D extends MovingParticlesBunch3D /* implements */ {
  private static final long serialVersionUID = 8545499030019472717l;
  /**
   * Create a black hole particles bunch.
   */
  public BlackHole3D() {
    super();
  }
  /**
   * Create a black hole particles bunch.
   * @param l the object that manages the particles' lifes
   * @param c the object that manages the particles' colors
   * @param d the object that draws the particles
   */
  public BlackHole3D(ParticlesLifes l, ParticlesColors c, ParticlesDrawer d) {
    super(l, c, d);
  }
  
  /**
   * Clones this black hole particles bunch.
   * @return the clone
   */
  public Object clone() {
    BlackHole3D o = (BlackHole3D) super.clone();
    o.acceleration = acceleration;
    o.minSpeed     = minSpeed    ;
    o.maxSpeed     = maxSpeed    ;
    return o;
  }
  
  public void initParticle(int id) {
    super.initParticle(id);
    
    double angle1 = Math.random() * 6.28318531d; // 0 to 360
    double angle2 = Math.random() * 3.142d     ; // 0 to 180
    float  length = ((float) Math.random()) * (maxSpeed - minSpeed) + minSpeed;
    
    double sin = Math.sin(angle2);
    
    float
      x = (1f + 1f * length) * (float) (sin * Math.cos(angle1)),
      y = (1f + 1f * length) * (float) (      Math.cos(angle2)),
      z = (1f + 1f * length) * (float) (sin * Math.sin(angle1));
    /*
    positions[id    ] = (1f + 1f * length) * (float) (sin * Math.cos(angle1));
    positions[id + 1] = (1f + 1f * length) * (float) (      Math.cos(angle2));
    positions[id + 2] = (1f + 1f * length) * (float) (sin * Math.sin(angle1));
    */
    
    angle1 = Math.random() * 6.28318531d; // 0 to 360
    angle2 = Math.random() * 3.142d     ; // 0 to 180
    //float  length = ((float) Math.random()) * (maxSpeed - minSpeed) + minSpeed;
    
    sin = Math.sin(angle2);
    
    float
      speedX = length * (float) (sin * Math.cos(angle1)),
      speedY = length * (float) (      Math.cos(angle2)),
      speedZ = length * (float) (sin * Math.sin(angle1));
    
    /*
    speeds[id    ] = length * (float) (sin * Math.cos(angle1));
    speeds[id + 1] = length * (float) (      Math.cos(angle2));
    speeds[id + 2] = length * (float) (sin * Math.sin(angle1));
    */
    if(
       speedX * x +
       speedY * y +
       speedZ * z   > 0f) {
      speedX = -speedX;
      speedY = -speedY;
      speedZ = -speedZ;
    }
    initParticlePosition(id, x, y, z);
    initParticleSpeed(id, speedX, speedY, speedZ);
  }
  
  private transient float centerX, centerY, centerZ;
  public void advanceBunch(float factor) {
    if(coordSyst == this)
      centerX = centerY = centerZ = 0f;
    else {
      if(coordSyst == null) {
        float[] m = getRootMatrix();
        centerX = m[12];
        centerY = m[13];
        centerZ = m[14];
      }
      else {
        Position p = new Point(0f, 0f, 0f, this);
        p.setCoordSyst(coordSyst);
        centerX = p.getX();
        centerY = p.getY();
        centerZ = p.getZ();
      }
    }
    
    super.advanceBunch(factor);
  }
  public void advanceParticle(int id, float factor) {
    super.advanceParticle(id, factor);
    
    id = id * 3;
    float dist2;
    if(coordSyst == this) {
      dist2 = positions[id] * positions[id] + positions[id + 1] * positions[id + 1] + positions[id + 2] * positions[id + 2];
      speeds[id    ] = speeds[id    ] - factor * acceleration * positions[id    ] / dist2;
      speeds[id + 1] = speeds[id + 1] - factor * acceleration * positions[id + 1] / dist2;
      speeds[id + 2] = speeds[id + 2] - factor * acceleration * positions[id + 2] / dist2;
    }
    else {
      dist2 =
        Matrix.pow2(positions[id    ] - centerX) +
        Matrix.pow2(positions[id + 1] - centerY) +
        Matrix.pow2(positions[id + 2] - centerZ);
      speeds[id    ] = speeds[id    ] - factor * acceleration * (positions[id    ] - centerX) / dist2;
      speeds[id + 1] = speeds[id + 1] - factor * acceleration * (positions[id + 1] - centerY) / dist2;
      speeds[id + 2] = speeds[id + 2] - factor * acceleration * (positions[id + 2] - centerZ) / dist2;
    }
  }
  public void deleteParticle(int id) {
    super.deleteParticle(id);
  }
  
  // Particles properties :
  
  // Bunch properties :
  protected float acceleration = .002f;
  /**
   * Gets the center's attraction's strenght. The higher the acceleration is, the more the
   * particles are deviated.
   * Default is .002f.
   * @return the acceleration
   */
  public float getAcceleration() { return acceleration; }
  /**
   * Sets the center's attraction's strenght.
   * @param f the acceleration
   */
  public void setAcceleration(float f) {
    acceleration = f;
    firePropertyChange("acceleration");
  }
  
  protected float minSpeed = .02f, maxSpeed = .07f;
  /**
   * Gets the minimum speed of a new created particle.
   * Default is 0.02f.
   * @return the minimum initial speed
   */
  public float getMinimumSpeed() { return minSpeed; }
  /**
   * Gets the maximum speed of a new created particle.
   * Default is 0.07f.
   * @return the maximum initial speed
   */
  public float getMaximumSpeed() { return maxSpeed; }
  /**
   * Sets the minimum speed of a new created particle.
   * @param f the new minimum initial speed
   */
  public void setMinimumSpeed(float f) {
    minSpeed = f;
    firePropertyChange("minimumSpeed");
  }
  /**
   * Sets the maximum speed of a new created particle.
   * @param f the new maximum initial speed
   */
  public void setMaximumSpeed(float f) {
    maxSpeed = f;
    firePropertyChange("maximumSpeed");
  }
}
