/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d.fx.particle;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import opale.soya.soya3d.model.*;
import opale.soya.soya3d.fx.*;
import gl4java.*;
import java.io.*;
import java.beans.*;

/**
 * A bunch of moving particles, like a fountain.
 * 
 * In such a bunch, particles are ejected in the fountain's y direction, and their speed is
 * modified by an acceleration vector. The acceleration is often used as a gavity. If you
 * want the particles to be fired in an other direction, just rotate the fountain.
 * 
 * @author Artiste on the Web
 */

public class Fountain3D extends MovingParticlesBunch3D /* implements */ {
  /**
   * Create a fountain particles bunch.
   */
  public Fountain3D() { super(); }
  /**
   * Create a fountain particles bunch.
   * @param l the object that manages the particles' lifes
   * @param c the object that manages the particles' colors
   * @param d the object that draws the particles
   */
  public Fountain3D(ParticlesLifes l, ParticlesColors c, ParticlesDrawer d) { super(l, c, d); }
  
  /**
   * Clones this fountain particles bunch.
   * @return the clone
   */
  public Object clone() {
    Fountain3D o = (Fountain3D) super.clone();
    o.angle        = angle       ;
    o.minSpeed     = minSpeed    ;
    o.maxSpeed     = maxSpeed    ;
    o.acceleration = (Vector) acceleration.clone();
    
    return o;
  }
  
  public void initParticle(int id) {
    super.initParticle(id);
    
    initParticlePosition(id, 0f, 0f, 0f);
    /*
    positions[id * 3    ] = 0f;
    positions[id * 3 + 1] = 0f;
    positions[id * 3 + 2] = 0f;
    */
    
    double angle1 = Math.random() * 6.28318531d; // 0 to 360
    double angle2 = .0174533d * (Math.random() * (double) angle);
    float  length = ((float) Math.random()) * (maxSpeed - minSpeed) + minSpeed;
    
    double sin = Math.sin(angle2);
    
    initParticleSpeed(id,
                      length * (float) (sin * Math.cos(angle1)),
                      length * (float) (      Math.cos(angle2)),
                      length * (float) (sin * Math.sin(angle1))
                      );
    /*
    speeds[id * 3    ] = length * (float) (sin * Math.cos(angle1));
    speeds[id * 3 + 1] = length * (float) (      Math.cos(angle2));
    speeds[id * 3 + 2] = length * (float) (sin * Math.sin(angle1));
    */
  }
  
  private transient float accelerationX, accelerationY, accelerationZ;
  public void advanceParticle(int id, float factor) {
    if(id == 0) {
      if(acceleration.getCoordSyst() == this) {
        accelerationX = acceleration.getX();
        accelerationY = acceleration.getY();
        accelerationZ = acceleration.getZ();
      }
      else {
        Vector v;
        if(acceleration.getCoordSyst() == null) v = (Vector) acceleration.clone(getRootParent());
        else                                    v = (Vector) acceleration.clone(this);
        accelerationX = v.getX();
        accelerationY = v.getY();
        accelerationZ = v.getZ();
      }
    }
    super.advanceParticle(id, factor);
    
    id = id * 3;
    speeds[id    ]    = speeds[id    ]    + factor * accelerationX;
    speeds[id + 1]    = speeds[id + 1]    + factor * accelerationY;
    speeds[id + 2]    = speeds[id + 2]    + factor * accelerationZ;
  }
  public void deleteParticle(int id) {
    super.deleteParticle(id);
  }
  
  // Particles properties :
  
  // Bunch properties :
  protected Vector acceleration = new Vector(0f, -.0012f, 0f, this);
  /**
   * Gets the particles' acceleration vector. In a fountain, all the particles share the same
   * acceleration vector.
   * Default is new Vector(0f, -.001f, 0f, this), so particles fall down.
   * @return the particles' acceleration vector
   */
  public Vector getAcceleration() { return acceleration; }
  /**
   * Sets the particles' acceleration vector.
   * @param v the new particles' acceleration vector
   */
  public void setAcceleration(Vector v) {
    acceleration = v;
    firePropertyChange("acceleration");
  }
  
  /** Gets the particles' acceleration vector x component */
  public float getAccelerationX() { return acceleration.getX(); }
  /** Gets the particles' acceleration vector y component */
  public float getAccelerationY() { return acceleration.getY(); }
  /** Gets the particles' acceleration vector z component */
  public float getAccelerationZ() { return acceleration.getZ(); }
  /** Sets the particles' acceleration vector x component */
  public void setAccelerationX(float f) {
    acceleration.setX(f);
    firePropertyChange("acceleration");
  }
  /** Sets the particles' acceleration vector y component */
  public void setAccelerationY(float f) {
    acceleration.setY(f);
    firePropertyChange("acceleration");
  }
  /** Sets the particles' acceleration vector z component */
  public void setAccelerationZ(float f) {
    acceleration.setZ(f);
    firePropertyChange("acceleration");
  }
  
  protected float angle = 20f;
  /**
   * Gets the angle of the fountain's stream, in degrees. 0f will give a straight stream.
   * Default is 20f.
   * @return the angle of the fountain's stream
   */
  public float getAngle() { return angle; }
  /**
   * Sets the angle of the fountain's stream, in degrees
   * @param f the new angle of the fountain's stream
   */
  public void setAngle(float f) {
    angle = f;
    firePropertyChange("angle");
  }

  protected float minSpeed = .05f, maxSpeed = .08f;
  /**
   * Gets the minimum speed of a new created particle.
   * Default is 0.02f.
   * @return the minimum initial speed
   */
  public float getMinimumSpeed() { return minSpeed; }
  /**
   * Gets the maximum speed of a new created particle.
   * Default is 0.07f.
   * @return the maximum initial speed
   */
  public float getMaximumSpeed() { return maxSpeed; }
  /**
   * Sets the minimum speed of a new created particle.
   * @param f the new minimum initial speed
   */
  public void setMinimumSpeed(float f) {
    minSpeed = f;
    firePropertyChange("minimumSpeed");
  }
  /**
   * Sets the maximum speed of a new created particle.
   * @param f the new maximum initial speed
   */
  public void setMaximumSpeed(float f) {
    maxSpeed = f;
    firePropertyChange("maximumSpeed");
  }
}
