/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d.fx.particle;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import opale.soya.soya3d.model.*;
import gl4java.*;
import java.io.*;
import java.beans.*;

/**
 * An object that manages particles' lifes.
 * 
 * The life of a particle starts at 1f, and when it's fallen below 0f, the particle is dead.
 * 
 * The life of each particle decrease with time, according to the life-speed. The higher the
 * life-speed is, the shorter is the particle life.
 * 
 * The life-speed is different for all particles, and it is determinated randomly at the
 * particle's birth, between 2 limits, the minimum and maximum life-speed.
 * 
 * @author Artiste on the Web
 */

public class ParticlesLifes extends AbstractBean implements ParticlesStuff {
  private static final long serialVersionUID = -1855721593693540841l;
  /**
   * Create a particles lifes.
   */
  public ParticlesLifes() {
    super();
    minLifeSpeed = .01f;
    maxLifeSpeed = .01f;
  }
  /**
   * Create a particles lifes.
   * @param minimumLifeSpeed the minimum of life speed
   * @param maximumLifeSpeed the maximum of life speed
   */
  public ParticlesLifes(float minimumLifeSpeed, float maximumLifeSpeed) {
    super();
    minLifeSpeed = minimumLifeSpeed;
    maxLifeSpeed = maximumLifeSpeed;
  }
  
  /**
   * Clones this particles lifes.
   * @return the clone
   */
  public Object clone() {
    ParticlesLifes o = null;
    try { o = (ParticlesLifes) getClass().newInstance(); }
    catch(Exception e) { System.out.println("Cannot clone !"); e.printStackTrace(); }
    o.minLifeSpeed = minLifeSpeed;
    o.maxLifeSpeed = maxLifeSpeed;
    
    return o;
  }
  
  protected ParticlesBunch3D bunch;
  public ParticlesBunch3D getBunch() { return bunch; }
  
  public void initParticle(int id) {
    lifes     [id] = 1f;
    lifeSpeeds[id] = minLifeSpeed + (maxLifeSpeed - minLifeSpeed) * (float) Math.random();
  }
  public void advanceParticle(int id, float factor) {
    lifes[id] = lifes[id] - factor * lifeSpeeds[id];
    if(lifes[id] <= 0f) bunch.deleteParticle(id);
  }
  public void deleteParticle(int id) {
    lifes[id] = 0f;
  }
  
  
  // Particles properties :
  protected float[] lifes     ;
  protected float[] lifeSpeeds;
  /**
   * Gets the life of the particle of the given ID.
   * @return the particle's life
   */
  public float getLife(int id) { return lifes[id]; }
  /**
   * Checks if the particle of the given ID is still alive.
   * @return true if the particle is not dead
   */
  public boolean isAlive(int id) { return lifes[id] > 0f; }
  
  // A few properties.
  protected int nbParticles;
  public int getNumberOfParticles() { return nbParticles; }
  public void setNumberOfParticles(int i) {
    if(lifes == null) {
      lifes      = new float[i];
      lifeSpeeds = new float[i];
    }
    else {
      float[] lifes2      = new float[i];
      float[] lifeSpeeds2 = new float[i];
      if(nbParticles < i) {
        System.arraycopy(lifes     , 0, lifes2     , 0, nbParticles);
        System.arraycopy(lifeSpeeds, 0, lifeSpeeds2, 0, nbParticles);
      }
      else {
        System.arraycopy(lifes     , 0, lifes2     , 0, i);
        System.arraycopy(lifeSpeeds, 0, lifeSpeeds2, 0, i);
      }
      lifes       = lifes2     ;
      lifeSpeeds  = lifeSpeeds2;
    }
    nbParticles = i;
    firePropertyChange("numberOfParticles");
  }
  
  protected float minLifeSpeed, maxLifeSpeed;
  /**
   * Gets the minimum life-speed for a new particle.
   * @return the minimum life-speed
   */
  public float getMinimumLifeSpeed() { return minLifeSpeed; }
  /**
   * Gets the maximum life-speed for a new particle.
   * @return the maximum life-speed
   */
  public float getMaximumLifeSpeed() { return maxLifeSpeed; }
  /**
   * Sets the minimum life-speed for a new particle.
   * @param f the new minimum life-speed
   */
  public void setMinimumLifeSpeed(float f) {
    minLifeSpeed = f;
    firePropertyChange("minimumLifeSpeed");
  }
  /**
   * Sets the maximum life-speed for a new particle.
   * @param f the new maximum life-speed
   */
  public void setMaximumLifeSpeed(float f) {
    maxLifeSpeed = f;
    firePropertyChange("maximumLifeSpeed");
  }
}
