/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d.model;

import opale.soya.*;
import opale.soya.util.Matrix;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import gl4java.*;
import java.util.Iterator;
import java.io.*;
import java.beans.*;

abstract class AbstractFragment extends Object implements Fragment, Lit, SmoothLitBuilder, FaceVisibility, Raypicking, Serializable /* really??? */{
  private static final long serialVersionUID = -732971049175716669l;
  public AbstractFragment() {  }
  
  public synchronized Object clone() {
    AbstractFragment f;
    try { f = (AbstractFragment) getClass().newInstance(); }
    catch(Exception e) { e.printStackTrace(); return null; }
    f.nbFaces          = nbFaces         ;
    f.nbNonHiddenFaces = nbNonHiddenFaces;
    f.nbPoints         = nbPoints        ;
    f.usePointsNormals = usePointsNormals;
    f.usePointsColors  = usePointsColors ;
    f.useTexCoords     = useTexCoords    ;
    f.normals = (float[]) normals.clone();
    f.points  = (float[]) points .clone();
    if(usePointsNormals) f.pointsNormals   = (float[]) pointsNormals.clone  ();
    if(usePointsColors)  f.pointsColors    = (float[]) pointsColors.clone   ();
    if(useTexCoords)     f.pointsTexCoords = (float[]) pointsTexCoords.clone();
    f.smoothLit      = smoothLit     ;
    f.staticLit      = staticLit     ;
    f.material       = material      ;
    f.visibility     = visibility    ;
    f.radiusSquarred = radiusSquarred;
    f.useAlpha       = useAlpha      ;
    f.minX = minX;
    f.minY = minY;
    f.minZ = minZ;
    f.maxX = maxX;
    f.maxY = maxY;
    f.maxZ = maxZ;
    return f;
  }

  protected transient int id;
  public void finalize() {
    Soya.releaseCallListID(id);
  }

  protected int nbFaces;
  protected int nbNonHiddenFaces;
  
  protected int nbPoints;
  protected float[] normals;              // 3 by face
  protected float[] points;               // 3 by point
  protected boolean usePointsNormals;
  protected float[] pointsNormals;        // 3 by point
  protected boolean usePointsColors;
  protected float[] pointsColors;         // 4 by point
  protected boolean useTexCoords;
  protected float[] pointsTexCoords;      // 2 by point
  
  public int getNumberOfFaces() { return nbFaces; }
  public int getNumberOfShapeElements() { return getNumberOfFaces(); }
  public boolean canBuildFromShapeElements() { return true; }
  public synchronized boolean canCombineWith(Fragment f) {
    synchronized(f) {
    if(f.getClass() != getClass()) return false;
      AbstractFragment af = (AbstractFragment) f;
      if(af.getMaterial()              != material                  ) return false;
      if(af.getNumberOfPointsPerFace() != getNumberOfPointsPerFace()) return false;
      if(af.isSmoothLit()              != smoothLit                 ) return false;
      if(af.isStaticLit()              != staticLit                 ) return false;
      if(af.getUseAlpha()              != useAlpha                  ) return false;
      if(af.getUsePointsColors()       != usePointsColors           ) return false;
      if(af.visibility                 != visibility                ) return false;
      return true;
    }
  }
  public synchronized Fragment combineWith(Fragment frag) {
    synchronized(frag) {
      AbstractFragment f = (AbstractFragment) frag;
      
      int oldNbFaces  = nbFaces ;
      int oldNbPoints = nbPoints;
      float[] oldNormals = normals;
      float[] oldPoints  = points ;
      int nbppf = getNumberOfPointsPerFace();
      
      int l1 = 3 *   nbNonHiddenFaces;
      int l2 = 3 * f.nbNonHiddenFaces;
      int l3 = 3 * (oldNbFaces - nbNonHiddenFaces);
      nbFaces = nbFaces + f.nbFaces;
      normals = new float[3 * nbFaces];
      System.arraycopy(oldNormals, 0 , normals, 0           , l1);
      System.arraycopy( f.normals, 0 , normals, l1          , l2);
      System.arraycopy(oldNormals, l1, normals, l1 + l2     , l3);
      System.arraycopy( f.normals, l2, normals, l1 + l2 + l3, 3 * (f.nbFaces - f.nbNonHiddenFaces));

      //System.arraycopy(oldNormals, 0, normals, 0             , 3 * oldNbFaces);
      //System.arraycopy( f.normals, 0, normals, 3 * oldNbFaces, 3 * f.nbFaces );
      
      nbPoints = nbppf * nbFaces;
      points = new float[3 * nbPoints];
      l1 = 3 * nbppf *   nbNonHiddenFaces;
      l2 = 3 * nbppf * f.nbNonHiddenFaces;
      l3 = 3 * nbppf * (oldNbFaces - nbNonHiddenFaces);
      System.arraycopy(oldPoints, 0 , points, 0           , l1);
      System.arraycopy( f.points, 0 , points, l1          , l2);
      System.arraycopy(oldPoints, l1, points, l1 + l2     , l3);
      System.arraycopy( f.points, l2, points, l1 + l2 + l3, 3 * nbppf * (f.nbFaces - f.nbNonHiddenFaces));
      
      //System.arraycopy(oldPoints, 0, points, 0              , 3 * oldNbPoints);
      //System.arraycopy( f.points, 0, points, 3 * oldNbPoints, 3 * f.nbPoints );
      
      if(usePointsNormals) {
        float[] oldPointsNormals = pointsNormals;
        pointsNormals = new float[3 * nbPoints];
        System.arraycopy(oldPointsNormals, 0 , pointsNormals, 0           , l1);
        System.arraycopy( f.pointsNormals, 0 , pointsNormals, l1          , l2);
        System.arraycopy(oldPointsNormals, l1, pointsNormals, l1 + l2     , l3);
        System.arraycopy( f.pointsNormals, l2, pointsNormals, l1 + l2 + l3, 3 * nbppf * (f.nbFaces - f.nbNonHiddenFaces));
        
        //System.arraycopy(oldPointsNormals, 0, pointsNormals, 0              , 3 * oldNbPoints);
        //System.arraycopy( f.pointsNormals, 0, pointsNormals, 3 * oldNbPoints, 3 * f.nbPoints );
      }
      if(usePointsColors) {
        float[] oldPointsColors = pointsColors;
        pointsColors = new float[4 * nbPoints];
        l1 = 4 * nbppf *   nbNonHiddenFaces;
        l2 = 4 * nbppf * f.nbNonHiddenFaces;
        l3 = 4 * nbppf * (oldNbFaces - nbNonHiddenFaces);
        System.arraycopy(oldPointsColors, 0 , pointsColors, 0           , l1);
        System.arraycopy( f.pointsColors, 0 , pointsColors, l1          , l2);
        System.arraycopy(oldPointsColors, l1, pointsColors, l1 + l2     , l3);
        System.arraycopy( f.pointsColors, l2, pointsColors, l1 + l2 + l3, 4 * nbppf * (f.nbFaces - f.nbNonHiddenFaces));
        
        //System.arraycopy(oldPointsColors, 0, pointsColors, 0              , 4 * oldNbPoints);
        //System.arraycopy( f.pointsColors, 0, pointsColors, 4 * oldNbPoints, 4 * f.nbPoints );
      }
      if(useTexCoords) {
        float[] oldPointsTexCoords = pointsTexCoords;
        pointsTexCoords = new float[2 * nbPoints];
        l1 = 2 * nbppf *   nbNonHiddenFaces;
        l2 = 2 * nbppf * f.nbNonHiddenFaces;
        l3 = 2 * nbppf * (oldNbFaces - nbNonHiddenFaces);
        System.arraycopy(oldPointsTexCoords, 0 , pointsTexCoords, 0           , l1);
        System.arraycopy( f.pointsTexCoords, 0 , pointsTexCoords, l1          , l2);
        System.arraycopy(oldPointsTexCoords, l1, pointsTexCoords, l1 + l2     , l3);
        System.arraycopy( f.pointsTexCoords, l2, pointsTexCoords, l1 + l2 + l3, 2 * nbppf * (f.nbFaces - f.nbNonHiddenFaces));
        
        //System.arraycopy(oldPointsTexCoords, 0, pointsTexCoords, 0              , 2 * oldNbPoints);
        //System.arraycopy( f.pointsTexCoords, 0, pointsTexCoords, 2 * oldNbPoints, 2 * f.nbPoints );
      }
      nbNonHiddenFaces = nbNonHiddenFaces + f.nbNonHiddenFaces;
      if((shape != null) && (!shape.isLocked())) reBuild();
      return this;
    }
  }

  protected transient Material material;
  public Material getMaterial() { return material; }
  public synchronized void setMaterial(Material m) {
    if(material != m) {
      material = m;
      useTexCoords = (material.getTexture() != null);
      defineUseAlpha();
      if(!shape.isLocked()) isBuilt = false;
    }
  }

  public int getNumberOfPoints() { return nbPoints; }
  public abstract int getNumberOfPointsPerFace();
  public boolean getUsePointsNormals() { return usePointsNormals; }
/*  public void setUsePointsNormals(boolean b) {
    if(usePointsNormals != b) {
      usePointsNormals = b;

    }
  }*/
  public boolean getUsePointsColors() { return usePointsColors; }
/*  public void setUsePointsColors(boolean b) {
    if(usePointsColors != b) {
      usePointsColors = b;

    }
  }*/
  public boolean getUseTexCoords() { return useTexCoords; }

  protected boolean useAlpha;
  public boolean getUseAlpha() { return useAlpha; }
  protected void defineUseAlpha() {
    if(material.getUseAlpha()) {
      useAlpha = true;
      return;
    }
    if(usePointsColors) {
      int max = nbPoints * 4 - 1;
      for(int i = 3; i <= max; i = i + 4) {
        if(pointsColors[i] != 1) {
          useAlpha = true;
          return;
        }
      }
    }
  }

  protected boolean staticLit;
  public boolean isStaticLit() { return staticLit; }
  public synchronized void setStaticLit(boolean b) {
    if(staticLit != b) {
      staticLit = b;
    }
  }
  public void applyStaticLighting(Light3D[] lights, CoordSyst fromCoordSyst) {
    staticLit = true;
    if(usePointsColors == false) {
      usePointsColors = true;
      pointsColors = new float[nbPoints * 4];
      int index = 0;
      for(int i = 0; i < nbPoints; i++) {
        pointsColors[index++] = material.getDiffuseRed  ();
        pointsColors[index++] = material.getDiffuseGreen();
        pointsColors[index++] = material.getDiffuseBlue ();
        pointsColors[index++] = material.getDiffuseAlpha();
      }
    }
    
    float[] initialColor = new float[4];
    float[] color;
    Position point = new Point();
    point.setCoordSyst(fromCoordSyst);
    Vector normal = new Vector();
    normal.setCoordSyst(fromCoordSyst);
    int normalIndex = 0;
    
    for(int i = 0; i < nbPoints; i++) {
      point.move(points[3 * i], points[3 * i + 1], points[3 * i + 2]);
      if(usePointsNormals) normal.move(pointsNormals[3 * i], pointsNormals[3 * i + 1], pointsNormals[3 * i + 2]);
      else {
        if((i % getNumberOfPointsPerFace()) == 0)
          normal.move(normals[normalIndex++], normals[normalIndex++], normals[normalIndex++]);
      }
      System.arraycopy(pointsColors, i * 4, initialColor, 0, 4);
      pointsColors[i * 4    ] = 0f;
      pointsColors[i * 4 + 1] = 0f;
      pointsColors[i * 4 + 2] = 0f;
      //pointsColors[i * 4 + 3] = initialColor[3];
      pointsColors[i * 4 + 3] = initialColor[3] * pointsColors[i * 4 + 3];
      for(int j = 0; j < lights.length; j++) {
        color = lights[j].diffuseColorAt(point, normal, initialColor);
        pointsColors[i * 4    ] = pointsColors[i * 4    ] + color[0];
        pointsColors[i * 4 + 1] = pointsColors[i * 4 + 1] + color[1];
        pointsColors[i * 4 + 2] = pointsColors[i * 4 + 2] + color[2];
        //pointsColors[i * 4 + 3] = pointsColors[i * 4 + 3] + color[3];
      }
      if(pointsColors[i * 4    ] < 0f) pointsColors[i * 4    ] = 0f;
      if(pointsColors[i * 4 + 1] < 0f) pointsColors[i * 4 + 1] = 0f;
      if(pointsColors[i * 4 + 2] < 0f) pointsColors[i * 4 + 2] = 0f;
    }
    reBuild();
  }
  
  protected boolean smoothLit;
  public boolean isSmoothLit() { return smoothLit; }
  public synchronized void setSmoothLit(boolean b) {
    if(smoothLit != b) {
      smoothLit = b;
      if(!shape.isLocked()) {
        //if(smoothLit) buildSmoothLit();
        isBuilt = false;
      }
    }
  }
  protected int visibility;
  public int getVisibility() { return visibility; }
  public synchronized void setVisibility(int v) {
    if(visibility != v) {
      visibility = v;
      if((shape != null) && (!shape.isLocked())) {
        buildVisibility();
        //if(smoothLit) buildSmoothLit();
        isBuilt = false;
      }
    }
  }

  public Shape shape;
  public Shape getShape() { return shape; }
  public synchronized void setShape(Shape newShape) { shape = newShape; }

  // Buildings :
  public synchronized void reBuild() {
    if(!shape.isLocked()) {
      defineUseAlpha();
      buildDimension();
      buildVisibility();
      //if(smoothLit) buildSmoothLit();
      isBuilt = false;
    }
  }
  private transient boolean isBuilt;
  public synchronized void build(GLFunc gl, GLUFunc glu) {
    if(!shape.isLocked()) {
      buildCallList(gl, glu);
      isBuilt = true;
    }
  }
  public synchronized void buildDimension() {
    int imax = nbPoints * 3;
    minX = Float.POSITIVE_INFINITY;
    minY = Float.POSITIVE_INFINITY;
    minZ = Float.POSITIVE_INFINITY;
    maxX = Float.NEGATIVE_INFINITY;
    maxY = Float.NEGATIVE_INFINITY;
    maxZ = Float.NEGATIVE_INFINITY;
    for(int i = 0; i < imax; ) {
      if(points[i] < minX) minX = points[i];
      else { if(points[i] > maxX) maxX = points[i]; }
      i++;
      
      if(points[i] < minY) minY = points[i];
      else { if(points[i] > maxY) maxY = points[i]; }
      i++;
      
      if(points[i] < minZ) minZ = points[i];
      else { if(points[i] > maxZ) maxZ = points[i]; }
      i++;
    }
    //radiusSquarred = Matrix.pow2(Math.abs(minX) - Math.abs(maxX)) + Matrix.pow2(Math.abs(minY) - Math.abs(maxY)) + Matrix.pow2(Math.abs(minY) - Math.abs(maxY));
    radiusSquarred = Matrix.pow2(Math.max(Math.abs(minX), Math.abs(maxX))) +
                     Matrix.pow2(Math.max(Math.abs(minY), Math.abs(maxY))) +
                     Matrix.pow2(Math.max(Math.abs(minZ), Math.abs(maxZ)));
  }
  public abstract void buildVisibility();
  public abstract void drawFaces(GLFunc gl, GLUFunc glu);
  public abstract void drawFacesWithoutArrays(GLFunc gl, GLUFunc glu);
  public synchronized void buildCallList(GLFunc gl, GLUFunc glu) {
    if(id == 0) id = gl.glGenLists(1);
    
    enableArrays(gl, glu);
    
    gl.glNewList(id, GLEnum.GL_COMPILE);
    
    defineArrays(gl, glu);
    
    if(visibility == FaceVisibility.VISIBILITY_ALL) {
      gl.glDisable(GLEnum.GL_CULL_FACE);  // Cull face is default
      gl.glLightModelf(GLEnum.GL_LIGHT_MODEL_TWO_SIDE, 1);
    }
    
    if(usePointsColors) {
      gl.glEnable (GLEnum.GL_COLOR_MATERIAL);
      /*
      if(staticLit) {
        gl.glColorMaterial(GLEnum.GL_FRONT_AND_BACK, GLEnum.GL_EMISSION);
        material.makeColorCurrent(gl, glu);
      }
      */
    }
    
    drawFaces(gl, glu);
    //drawFacesWithoutArrays(gl, glu);
    
    if(visibility == FaceVisibility.VISIBILITY_ALL) {
      gl.glEnable(GLEnum.GL_CULL_FACE);
      gl.glLightModelf(GLEnum.GL_LIGHT_MODEL_TWO_SIDE, 0);
    }
    if(usePointsColors) {
      /*
      if(staticLit) {
        gl.glColor4f(0f, 0f, 0f, 1f);
        gl.glColorMaterial(GLEnum.GL_FRONT_AND_BACK, GLEnum.GL_AMBIENT_AND_DIFFUSE);
      }
      */
      gl.glDisable(GLEnum.GL_COLOR_MATERIAL);
      material.makeColorCurrent(gl, glu); // reset the color to the material default, if it has been changed.
    }
    
    gl.glEndList();
    
    checkError(gl, glu);
  }
  
  public abstract void buildSmoothLit(PointsIdentifier pi);
  protected class WriteInArrayListener implements PointRecord.NormalComputationListener {
    public WriteInArrayListener(int newIndex) { index = newIndex; }
    private int index;
    public void normalComputed(float nx, float ny, float nz) {
      pointsNormals[index    ] = nx;
      pointsNormals[index + 1] = ny;
      pointsNormals[index + 2] = nz;
    }
  }
  
  public synchronized String toString() {
    String s = getClass().getName() + "@ {\n" + Integer.toString(nbFaces) + " faces\n";
    s = s + "Material : " + material.getName() + "\n";
    if(staticLit) s = s + "static lit\n";
    s = s + "Call list ID : " + Integer.toString(id) + "\n";
    int normalIndex = 0, pointIndex = 0, pointNormalIndex = 0, pointTexCoordIndex = 0, pointColorIndex = 0;
    for(int j = 0; j < nbFaces; j++) {
      s =  s + "Normal " + normals[normalIndex++] + ", " +  normals[normalIndex++] + ", " +  normals[normalIndex++] + "\n";
      for(int i = 0; i < getNumberOfPointsPerFace(); i++) {
        s =  s + "  Point " + points[pointIndex++] + ", " +  points[pointIndex++] + ", " +  points[pointIndex++] + "\n";
        if(usePointsNormals) s =  s + "  Normal " + pointsNormals[pointNormalIndex++] + ", " +  pointsNormals[pointNormalIndex++] + ", " +  pointsNormals[pointNormalIndex++] + "\n";
        if(usePointsColors) s =  s + "  Color " + pointsColors[pointColorIndex++] + ", " +  pointsColors[pointColorIndex++] + ", " +  pointsColors[pointColorIndex++] + ", " +  pointsColors[pointColorIndex++] + "\n";
        if(useTexCoords) s =  s + "  Tex Coord " + pointsTexCoords[pointTexCoordIndex++] + ", " +  pointsTexCoords[pointTexCoordIndex++] + "\n";
      }
    }
    s = s + "}\n";
    return s;
  }
  
  protected float minX, minY, minZ, maxX, maxY, maxZ;
  protected double radiusSquarred;
  public float getRadiusSquarred() { return (float) radiusSquarred; }
  public boolean ensureRaypickIsUsefull(float f) { return true; }
  public float getWidth () { return maxX - minX; }
  public float getHeight() { return maxY - minY; }
  public float getDepth () { return maxZ - minZ; }
  public synchronized void setWidth (float f) { scale(f / getWidth (), 1f, 1f); }
  public synchronized void setHeight(float f) { scale(1f, f / getHeight(), 1f); }
  public synchronized void setDepth (float f) { scale(1f, 1f, f / getDepth ()); }
  public void setDims(float w, float h, float d) { scale(w / getWidth(), h / getHeight(), d / getDepth()); }
  
  public CoordSyst getCoordSyst() { return null; }
  public synchronized DimensionWrapper wrapper() {
    return new Box(new Point(minX, minY, minZ), new Point(maxX, maxY, maxZ));
  }
  public float getXFactor() { return 1f; }
  public float getYFactor() { return 1f; }
  public float getZFactor() { return 1f; }
  public void setXFactor(float f) {  }
  public void setYFactor(float f) {  }
  public void setZFactor(float f) {  }
  
  public synchronized void scale(float f) { scale(f, f, f); }
  public synchronized void scale(float x, float y, float z) {
    int pointIndex = 0;
    for(int i = 0; i < nbPoints; i++) {
      points[pointIndex] = points[pointIndex++] * x;
      points[pointIndex] = points[pointIndex++] * y;
      points[pointIndex] = points[pointIndex++] * z;
    }
    if(!shape.isLocked()) {
      buildDimension();
      isBuilt = false;
    }
  }

  // No event required for fragment.
  /** Not supported. See soya.soya3d.model.Fragment. */ public void fireResized() {  }
  /** Not supported. See soya.soya3d.model.Fragment. */ public void addPropertyChangeListener   (PropertyChangeListener listener) {  }
  /** Not supported. See soya.soya3d.model.Fragment. */ public void removePropertyChangeListener(PropertyChangeListener listener) {  }
  /** Not supported. See soya.soya3d.model.Fragment. */ public java.util.Collection propertyChangeListeners() { return null; }
  /** Not supported. See soya.soya3d.model.Fragment. */ public void firePropertyChange() {  }
  /** Not supported. See soya.soya3d.model.Fragment. */ public void firePropertyChange(String propertyName) {  }
  /** Not supported. See soya.soya3d.model.Fragment. */ public void firePropertyChange(String propertyName, Object oldValue, Object newValue) {  }
  /** Not supported. See soya.soya3d.model.Fragment. */ public void firePropertyChange(PropertyChangeEvent e) {  }
  /** Not supported. See soya.soya3d.model.Fragment. */ public void fireResize() {  }
  /** Not supported. See soya.soya3d.model.Fragment. */ public void fireResize(String propertyName) {  }
  /** Not supported. See soya.soya3d.model.Fragment. */ public void fireResize(String propertyName, Object oldValue, Object newValue) {  }

  protected void defineArrays(GLFunc gl, GLUFunc glu) {
    gl.glVertexPointer(3, GLEnum.GL_FLOAT, 0, points);
    if(usePointsNormals) gl.glNormalPointer(GLEnum.GL_FLOAT, 0, pointsNormals);
    if(useTexCoords) gl.glTexCoordPointer(2, GLEnum.GL_FLOAT, 0, pointsTexCoords);
    if(usePointsColors) {
      if(useAlpha) gl.glColorPointer(4, GLEnum.GL_FLOAT, 0, pointsColors);
      else gl.glColorPointer(3, GLEnum.GL_FLOAT, 16, pointsColors);
    }
    checkError(gl, glu);
  }
  protected void enableArrays(GLFunc gl, GLUFunc glu) {
    gl.glEnableClientState(GLEnum.GL_VERTEX_ARRAY);
    if(usePointsNormals) gl.glEnableClientState(GLEnum.GL_NORMAL_ARRAY);
    else gl.glDisableClientState(GLEnum.GL_NORMAL_ARRAY);
    if(useTexCoords) gl.glEnableClientState(GLEnum.GL_TEXTURE_COORD_ARRAY);
    else gl.glDisableClientState(GLEnum.GL_TEXTURE_COORD_ARRAY);
    if(usePointsColors) gl.glEnableClientState(GLEnum.GL_COLOR_ARRAY);
    else gl.glDisableClientState(GLEnum.GL_COLOR_ARRAY);
  }
  public void fillCollector(DrawablesCollector f, Renderer r, float[] mat) { // Optimizable.
    if(!isBuilt) build(r.getGLFunc(), r.getGLUFunc());
    if(mat[14] * mat[14] > -radiusSquarred) f.collect(this, mat);
  }
  public void draw(Renderer r, GLFunc gl, GLUFunc glu) {
    r.setDynamicLightsVisible(!staticLit);
    enableArrays(gl, glu);
    
    if(staticLit) {
      gl.glColorMaterial(GLEnum.GL_FRONT_AND_BACK, GLEnum.GL_EMISSION);
      material.makeColorCurrent(gl, glu);
      
      gl.glCallList(id);
      
      gl.glColor4f(0f, 0f, 0f, 1f);
      gl.glColorMaterial(GLEnum.GL_FRONT_AND_BACK, GLEnum.GL_AMBIENT_AND_DIFFUSE);
    }
    else gl.glCallList(id);
  }
  public void draw(Renderer r, GLFunc gl, GLUFunc glu, SpecialEffect[] specialEffects) {
    r.setDynamicLightsVisible(!staticLit);
    enableArrays(gl, glu);
    
    if(staticLit) {
      gl.glColorMaterial(GLEnum.GL_FRONT_AND_BACK, GLEnum.GL_EMISSION);
      material.makeColorCurrent(gl, glu);
      
      for(int i = 0; i < specialEffects.length; i++) specialEffects[i].  makeCurrent(gl, glu);
      gl.glCallList(id);
      for(int i = 0; i < specialEffects.length; i++) specialEffects[i].unmakeCurrent(gl, glu);
      
      gl.glColor4f(0f, 0f, 0f, 1f);
      gl.glColorMaterial(GLEnum.GL_FRONT_AND_BACK, GLEnum.GL_AMBIENT_AND_DIFFUSE);
    }
    else {
      for(int i = 0; i < specialEffects.length; i++) specialEffects[i].  makeCurrent(gl, glu);
      gl.glCallList(id);
      for(int i = 0; i < specialEffects.length; i++) specialEffects[i].unmakeCurrent(gl, glu);
    }
  }
  
  // PreTransformable :
  public synchronized void preTransform(float[] matrix) {
    int i, index1, index2;
    float x, y, z;
    index1 = 0; index2 = 0;
    for(i = 0; i < nbFaces; i++) {
      x = normals[index1++];
      y = normals[index1++];
      z = normals[index1++];
      normals[index2++] = x * matrix[0] + y * matrix[4] + z * matrix[ 8];
      normals[index2++] = x * matrix[1] + y * matrix[5] + z * matrix[ 9];
      normals[index2++] = x * matrix[2] + y * matrix[6] + z * matrix[10];
    }
    index1 = 0; index2 = 0;
    for(i = 0; i < nbPoints; i++) {
      x = points[index1++];
      y = points[index1++];
      z = points[index1++];
      points[index2++] = x * matrix[0] + y * matrix[4] + z * matrix[ 8] + matrix[12];
      points[index2++] = x * matrix[1] + y * matrix[5] + z * matrix[ 9] + matrix[13];
      points[index2++] = x * matrix[2] + y * matrix[6] + z * matrix[10] + matrix[14];
    }
    if(usePointsNormals) {
      index1 = 0; index2 = 0;
      for(i = 0; i < nbPoints; i++) {
        x = pointsNormals[index1++];
        y = pointsNormals[index1++];
        z = pointsNormals[index1++];
        pointsNormals[index2++] = x * matrix[0] + y * matrix[4] + z * matrix[ 8];
        pointsNormals[index2++] = x * matrix[1] + y * matrix[5] + z * matrix[ 9];
        pointsNormals[index2++] = x * matrix[2] + y * matrix[6] + z * matrix[10];
      }
    }
    if(matrix[0] * matrix[5] * matrix[10] < 0) returnAllFaces();
    if((shape != null) && (!shape.isLocked())) {
      buildDimension();
      isBuilt = false;
    }
  }
  public abstract void returnAllFaces();

  // Serializable :
  private void writeObject(ObjectOutputStream s) throws IOException {
    s.defaultWriteObject();
    material.write(s);
  }
  private void readObject(ObjectInputStream s) throws IOException, ClassNotFoundException {
    s.defaultReadObject();
    material = Material.read(s);
  }
  
  public void checkError(GLFunc gl, GLUFunc glu) {
    Soya.checkError(gl, glu, "error in a " + getClass().getName() + " of " + shape.getName() + " :");
  }
  public void checkError(GLFunc gl, GLUFunc glu, String message) {
    Soya.checkError(gl, glu, "error in a " + getClass().getName() + " of " + shape.getName() + " : " + message);
  }
}
