/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d.model;

import opale.soya.*;
import opale.soya.util.Matrix;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import opale.soya.soya3d.animation.*;
import gl4java.*;
import java.util.Iterator;
import java.io.*;
import java.beans.*;

abstract class AbstractOptimizedFragment extends AbstractFragment /* implements */ {
  private static final long serialVersionUID = -732971049175716669L;
  
  public AbstractOptimizedFragment() {  }
  public AbstractOptimizedFragment(AbstractFragment f) {
    shape            = f.shape           ;
    nbFaces          = f.nbFaces         ;
    nbPoints         = f.nbPoints        ;
    nbNonHiddenFaces = f.nbNonHiddenFaces;
    usePointsNormals = f.usePointsNormals;
    usePointsColors  = f.usePointsColors ;
    useTexCoords     = f.useTexCoords    ;
    smoothLit        = f.smoothLit       ;
    staticLit        = f.staticLit       ;
    
    normals = (float[]) f.normals.clone();
    points  = (float[]) f.points .clone();
    if(usePointsColors ) pointsColors    = (float[]) f.pointsColors   .clone();
    if(usePointsNormals) pointsNormals   = (float[]) f.pointsNormals  .clone();
    if(useTexCoords    ) pointsTexCoords = (float[]) f.pointsTexCoords.clone();
    material       = f.material      ;
    visibility     = f.visibility    ;
    radiusSquarred = f.radiusSquarred;
    minX = f.minX;
    minY = f.minY;
    minZ = f.minZ;
    maxX = f.maxX;
    maxY = f.maxY;
    maxZ = f.maxZ;
    
    buildIndexes();
    
    if(indexesPoints.length >= nbPoints * 3) {
      System.out.println("Don't worth to optimize. " + nbPoints);
      throw new UnsupportedOperationException("Don't worth to optimize");
    }
    else {
      System.out.println("Optimized OK. " + nbPoints + " => " + (((float) indexesPoints.length) / 3f));
    }
  }
  
  protected void buildIndexes() {
    indexes = new int[nbPoints];
                         indexesPoints          = new float[nbPoints * 3];
    if(usePointsNormals) indexesPointsNormals   = new float[nbPoints * 3];
    int nextIndex = 0;
    PointsIdentifier pi = new PointsIdentifier();
    for(int i = 0; i < nbPoints; i++) {
      PointRecord pr = pi.searchOrCreate(points[3 * i], points[3 * i + 1], points[3 * i + 2]);
      if(pr.index == -1) {
        pr.index = nextIndex++; // New.
        indexesPoints[pr.index * 3    ] = pr.x;
        indexesPoints[pr.index * 3 + 1] = pr.y;
        indexesPoints[pr.index * 3 + 2] = pr.z;
        if(usePointsNormals) {
          indexesPointsNormals[pr.index * 3    ] = pointsNormals[3 * i    ];
          indexesPointsNormals[pr.index * 3 + 1] = pointsNormals[3 * i + 1];
          indexesPointsNormals[pr.index * 3 + 2] = pointsNormals[3 * i + 2];
        }
      }
      indexes[i] = pr.index;
    }
    
    float[] indexesPoints2 = new float[nextIndex * 3];
    System.arraycopy(indexesPoints, 0, indexesPoints2, 0, indexesPoints2.length);
    indexesPoints = indexesPoints2;
    if(usePointsNormals) {
      float[] indexesPointsNormals2 = new float[nextIndex * 3];
      System.arraycopy(indexesPointsNormals, 0, indexesPointsNormals2, 0, indexesPointsNormals2.length);
      indexesPointsNormals = indexesPointsNormals2;
    }
  }
  
  public synchronized Object clone() {
    AbstractOptimizedFragment f = (AbstractOptimizedFragment) super.clone();
    f.indexes                  = (int  []) indexes               .clone();
    f.indexesPoints            = (float[]) indexesPoints         .clone();
    if(usePointsNormals)
      f.indexesPointsNormals   = (float[]) indexesPointsNormals  .clone();
    return f;
  }
  
  protected int  [] indexes;
  protected float[] indexesPoints;
  protected float[] indexesPointsNormals;
  
  /*
  public synchronized boolean canCombineWith(Fragment f) { return false; }
  public synchronized Fragment combineWith(Fragment frag) { return this; }
  */
  
  // Overrides :
  public synchronized void reBuild() {
    super.reBuild();
    if(!shape.isLocked()) buildIndexes();
  }
  protected void defineArrays(GLFunc gl, GLUFunc glu) throws GLErrorException {
    // Indexed :
    gl.glVertexPointer(3, GLEnum.GL_FLOAT, 0, indexesPoints);
    if(usePointsNormals) gl.glNormalPointer(GLEnum.GL_FLOAT, 0, indexesPointsNormals);
    // Other arrays are non-indexed and not pass as array.
    
    int error = gl.glGetError();
    if(error != 0) throw (RuntimeException) (new GLErrorException("Error while building call list :" + Integer.toString(error)));
  }
  protected void enableArrays(GLFunc gl, GLUFunc glu) {
    gl.glEnableClientState(GLEnum.GL_VERTEX_ARRAY);
    if(usePointsNormals) gl.glEnableClientState(GLEnum.GL_NORMAL_ARRAY);
    else gl.glDisableClientState(GLEnum.GL_NORMAL_ARRAY);
    gl.glDisableClientState(GLEnum.GL_TEXTURE_COORD_ARRAY);
    gl.glDisableClientState(GLEnum.GL_COLOR_ARRAY);
  }
  
  /*
  public synchronized String toString() {
    String s = getClass().getName() + "@ {\n" + Integer.toString(nbFaces) + " faces\n";
    s = s + nbPoints + " points\n";
    s = s + "Material : " + material.getName() + "\n";
    if(staticLit) s = s + "static lit\n";
    s = s + "Call list ID : " + Integer.toString(id) + "\n";
    int normalIndex = 0, pointIndex = 0, pointTexCoordIndex = 0, pointColorIndex = 0;
    for(int j = 0; j < nbFaces; j++) {
      s =  s + "Normal " + normals[normalIndex++] + ", " +  normals[normalIndex++] + ", " +  normals[normalIndex++] + "\n";
      for(int i = 0; i < getNumberOfPointsPerFace(); i++) {
        s =  s + "  Point " + points[indexes[pointIndex]] + ", " +  points[indexes[pointIndex] + 1] + ", " +  points[indexes[pointIndex] + 2] + "\n";
        if(usePointsNormals) s =  s + "  Normal " + pointsNormals[indexes[pointIndex]] + ", " +  pointsNormals[indexes[pointIndex] + 1] + ", " +  pointsNormals[indexes[pointIndex] + 2] + "\n";
        if(usePointsColors) s =  s + "  Color " + pointsColors[pointColorIndex++] + ", " +  pointsColors[pointColorIndex++] + ", " +  pointsColors[pointColorIndex++] + ", " +  pointsColors[pointColorIndex++] + "\n";
        if(useTexCoords) s =  s + "  Tex Coord " + pointsTexCoords[pointTexCoordIndex++] + ", " +  pointsTexCoords[pointTexCoordIndex++] + "\n";
        pointIndex++;
      }
    }
    s = s + "}\n";
    return s;
  }
  */
}
