/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d.model;

import opale.soya.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import gl4java.*;
import java.io.*;

/**
 * A fragment is a the lowest rendering entity. It can use only a single Material.
 * A one-fragment-shape is a shape that composed of a single fragment.
 * A fragmented shape is a shape that composed of a collection of fragments. fragmented shapes
 * are the most common concrete class of shape.
 * 
 * Fragments are manipulated internaly by one-fragment-shape and fragmented-shape, but they
 * are not intended to be use by a final user. They are designed for speed and performance,
 * contrary to shape element.
 * 
 * The following is written only for developpers who want to create their own fragments.
 * 
 * You cannot add directly a fragment into a shape. You must add a shape element : it is a
 * wrapper for all the properties of the fragment, many shape element can be regrouped into a
 * single fragment (eg two triangles with the same texture) for performance boost, or one
 * shape element can correspond to several fragment (eg if it use more than one material; a
 * fragment can have only one Material).
 *
 * A fragment class must have the following static method :
 *   public static Fragment[] fragmentsFromShapeElements(Shape, ShapeElement[]);
 * fragmentsFromShapeElements(Shape, ShapeElement[]) must return an array of fragments that
 * correspond to the given collection of shape elements. The given shape is the one that will
 * contain the fragment. The returned array should contains at least one fragment (except for
 * non graphical shape element), it can contains more than one if needed.
 * You can assume that all shape elements in the given collection are of the same class, and
 * are combinable (Two shape elements are considered as combinable if s1.canShareFragmentWith(s2)
 * return true).
 * 
 * Notice that no events are required for fragments, even the resize-event specified by the
 * Dimension interface.
 *
 * @author Artiste on the web
 */

public interface Fragment extends PreTransformable, Drawable, Dimension, Cloneable, Serializable {
  //public static Fragment[] fragmentsFromShapeElements(Shape newShape, FragmentedShapeElement[] ses);
  
  /**
   * Checks if this fragment is combinable with the given fragment.
   * @param f the other fragment
   * @return true if combinable
   */
  public boolean canCombineWith(Fragment f);
  /**
   * Combines this fragment with the given one. You should check before if it is possible with
   * boolean canCombineWith(Fragment) . The returned fragment can be this fragment, or a new
   * one.
   * @param f the other fragment
   * @return the combined fragment
   */
  public Fragment combineWith(Fragment f);

  /**
   * Gets the number of shape elements this frament draws.
   * @return the number of shape elements
   */
  public int getNumberOfShapeElements();
  /**
   * Computes and returns the shape elements that this frament draws.
   * @return the shape elements
   */
  public FragmentedShapeElement[] getFragmentedShapeElements();

  /**
   * Rebuilds this fragment after a change.
   */
  public void reBuild();

  /**
   * Sets this fragment's shape.
   * @param newShape the new shape
   */
  public void setShape(Shape newShape);
  /**
   * Gets this fragment's shape.
   * @return the shape
   */
  //public Shape getShape();

  public Object clone();

  /**
   * Computes and returns an optimized version of this fragment. It's may be the same fragment
   * if no optimization is possible.
   * @return the optimized fragment
   */
  public Fragment getOptimizedFragment();
}
