/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d.model;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import opale.soya.soya3d.model.*;
import gl4java.*;
import java.io.*;
import java.beans.*;
import java.util.Iterator;
import java.util.Map;
import java.util.HashMap;
import java.util.Arrays;
import java.lang.reflect.*;

public class FragmentedShape extends Shape implements opale.soya.util.Collection, SmoothLitBuilder /* , ConvertableToText */ {
  private static final long serialVersionUID = 3400801604056215188l;
  public FragmentedShape() { super(); }
  
  /**
   * Clones this fragmented shape. All its fragments are cloned. Warning : the lock state is
   * not cloned; the clone is always unlocked. If you clone a locked shape, the result can
   * be undefined.
   * @see opale.soya.soya3d.model.Shape#clone
   * @return the clone.
   */
  public synchronized Object clone() {
    FragmentedShape fs = (FragmentedShape) super.clone();
    int nb = fragments.length;
    fs.fragments = new Fragment[nb];
    Fragment[] frags = fs.fragments;
    for(int i = 0; i < nb; i++) frags[i] = (Fragment) fragments[i].clone();
    return fs;
  }
  
  // Serializable :
  /*
  private void readObject(ObjectInputStream s) throws IOException, ClassNotFoundException { // Initialisation
    s.defaultReadObject();
  }
  */
  
  public String toText() { // TODO
    return null;
  }

  public void optimize() {
    Fragment[] fs = new Fragment[fragments.length];
    for(int i = 0; i < fragments.length; i++) {
      fs[i] = fragments[i].getOptimizedFragment();
    }
    fragments = fs;
  }

  public synchronized void setMaterial(Material m) {
    Class[]  paramsClazz = { Material.class };
    Object[] params = { m };
    if(isElementsCreated()) {
      lock(); // Avoid sending a bunch of event. Also lock the elements().
      for(Iterator j = elements().iterator(); j.hasNext(); ) {
        FragmentedShapeElement se = (FragmentedShapeElement) j.next();
        try {
          Method me = se.getClass().getMethod("setMaterial", paramsClazz);
          me.invoke(se, params);
        }
        catch(Exception e) {  }
      }
      unlock();
      reBuild();
    }
    else {
      for(int j = 0; j < fragments.length; j++) {
        try {
          Method me = fragments[j].getClass().getMethod("setMaterial", paramsClazz);
          me.invoke(fragments[j], params);
        }
        catch(Exception e) {  }
      }
    }
    firePropertyChange("material");
  }
  public synchronized void setVisibility(int i) {
    if(isElementsCreated()) {
      lock(); // Avoid sending a bunch of event. Also lock the elements().
      for(Iterator j = elements().iterator(); j.hasNext(); ) {
        FragmentedShapeElement se = (FragmentedShapeElement) j.next();
        if(se instanceof FaceVisibility) ((FaceVisibility) se).setVisibility(i);
      }
      unlock();
      reBuild();
    }
    else {
      for(int j = 0; j < fragments.length; j++) {
        if(fragments[j] instanceof FaceVisibility) ((FaceVisibility) fragments[j]).setVisibility(i);
      }
    }
    firePropertyChange("visibility");
  }
  public synchronized void setSmoothLit(boolean b) {
    if(isElementsCreated()) {
      lock(); // Avoid sending a bunch of event. Also lock the elements().
      for(Iterator j = elements().iterator(); j.hasNext(); ) {
        FragmentedShapeElement se = (FragmentedShapeElement) j.next();
        if(se instanceof Lit) ((Lit) se).setSmoothLit(b);
      }
      unlock();
      reBuild();
    }
    else {
      for(int j = 0; j < fragments.length; j++) {
        if(fragments[j] instanceof Lit) ((Lit) fragments[j]).setSmoothLit(b);
      }
      buildSmoothLit();
    }
    firePropertyChange("smoothLit");
  }
  public void applyStaticLighting(Light3D[] lights, CoordSyst fromCoordSyst) {
    for(int j = 0; j < fragments.length; j++) {
      if(fragments[j] instanceof Lit) ((Lit) fragments[j]).applyStaticLighting(lights, fromCoordSyst);
    }
  }
  
  public synchronized void fillCollector(DrawablesCollector c, Renderer r, float[] mat) {
    for(int i = 0; i < fragments.length; i++) fragments[i].fillCollector(c, r, mat);
  }
  
  public String toString() {
    String s = getClass().getName() + "@ {\n" + Integer.toString(fragments.length) + " fragments\n";
    for(int i = 0; i < fragments.length; i++) s = s + fragments[i].toString();
    s = s + '}';
    return s;
  }
  
  // Fragments :
  private Fragment[] fragments = new Fragment[0];  // Always created.
  private synchronized void buildFragments() {  // From elements
    if(lockLevel > 0) return;
    fragments = null; // frees memory.
    
    Map classesMap = new HashMap(); // Sort the shape elements by fragment class.
    for(Iterator i = elements().iterator(); i.hasNext(); ) {
      FragmentedShapeElement se = (FragmentedShapeElement) i.next(); // Current ShapeElement
      Class clazz = se.getFragmentClass();
      java.util.Collection clazzBag = (java.util.Collection) classesMap.get(clazz);
      if(clazzBag == null) { // No other yet with this clazz.
        clazzBag = new java.util.Vector();
        classesMap.put(clazz, clazzBag);
      }
      clazzBag.add(se);
    }
    java.util.Collection classSorted = classesMap.values();
    
    java.util.Collection fullySorted = new java.util.Vector(); // Sort each sub group of shape elements by options.
    for(Iterator i = classSorted.iterator(); i.hasNext(); ) {
      java.util.Collection classBag = (java.util.Collection) i.next();
      
      java.util.Collection classOptionSorted = new java.util.Vector(classBag.size()); // Will contains all the shape elements of this class, sorted in sub collection by combinability.
      nextShapeElement:
      for (Iterator j = classBag.iterator(); j.hasNext(); ) {
        FragmentedShapeElement se = (FragmentedShapeElement) j.next();
        
        for(Iterator k = classOptionSorted.iterator(); k.hasNext(); ) {
          java.util.Collection classOptionBag = (java.util.Collection) k.next();
          FragmentedShapeElement oneSe = (FragmentedShapeElement) classOptionBag.iterator().next();
          if(se.canShareFragmentWith(oneSe)) {
            classOptionBag.add(se);
            continue nextShapeElement;
          }
        }
        // Not found => add a new group.
        java.util.Collection classOptionBag = new java.util.Vector();
        classOptionBag.add(se);
        classOptionSorted.add(classOptionBag);
      }
      fullySorted.addAll(classOptionSorted);
    }
    // Now, fullySorted contains collections that contain only combinable fragment (same class and combinable).
    
    java.util.Collection allFragments = new java.util.Vector();  // Create the fragments.
    FragmentedShapeElement[] shapeElement0Array = new FragmentedShapeElement[0];
    Class[] paramsClass = { Shape.class, shapeElement0Array.getClass() };
    for(Iterator i = fullySorted.iterator(); i.hasNext(); ) {
      java.util.Collection classOptionSorted = (java.util.Collection) i.next();
      FragmentedShapeElement oneSe = (FragmentedShapeElement) classOptionSorted.iterator().next();
      Class fragmentClass = oneSe.getFragmentClass();
      
      Fragment[] newFragments;
      try {
        Method build = fragmentClass.getMethod("fragmentsFromShapeElements", paramsClass); // This static method must be supported for all fragments classes.
        Object[] params = { this, (FragmentedShapeElement[]) classOptionSorted.toArray(shapeElement0Array) };
        newFragments = (Fragment[]) build.invoke(null, params);
      }
      catch(Exception e) { System.out.println("Can't create a fragment! = in class : " + fragmentClass.getName()); e.printStackTrace(); continue; }
      
      for(int j = 0; j < newFragments.length; j++) allFragments.add(newFragments[j]);
    }
    
    fragments = (Fragment[]) allFragments.toArray(new Fragment[0]);  // Create the fragments.
  }
  protected synchronized void buildDimension() {
    if(fragments.length == 0) {
      Arrays.fill(min , 0f);
      Arrays.fill(max , 0f);
      Arrays.fill(dims, 0f);
    }
    else {
      Position minp = new Point(Float.POSITIVE_INFINITY, Float.POSITIVE_INFINITY, Float.POSITIVE_INFINITY);
      Position maxp = new Point(Float.NEGATIVE_INFINITY, Float.NEGATIVE_INFINITY, Float.NEGATIVE_INFINITY);
      DimensionWrapper b;
      float f;
      for(int i = 0; i < fragments.length; i++) {
        b = fragments[i].wrapper();
        b.setCoordSyst(this); // Coordinates system conversion.
        b.minimize(minp);
        b.maximize(maxp);
        f = b.getSquareRadius();
        if(f > radiusSquarred) radiusSquarred = f;
      }
      min[0] = minp.getX();
      min[1] = minp.getY();
      min[2] = minp.getZ();
      max[0] = maxp.getX();
      max[1] = maxp.getY();
      max[2] = maxp.getZ();
      dims[0] = max[0] - min[0];
      dims[1] = max[1] - min[1];
      dims[2] = max[2] - min[2];
    }
    fireResize();
  }
  public synchronized void reBuild() { // Destroy and recreate all fragments.
    if(lockLevel > 0) return;
    if(isElementsCreated()) {
      lockElements(); // Avoid loosing elements() while building fragments
      buildFragments();
      unlockElements();
    }
    for(int i = 0; i < fragments.length; i++) fragments[i].reBuild();
    buildSmoothLit();
    buildDimension();
  }
  public synchronized Fragment[] toFragments() {
    Fragment[] fs = new Fragment[fragments.length];
    int j = 0;
    for(int i = 0; i < fragments.length; i++) {
      fs[j++] = (Fragment) fragments[i].clone();
    }
    return fs;
  }
  public void addAllFragments(Fragment[] fs) {
    releaseElements(); // Can be dangerous !!!!!!!!!
    
    for(int i = 0; i < fs.length; i++) {
      if(!tryCombine(fs[i])) {
        fs[i].setShape(this);
        Fragment[] newFragments = new Fragment[fragments.length + 1];
        System.arraycopy(fragments, 0, newFragments, 0, fragments.length);
        newFragments[newFragments.length - 1] = fs[i];
        fragments = newFragments;
      }
    }
    if(lockLevel <= 0) buildDimension();
  }
  public synchronized void addFragment(Fragment f) {
    releaseElements(); // Can be dangerous !!!!!!!!!
    
    if(!tryCombine(f)) {
      f.setShape(this);
      Fragment[] newFragments = new Fragment[fragments.length + 1];
      System.arraycopy(fragments, 0, newFragments, 0, fragments.length);
      newFragments[newFragments.length - 1] = f;
      fragments = newFragments;
    }
    if(lockLevel <= 0) buildDimension();
  }
  private synchronized boolean tryCombine(Fragment f) {
    for(int i = 0; i < fragments.length; i++) {
      if(fragments[i].canCombineWith(f)) {
        fragments[i] = fragments[i].combineWith(f);
        return true;
      }
    }
    return false;
  }
  
  // Collection :
  private transient java.util.Collection myElements = null;    // Not kept in memory for performance boost.
  private synchronized java.util.Collection elements() {  // Return the set; create it if necessary.
    if(myElements == null) myElements = buildElements();
    return myElements;
  }
  public synchronized void releaseElements() {
    if(isElementsLocked()) return;
    myElements = null;
    if(subEventListener != null) {
      subEventListener.destroy();
      subEventListener = null;
    }
  }
  public boolean isElementsCreated() { return myElements != null; }
  private synchronized java.util.Collection buildElements() { // Creation from fragments.
    if(isElementsLocked()) System.out.println("Impossible error occurs in Shape.buildElements");
    
    if(subEventListener != null) subEventListener.destroy();
    subEventListener = new SubEventListener();
    	
    int nbElements = 0;
    for(int i = 0; i < fragments.length; i++) nbElements = nbElements + fragments[i].getNumberOfShapeElements();
    java.util.Collection h = new java.util.Vector(nbElements);
    	
    for(int i = 0; i < fragments.length; i++) {
      FragmentedShapeElement[] elements = fragments[i].getFragmentedShapeElements();
      for(int j = 0; j < elements.length; j++) {
        elements[j].addPropertyChangeListener(subEventListener);
        h.add(elements[j]);
      }
    }
    return h;
  }
  public synchronized boolean add(FragmentedShapeElement e) {
    lockElements(); // Avoid loosing elements() while adding.
    boolean b = elements().add(e);
    reBuild();
    e.addPropertyChangeListener(subEventListener);
    if(b) fireAdd(e);
    unlockElements(); // Must be done after all use of e. (if elements is released, e will no longer be valid).
    return b;
  }
  public synchronized boolean remove(FragmentedShapeElement e) {
    lockElements(); // Avoid loosing elements() while removing.
    boolean b = elements().remove(e);
    reBuild();
    e.removePropertyChangeListener(subEventListener);
    if(b) fireRemove(e);
    unlockElements();
    return b;
  }
  public boolean add(Object e) { return add((FragmentedShapeElement) e); }
  public synchronized boolean addAll(java.util.Collection c) {
    lockElements(); // Avoid loosing elements() while adding.
    boolean b = elements().addAll(c);
    reBuild();
    if(b) { // Should be done in 2 for() {  } because all event must be OK before sending event. Else we risk loosing subevent sent by the receiver of the add event.
      for(Iterator i = c.iterator(); i.hasNext(); ) ((FragmentedShapeElement) i.next()).addPropertyChangeListener(subEventListener);
      for(Iterator i = c.iterator(); i.hasNext(); ) fireAdd((FragmentedShapeElement) i.next());
    }
    unlockElements();
    return b;
  }
  public synchronized void clear() {
    lockElements(); // Avoid loosing elements() while adding.
    java.util.Collection es = elements();
    java.util.Collection s = new java.util.Vector(es.size());
    s.addAll(es);
    es.clear();
    reBuild();
    subEventListener.destroy();
    subEventListener = new SubEventListener();
    for(Iterator i = s.iterator(); i.hasNext(); ) fireRemove(i.next());
    unlockElements();
  }
  public boolean contains(Object o) { return elements().contains(o); }
  public boolean containsAll(java.util.Collection c) { return elements().containsAll(c); }
  public boolean isEmpty() { return elements().isEmpty(); }
  public Iterator iterator() { return elements().iterator(); }
  public Iterator iterator(Class clazz) { return new Collection.ClassSpecificIterator(this, clazz); }
  public Iterator iteratorRecursive() { return new Collection.RecursiveIterator(this); }
  public Iterator iteratorRecursive(Class clazz) { return new Collection.ClassSpecificIterator(iteratorRecursive(), clazz); }

  public boolean remove(Object e) { return remove((FragmentedShapeElement) e); }
  public synchronized boolean removeAll(java.util.Collection c) {
    lockElements(); // Avoid loosing elements() while removing.
    boolean b = elements().removeAll(c);
    reBuild();
    if(b) {
      for(Iterator i = c.iterator(); i.hasNext(); ) ((FragmentedShapeElement) i.next()).removePropertyChangeListener(subEventListener);
      for(Iterator i = c.iterator(); i.hasNext(); ) fireRemove((FragmentedShapeElement) i.next());
    }
    unlockElements();
    return b;
  }
  public synchronized boolean retainAll(java.util.Collection c) {
    lockElements(); // Avoid loosing elements() while retaining.
    java.util.Collection s = new java.util.Vector(c.size());
    java.util.Collection es = elements();
    for(Iterator i = c.iterator(); i.hasNext(); ) {
      Object o = i.next();
      if(es.contains(o)) s.add(o);
    }
    boolean b = es.retainAll(c);
    reBuild();
    for(Iterator i = s.iterator(); i.hasNext(); )
      ((FragmentedShapeElement) i.next()).removePropertyChangeListener(subEventListener);
    for(Iterator i = s.iterator(); i.hasNext(); )
      fireRemove((FragmentedShapeElement) i.next());
    unlockElements();
    return b;
  }
  public int size() { return elements().size(); }
  public Object[] toArray() { return elements().toArray(); }
  public Object[] toArray(Object[] a) { return elements().toArray(a); }

  // Receive sub-Events :
  private class SubEventListener implements PropertyChangeListener {
    public void propertyChange(PropertyChangeEvent e) {
      if(alive) reBuild(); // will raise a changed event.
      else {
        ((FragmentedShapeElement) e.getSource()).removePropertyChangeListener(this);
      }
    }
    private boolean alive = true;
    public void destroy() { alive = false; }
  }
  private transient SubEventListener subEventListener = null;

  // Collection events :
  protected transient java.util.Collection collectionListeners;
  public java.util.Collection collectionListeners() { return collectionListeners; }
  public void addCollectionListener (CollectionListener l) {
    if(collectionListeners == null) collectionListeners = new java.util.Vector();
    synchronized(collectionListeners) { collectionListeners.add(l); }
  }
  public void removeCollectionListener (CollectionListener l) {
    if(collectionListeners != null) {
      synchronized(collectionListeners) { collectionListeners.remove(l); }
    }
  }
  public void fireAdd(Object element) {
    if(isWorthFiringCollectionEvent()) {
      AddEvent e = new AddEvent(this, element);
      int size;
      Object[] ls;
      synchronized (collectionListeners) {
        size = collectionListeners.size();
        ls = collectionListeners.toArray();
      }
      for (int i = 0; i < size; i++) ((CollectionListener) ls[i]).added(e);
    }
  }
  public void fireRemove(Object element) {
    if(isWorthFiringCollectionEvent()) {
      RemoveEvent e = new RemoveEvent(this, element);
      int size;
      Object[] ls;
      synchronized (collectionListeners) {
        size = collectionListeners.size();
        ls = collectionListeners.toArray();
      }
      for (int i = 0; i < size; i++) ((CollectionListener) ls[i]).removed(e);
    }
  }
  protected boolean isWorthFiringCollectionEvent() {
    return (collectionListeners != null) && (!collectionListeners.isEmpty());
  }
  
  // Dimension :
  public synchronized void scale(float f) {
    if(Float.isNaN(f)) f = 1f;
    if(isElementsCreated()) {
      lock();
      for(Iterator i = elements().iterator(); i.hasNext(); ) ((FragmentedShapeElement) i.next()).scale(f);
      unlock();
    }
    else for(int i = 0; i < fragments.length; i++) fragments[i].scale(f);
    buildDimension();
  }
  public synchronized void scale(float fx, float fy, float fz) {
    if(Float.isNaN(fx)) fx = 1f;
    if(Float.isNaN(fy)) fy = 1f;
    if(Float.isNaN(fz)) fz = 1f;
    if(isElementsCreated()) {
      lock();
      for(Iterator i = elements().iterator(); i.hasNext(); ) ((FragmentedShapeElement) i.next()).scale(fx, fy, fz);
      unlock();
    }
    else for(int i = 0; i < fragments.length; i++) fragments[i].scale(fx, fy, fz);
    buildDimension();
  }
  
  // Transformable :
  public synchronized void transform(float[] m) {
    if(isElementsCreated()) {
      lock();
      for(Iterator i = elements().iterator(); i.hasNext(); ) ((FragmentedShapeElement) i.next()).preTransform(m);
      unlock();
    }
    else for(int i = 0; i < fragments.length; i++) fragments[i].preTransform(m);
    buildDimension();
  }
  public void addVector(Vector v) {
    addVector(v.getX(), v.getY(), v.getZ());
  }
  public void addVector(float x, float y, float z) {
    transform(Matrix.matrixTranslate(x, y, z));
  }
  
  // Raypicking :
  public Position raypick(Position origin, Vector direction, int sens, int intersection) {
    Position best = null, current;
    double bestLenght = Double.POSITIVE_INFINITY, currentLenght;  // Squarred lenght used for speed boost.
    for(int i = 0; i < fragments.length; i++) {
      if(fragments[i] instanceof Raypicking) {
        current = ((Raypicking) fragments[i]).raypick(origin, direction, sens, intersection);
        if(current != null) {
          currentLenght = Matrix.pow2(origin.getX() - current.getX()) + Matrix.pow2(origin.getY() - current.getY()) + Matrix.pow2(origin.getZ() - current.getZ());
          if(currentLenght < bestLenght) {
            best = current;
            bestLenght = currentLenght;
          }
        }
      }
    }
    return best;
  }
  public int raypickNumber(Position origin, Vector direction, int sens, int intersection, float maxLenght) {
    int n = 0;
    for(int i = 0; i < fragments.length; i++) {
      if(fragments[i] instanceof Raypicking)
        n = n + ((Raypicking) fragments[i]).raypickNumber(origin, direction, sens, intersection, maxLenght);
    }
    return n;
  }
  public boolean raypickBoolean(Position origin, Vector direction, int sens, int intersection, float maxLenght) {
    for(int i = 0; i < fragments.length; i++) {
      if(fragments[i] instanceof Raypicking) {
        if(((Raypicking) fragments[i]).raypickBoolean(origin, direction, sens, intersection, maxLenght)) return true;
      }
    }
    return false;
  }


  // Lockable : override :
  //private transient int lockLevel;
  public boolean isLocked() { return lockLevel > 0; }
  public synchronized void lock() {
    if(isElementsCreated()) {
      lockElements();
    }
    lockLevel++;
  }
  public synchronized void unlock() {
    if(lockLevel > 0) {
      lockLevel--;
      if(lockLevel == 0) {
        reBuild();
      }
      if(isElementsCreated()) unlockElements(); // After rebuilding.
    }
  }

  private boolean isElementsLocked() { return false; }
  private synchronized void lockElements() {  }
  private synchronized void unlockElements() {  }
  /*
  // Allow to force elements() to be kept in memory (while fragments cannot be used to recover it).
  private transient int lockLevelElements;
  private transient Set keepElements; // Keep element in memory with a strong reference while locking because the fragments will be lost.
  private boolean isElementsLocked() { return lockLevelElements > 0; }
  private synchronized void lockElements() {
    if(lockLevelElements == 0) keepElements = elements();
    lockLevelElements++;
  }
  private synchronized void unlockElements() {
    if(lockLevelElements > 0) {
      lockLevelElements--;
      if(lockLevelElements == 0) {
        keepElements = null;
      }
    }
  }
  */
  
  public synchronized void buildSmoothLit() { buildSmoothLit(new PointsIdentifier()); }
  public synchronized void buildSmoothLit(PointsIdentifier pi) {
    for(int i = 0; i < fragments.length; i++)
      if(fragments[i] instanceof SmoothLitBuilder) ((SmoothLitBuilder) fragments[i]).buildSmoothLit(pi);
    pi.computeNormals();
  }
}
