/*
 * Soya3D tutorial
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
Lesson 4 : splitting cube
You'll learn how to scale an element and how to create complex elements. This lesson is a bit
harder and longer than the previous... :-<

The Dimension interface corresponds to object that have 3 dimensions and that can be scaled.
A 3D element that has dimensions is a PhysicalElement3D (an abstract class that extends the
DirectionnalElement3D); such an element has a position, an orientation and dimensions.
A Volume3D is an example.

If you need to perform frame-coordinate conversion with dimension, you will use a
DimensionPropertiesWrapper object. One example is the Box class, that assimilate any
physical element to a right-angled parallelepiped. This is the default wrapper, used
by the PhysicalElement3D class.
*/

package opale.soya.tutorial;

import opale.soya.*;              // contains the basic stuff (initialization,...).
import opale.soya.awt.*;          // contains the GUI stuff (RenderingCanvas, RenderingFrame,...).
import opale.soya.soya2d.*;       // contains the 2D stuff (Texturing and Material).
import opale.soya.soya3d.*;       // contains the 3D stuff (A lot of things).
import opale.soya.soya3d.model.*; // contains the modeling stuff (Shape,...).

import java.util.Iterator; // contains java stuff...

/*
The Lesson4 is a rendering frame.
*/
public class Lesson4 extends RenderingFrame {
  public static void main(String[] args) { (new Lesson4()).setVisible(true); }
  
  public Lesson4() {
    super(320, 200);
    
    Soya.init(this);
    Shape.path = getClass().getResource("").getFile();
    
    scene = new Environment3D();
    scene.setAmbientColor(.5f, .5f, .5f);
    
    /*
    Create a splitting cube, recover its class (for a future use), and add it into the scene.
    */
    SplittingCube3D cube = new SplittingCube3D();
    splittingCubeClass = cube.getClass();
    scene.add(cube);
    cube.lookAt(new Point(0f, 0f, 1f, scene));
    
    Light3D light = new Light3D();
    scene.add(light);
    light.move(0f, 0f, 4f);
    light.setAttenuation(0f, .3f, 0f);
    
    Camera3D camera = new Camera3D();
    scene.add(camera);
    camera.move(0f, -1f, 5f);
    
    this.setRenderer(camera);
    
    /*
    Create a thread in order to animate the cube and render the view.
    */
    animation = new Thread(new Runnable() {
      public void run() {
        while(true) {
          Object[] elements = scene.toArray();
          for(int i = 0; i < elements.length; i++)
            if(elements[i] instanceof SplittingCube3D) ((SplittingCube3D) elements[i]).play();
            render();
            try { Thread.currentThread().sleep(50); }
            catch(Exception e) { e.printStackTrace(); }
          }
        }
      }
    });
    animation.start();
  }
  private Environment3D scene;
  private Thread animation;
  private Class splittingCubeClass;

  /*
  A splitting cube is a volume that moves randomly, grows, and then split into 2 smaller cubes.
  */
  private static class SplittingCube3D extends Volume3D {
    /*
    Create a new splitting cube with a default shape : a red cube.
    */
    public SplittingCube3D() { super(getMyShape()); }
    /*
    Return the default shape.
    */
    private static Shape getMyShape() {
      try { return Shape.get("test"); }
      catch(Exception e) {
        System.out.println("Can't find Shape : cube");
        e.printStackTrace();
        return null;
      }
    }

    /*
    Some constants.
    */
    public static final float MOVING_SPEED = 0.05f;    // length
    public static final float GROWING_SPEED = 0.01f;   // length
    public static final float SPLITTING_SPEED = 0.05f; // length
    public static final float ROTATING_SPEED = 20f;    // degrees
    public static final float VOLUME_MAX = 2;          // volume

    /*
    Calculs the volume of the cube.
    */
    public float getVolume() {
      /*
      Get all the dimensions of the volume (a splitting cube is a volume),
      and multiply them. WARNING: you should not modify the array returned by getDims().
      */
      float[] dims = getDims();
      return dims[0] * dims[1] * dims[2];
    }
    public void setVolume(float v) {
      /*
      Calculs the ratio between the new volume and the old one.
      */
      float f = v / getVolume();
      /*
      Scale the volume. You can scale with different coeficients for each direction
      with scale(float, float, float).
      */
      scale(f);
    }

    /*
    Constants for states : a cube starts in the growing state, and when it is enought big,
    it change to the splitting state ; and field and getter for the current state of the cube.
    */
    public static final int STATE_GROWING = 0;
    public static final int STATE_SPLITTING = 1;
    private int state = STATE_GROWING;
    public int getState() { return state; }

    /*
    The play method is called every frame. It moves/rotates/splits/... the cube.
    */
    public void play() {
      /*
      Grows the cube or split it, according to its state.
      */
      if(state == STATE_GROWING) grow();
      else split();
    }
    private void grow() {
      /*
      Advance the cube in its front direction.
      */
      addVector(new Vector(0f, 0f, -MOVING_SPEED, this));
      /*
      Rotate randomly the cube.
      */
      rotateLateral ((((float) java.lang.Math.random()) -0.5f) * ROTATING_SPEED);
      rotateVertical((((float) java.lang.Math.random()) -0.5f) * ROTATING_SPEED);

      /*
      Get the volume of the cube. If it is enought big, make it split !!
      Else, it grow bigger. :-)
      */
      float v = getVolume();
      if(v >= VOLUME_MAX) startSplitting();
      else setVolume(getVolume() + GROWING_SPEED);
    }
    private void split() {
      /*
      Get the width of the volume (Similarly, you can use getHeight() and getDepth()
      for other dimension). If the cube is enought stretched, the split is ended, else
      it is stretched again in the X direction.
      */
      float w = getWidth();
      if(w >= VOLUME_MAX) endSplitting();
      else {
        /*
        Get the volume of the cube. When stretching, the volume of the cube is maintened
        constant, so the width is increased and the other dimensions are decreased.
        setWidth(), setHeight() and setDepth() are the setter used for defining a dimension.
        */
        float v = getVolume();
        w = w + SPLITTING_SPEED;
        setWidth(w);
        float d = (float) java.lang.Math.sqrt((double) (v / w));
        setHeight(d);
        setDepth(d);
      }
    }

    private void startSplitting() {
      /*
      Change the state into splitting mode.
      */
      state = STATE_SPLITTING;
    }
    /*
    Ends the split and create child cubes.
    */
    private void endSplitting() {
      /*
      Get the parent world.
      */
      World3D parent = getParent();
      /*
      Forget this line.
      */
      //Vector motherFront = new Vector(0f, 0f, -1f, this);

      /*
      Create the left child cube.
      */
      SplittingCube3D leftChild  = new SplittingCube3D();
      /*
      Add it into the parent world.
      */
      parent.add(leftChild);
      /*
      Move it at the left side of the mother cube (the mother should have those
      dimension: 2 * 1 * 1 when it finish splitting). The position is -0.5, 0, 0 defined
      in mother frame-coordinates. An automatic frame-coordinate conversion is performed
      by soya (from this to parent).
      */
      //leftChild.move(this);
      //leftChild.lookAt((new Point(0f, 0f, -1f, this)));
      leftChild.move(new Point(-0.25f, 0f, 0f, this));
      /*
      Rotate the child in order to, when it will be moved (by the grow() method), it will
      go away from the mother position. The lookAt() orientate the child into the left
      direction of its mother.
      */ 
      leftChild.lookAt(new Point(-1.25f, 0f, 0f, this));
      
      /*
      Equivalent work for the right child... No comment! :->
      */
      SplittingCube3D rightChild = new SplittingCube3D();
      parent.add(rightChild);
      rightChild.move(new Point(0.25f, 0f, 0f, this));
      rightChild.lookAt(new Point(1.25f, 0f, 0f, this));

      /*
      Remove the mother.
      */
      remove();
    }
  }
}
