/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.util;

import java.io.*;
import java.lang.reflect.*;
import java.beans.*;

/**
 * Abstract implementation of a bean object, that can fire property chang event. Even if i
 * use this class as a "bean class", you should never consider that all beans extends this
 * class !
 * 
 * Notice than the listeners collection is not serialized.
 * 
 * @author Artiste on the Web
 */

public abstract class AbstractBean implements Bean {
  /**
   * Clones this bean. Warning : the listeners are not cloned. The clone has no listener.
   * @return a clone
   */
  public abstract Object clone();
  
  protected transient java.util.Collection changeListeners;
  /**
   * Gets all the listeners for property change event.
   * @return the collection of listener.
   */
  public java.util.Collection propertyChangeListeners() { return changeListeners; }
  
  /**
   * Adds a listener for property change event.
   * @param listener the listener.
   */
  public void addPropertyChangeListener   (PropertyChangeListener l) {
    if(changeListeners == null) changeListeners = new java.util.Vector();
    changeListeners.add(l);
  }
  /**
   * Removes a listener for property change event.
   * @param listener the listener.
   */
  public void removePropertyChangeListener(PropertyChangeListener l) {
    if(changeListeners != null) {
      changeListeners.remove(l);
    }
  }
  
  /**
   * Checks if it is really necessary to fire an event. This method returns false if there is
   * no listener, else it returns true. You can override it to add your own condition.
   * This method is called by firePropertyChange(), firePropertyChange(String) and
   * firePropertyChange(String, Object, Object), but not by
   * firePropertyChange(PropertyChangeEvent) (in order that you can check it yourself before
   * creating the event object).
   * @return true if firing an event if necessary
   */
  protected boolean isWorthFiringEvent() {
    return (changeListeners != null) && (!changeListeners.isEmpty());
  }
  
  /**
   * Fires a property change event on this bean, with no other information.
   */
  public void firePropertyChange() {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeEvent(this, "", null, null));
  }
  /**
   * Fires a property change event on this bean, for the given property.
   * @param propertyName the name of the changed property
   */
  public void firePropertyChange(String propertyName) {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeEvent(this, propertyName, null, null));
  }
  /**
   * Fires a property change event on this bean, for the given property.
   * @param propertyName the name of the property that have changed.
   * @param oldValue the old value of the property, or null.
   * @param newValue the new value of the property, or null.
   */
  public void firePropertyChange(String propertyName, Object oldValue, Object newValue) {
    if(oldValue != null && oldValue.equals(newValue)) return;
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeEvent(this, propertyName, oldValue, newValue));
  }
  /**
   * Fires the given property change event on this bean.
   * Warning : this method do not check if the event if usefull. You can check it before with
   * isWorthFiringEvent() .
   * @param e the property change event that will be fired
   */
  protected void firePropertyChange(PropertyChangeEvent e) {
    int size;
    Object[] ls;
    synchronized (changeListeners) {
      size = changeListeners.size();
      ls = changeListeners.toArray();
    }
    for (int i = 0; i < size; i++) ((PropertyChangeListener) ls[i]).propertyChange(e);
  }
  
  /**
   * Tries to add the given property change listener to the given bean. This can only work
   * if the bean has a "addPropertyChangeListener" method.
   * @return true if success
   */
  public static boolean addPropertyChangeListenerToBean(Object bean, PropertyChangeListener l) {
    Class[] paramsClass = { PropertyChangeListener.class };
    Method registerListener;
    Object[] params = { l };
    try {
      registerListener = bean.getClass().getMethod("addPropertyChangeListener", paramsClass);
      registerListener.invoke(bean, params);
      return true;
    }
    catch(Exception e) {
      return false;
    }
  }
  /**
   * Tries to remove the given property change listener to the given bean. This can only work
   * if the bean has a "removePropertyChangeListener" method.
   * @return true if success
   */
  public static boolean removePropertyChangeListenerToBean(Object bean, PropertyChangeListener l) {
    Class[] paramsClass = { PropertyChangeListener.class };
    Method registerListener;
    Object[] params = { l };
    try {
      registerListener = bean.getClass().getMethod("removePropertyChangeListener", paramsClass);
      registerListener.invoke(bean, params);
      return true;
    }
    catch(Exception e) {
      return false;
    }
  }
}

