/*
Copyright (C) 2000  Groupe Opale (http://www.opale.ovh.org)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

You can visit the web site http://www.opale.ovh.org to obtain more informations about this program and/or to contact the coders.
*/
package opale.ode.ker;

import opale.mathtools.*;
import opale.tools.*;
import java.io.*;

/**
* Classe qui modlise un problme d'quations diffrentielles ordinaires. Une instance de cette classe doit tre associ (sens UML)  une instance de TimeScheme (un schma en temps) et une instance de Equation (une quation) afin de pouvoir appeler la mthode solve() charge de rsoudre le problme ainsi form.
* @since Opale-ODE 0.1
* @author O.C.
*/

final public class Problem extends ObjectODE
{
private TimeScheme sch;
private Equation eqn;
private DVect cdinit;
private DVect[] inco;
private double[] times;
private int dim;
private boolean issolve = false;

/**
* Constructeur par dfaut. Initialise un problme de dimension 1.
*/
public Problem()
	{
	dim = 1;
	cdinit = new DVect(dim);
	}	

/**
* Constructeur pour initialiser un problme avec une dimension donne.
* @param int dim, la dimension du problme.
*/
public Problem(int dim)
	{
	this.dim = Math.max(dim,1);
	cdinit = new DVect(this.dim);
	}	

	
/**
* Methode principale de la classe. Elle est charge de lancer la rsolution du problme d'ODE aprs avoir vrifi que tous les lments sont bien prsents (schma en temps, quation...).
*/
public void solve()
	{
	issolve = false;	System.err.println("\n**************************************************");
	System.err.println("************ Rsolution d'un problme ************");
	System.err.println("**************************************************");
	//Vrification des composants du problme.
	System.err.println("*** Vrification des composants du problme ***");
	
	if (eqn == null)
		{
		System.err.println("Vous n'avez pas associ d'quation au problme.\nArret anormal du calcul.");
		System.exit(-1);
		}
	if ( dim!= eqn.dim())
		{
		System.err.println("L'quation et le problme n'ont pas la mme dimension.\nArret anormal.");
		System.exit(-1);
		}
		
	System.err.println(" --> Equation OK");

	if (sch == null)
		{
		System.err.println("Vous n'avez pas associ de schma en temps au problme.\nArret anormal du calcul.");
		System.exit(-1);
		}
	System.err.println(" --> Schma utilis : "+sch);
		
	
	int i;
	System.err.println(" --> Conditions initiales : ");
		for (i=0;i<dim;i++)
			System.err.println("\t"+cdinit.get(i));
		
	System.err.println("*** Problme OK ***\n");
	
	
	//Rservation mmoire pour l'inconnue
	
	System.err.println("*** Rservation mmoire pour l'inconnu ***\n");
	inco = new DVect[sch.getNstep()];
	for (i=0;i<sch.getNstep();i++)
		inco[i] = new DVect(dim);
	times = new double[sch.getNstep()];	

	System.err.println("*** Initialisation du schma en temps ***\n");
	sch.init();
	inco[0].set(sch.present());
	times[0] = sch.tmin();
	
	System.err.println("*** Dbut du calcul ***");

	
	Chronometer.start();
		
	for(i=1;(i<sch.getNstep()) && (sch.time() < sch.tmax());i++)
		{
		sch.forward(eqn);
		sch.update();
		inco[i].set(sch.present());
		times[i] = sch.time();
		//System.err.println("t = "+sch.time()+"\n"+sch.present());
		}
	Chronometer.stop();
	System.err.println("*** Fin du calcul ***");
	System.err.println("*** Temps pass pour ce calcul : "+Chronometer.time()+" ms\n");
	System.err.println("**********************************************");
	System.err.println("************ Fin de la rsolution ************");
	System.err.println("**********************************************\n");
	issolve = true;
	}

/**
* Renvoie la solution calcule sous forme de tableau dans une chaine String : la premire colonne reprsente le temps, puis les suivantes les inconnues.
* @return String, la chaine contenant la solution.
*/
public String printSol()
	{
	StringBuffer sol=new StringBuffer();
	if (issolve)
	{
	int i,j;
	for (i=0;i<sch.getNstep();i++)
		{
		sol.append( times[i]+" ");
		for (j=0;j<dim;j++)
			sol.append(inco[i].get(j)+" "); 
		sol.append("\n");
		}
	}
	else sol.append("Problme pas encore rsolu\n");
	return sol.toString();	
	}
	
/**
* Renvoie la solution dans un tableau  deux entres t[i][j]  0<= i <= dim ; 0 <= j <= N  o N est le nombre de pas de temps calcul : la composante t[0][j] correspond en fait aux pas de temps du calcul et les composantes suivantes t[i>=1][j] aux composantes  proprement parl de la solution.
* @return double[][], le tableau contenant la solution, null si le problme est non rsolu.
*/
public double[][] toTab()
	{
	if (issolve)
	{
	int i,j;
	double[][] tab = new double[dim+1][sch.getNstep()];
	for (i=0;i<sch.getNstep();i++)
		{
		tab[0][i]=times[i];
		for (j=1;j<=dim;j++)
			tab[j][i] = inco[i].get(j-1); 
		}
	return tab;	
	}
	return null;
	}
	
public void writeSol(WFile wf)
	{
	wf.writeString(printSol());
	}
	
/**
* Renvoie la dimension du problme.
* @return int, la dimension.
*/
public int dim()
	{
	return dim;
	}

/**
* Fixe un vecteur pour la condition initiale. Ce vecteur doit avoir la dimension du problme pour taille.
* @param DVect, le vecteur des conditions initiales.
*/	
public void setInit(DVect init)
	{
	cdinit.set(init);
	}

/**
* Renvoie le vecteur des conidtions initiales.
* @return DVect, le vecteur des conditions initiales.
*/
public DVect getInit()
	{
	return cdinit;
	}

/**
* Associe une quation au problme. La mthode vrifie aussi que l'quation est compatible avec le problme (mme dimension).
* @param Equation, l'quation.
*/
public void setEqn(Equation eq)
	{
	if ( dim!= eq.dim())
		{
		System.err.println("L'quation et le problme n'ont pas la mme dimension.\nArret anormal.");
		System.exit(-1);
		}
	eqn = eq;
	}
/**
* REnvoie l'quation associe au problme.
* @return Equation, l'quation.
*/
public Equation getEqn()
	{
	return eqn;
	}
	
/**
* Associe un schma en temps au problme. 
* @param TimeScheme, le schma en temps.
*/
public void setTS(TimeScheme sch)
	{
	this.sch = sch;
	sch.setPb(this);
	}		
/**
* Renvoie le schma en temps associ au problme. 
* @return TimeScheme, le schma en temps.
*/
public TimeScheme getTS()
	{
	return sch;
	}		

public int readKeyWord(String word,RFile f,ODE ode)  throws java.io.IOException,MalformedFileException
	{
	int type,i;
	if ( word.equals("dim"))
		{
		type = f.nextToken();
		if (type != StreamTokenizer.TT_NUMBER) return -1;
		this.dim = (int) f.nval;
		cdinit = new DVect(this.dim);
		return 0;
		}
	else if (word.equals("y0"))
		{
		for(i=0;i<dim;i++)
			{
			type = f.nextToken();
			if (type != StreamTokenizer.TT_NUMBER) return -1;
			cdinit.set(i,f.nval);
			}
		return 0;
		}
		
	else if ( word.equals("sch"))
		{
		type = f.nextToken();
		if (type != StreamTokenizer.TT_WORD) throw new MalformedFileException("Fichier de donnes au mauvais format : erreur  la lecture d'un schma en temps attach  un problme");
		if ( (!ode.contains(f.sval)) || (ode.fromId(f.sval)) == null) throw new MalformedFileException("Fichier de donnes au mauvais format : l'identificateur "+f.sval+" n'existe pas !!");
		if (!(ode.fromId(f.sval) instanceof TimeScheme) ) throw new MalformedFileException("Fichier de donnes au mauvais format : l'identificateur "+f.sval+" ne correspond pas  un schma en temps !!");
		setTS((TimeScheme) ode.fromId(f.sval));
		return 0;
		}
	else if ( word.equals("eqn"))
		{
		type = f.nextToken();
		if (type != StreamTokenizer.TT_WORD) throw new MalformedFileException("Fichier de donnes au mauvais format : erreur  la lecture d'une quation attache  un problme");
		if ( (!ode.contains(f.sval)) || (ode.fromId(f.sval)) == null) throw new MalformedFileException("Fichier de donnes au mauvais format : l'identificateur "+f.sval+" n'existe pas !!");
		if (!(ode.fromId(f.sval) instanceof Equation) ) throw new MalformedFileException("Fichier de donnes au mauvais format : l'identificateur "+f.sval+" ne correspond pas  une quation !!");
		setEqn((Equation) ode.fromId(f.sval));
		return 0;
		}
		
	return -1;
	}
public void writeFile(WFile f, ODE ode) throws MalformedFileException
	{
	f.writeln("{");
	if (!ode.contains(sch)) throw new MalformedFileException("Erreur  l'criture du fichier : le schma en temps "+sch+" n'existe pas !!");
	if (!(eqn instanceof ObjectODE))  throw new MalformedFileException("Erreur  l'criture du fichier : l'equation "+eqn+" de type "+eqn.getClass()+" n'est pas inscriptible dans un fichier !!");
	if (!ode.contains((ObjectODE) eqn)) throw new MalformedFileException("Erreur  l'criture du fichier : l'equation "+eqn+" n'existe pas !!");
	f.writeln("dim "+dim);
	f.writeString("y0");
	int i;
	for (i=0;i<dim;i++)
		f.writeString(" "+cdinit.get(i));
	f.writeln("");
	f.writeln("sch "+ode.getId(sch));
	f.writeln("eqn "+ode.getId((SEquation) eqn));
	
	f.writeln("}");
	}

public String toString()
	{
	return "Problem of ODE";
	}



}


	
