/*
Copyright (C) 2000  Groupe Opale

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

You can visit the web site http://www.opale.ovh.org to obtain more informations about this program and/or to contact the coders.
*/


package opale.m2d;
import opale.tools.*;
import java.io.*;

/**
* This class represents a vector in R.
* @author O.C.
* @since Opale-2d 0.1
*/

public class Vector2D extends Object2D implements Transformable, Cloneable
{

// membres protected ou public ??!!
/**
* public members.
* The coordinates of the vector
*/
public double x,y;	// les composantes du vecteur

/**
* Constructor by default.
*/
public Vector2D()
	{
	super();
	if (Debug.On) Debug.print("ici dan Vector2D");
	}


/**
* Constucts and intializes the vector at (0; 0) with a OIJ specified.
* @param OIJ rep, the basis
*/
public Vector2D(OIJ rep)
	{
	super(rep);
	}

/**
* Constucts and intializes the vector at (x; y) with a OIJ specified.
* @param OIJ rep, the basis
* @param double x,y, coords of the vector.
*/
public Vector2D(OIJ rep,double x, double y)
	{
	super(rep);
	this.x = x;
	this.y = y;
	}

/**
* Constructs and initializes a vector with the same location as the specified <code>Vector2D</code> object.
* @param Vector2D v, a vector.
*/
public Vector2D(Vector2D v)
	{
	super(v.getOIJ());
	this.x = v.x;
	this.y = v.y;
	}

/**
* Constructs a vector from two objects <code>Point2D</code>.
* @param Point2D p, the origin
* @param Point2D q, the end
*/
public Vector2D(Point2D p, Point2D q)
	{
	super(p.getOIJ());
	Point2D qq = (Point2D) q.clone();
	qq.changeOIJ(p.getOIJ());
	
	this.x = qq.x -p.x;
	this.y = qq.y - p.y;
	}

//Les methodes publiques

/**
* Returns the X coordinate of the vector.
* @return double, the X coordinate of this vector.
*/
public final double getX()
	{
	return x;
	}

/**
* Returns the Y coordinate of the vector.
* @return double, the Y coordinate of this vector.
*/
public final double getY()
	{
	return y;
	}

/**
* Sets the X coordinate of this vector to the specified double coordinate.
* @param double x, the new X coordinate.
*/
public final void setX(double x)
	{
	this.x = x;
	}

/**
* Sets the Y coordinate of this vector to the specified double coordinate.
* @param double y, the new Y coordinate.
*/
public final void setY(double y)
	{
	this.y = y;
	}



/**
* Sets the coordinates of this vector.
* @param x, y, the coordinates of the vector.
*/
public final void set(double x, double y)
	{
	this.x = x;
	this.y = y;
	}

/**
* Sets the coordinates of the  vector to the specified vector.
* @param Vector2D v, the specified vector.
*/
public final void set(Vector2D v)
	{
	Vector2D vv = (Vector2D) v.clone();
	vv.changeOIJ(getOIJ());
	
	set(vv.x,vv.y);
	}




/**
* Computes the square of the norm of this vector.
* @return double, the square of the norm.
*/
public final double normSq()
	{
	return getOIJ().normSq(x,y);
	}

/**
* Computes the norm of this vector.
* @return double, the norm.
*/
public final double norm()
	{
	return Math.sqrt(normSq());
	}

/**
* Computes the square of the norm of a vector (x ; y).
* @param double x,y, the coordinates of the vector.
* @return double, the square of the norm.
*/
static final public double normSq(double x, double y)
	{
	return x*x + y*y;
	}

/**
* Computes the norm of a vector (x ; y).
* @param double x,y, the coordinates of the vector.
* @return double, the norm.
*/
static public double norm(double x, double y)
	{
	return Math.sqrt(normSq(x,y));
	}

/**
* Computes the square of the norm of a vector v.
* @param <code>Vector2D</code> v, the vector.
* @return double, the square of the norm.
*/
static public double normSq(Vector2D v)
	{
	return v.getOIJ().normSq(v.x,v.y);
	}

/**
* Computes  the norm of a vector v.
* @param <code>Vector2D</code> v, the vector.
* @return double, the norm.
*/
static public double norm(Vector2D v)
	{
	return Math.sqrt(normSq(v));
	}





/**
* Determines whether two .vectors are equal.
* @param Object obj, an object to be compared with this vector.
* @return <code>true</code> the object to be compared is an instance of <code>Vector2D</code> and has the same value; <code>false</code> otherwise.
*/
public boolean equals(Object obj)
	{
	if (obj instanceof Vector2D)
		{
        	Vector2D v = (Vector2D) obj;
		return (x == v.x) && (y == v.y) && (getOIJ().equals(v.getOIJ()));
        	}
    	return false;
    }

/**
* Creates a new object of the same class and with the same contents as this vector.
* @return Object, a clone of this instance.
* @exception  OutOfMemoryError if there is not enough memory.
* @see        java.lang.Cloneable
*/
public Object clone()
	{
	Vector2D v = (Vector2D) super.clone();
	v.x = x;
	v.y = y;
	return v;
	}



//les mthodes de l'interface Transformable
public void translate(double dx,double dy) {}


	
public void rotate(double theta)
	{
	Point2D p = new Point2D(getOIJ(),x,y);
	p.rotate(theta);
	set(p.x,p.y);
	}
	
public void rotate(double x, double y,double theta)
	{
	this.translate(-x,-y);
	this.rotate(theta);
	this.translate(x,y);

	}

public void transform(AffineTransform2D t)
	{
	double[] tab = {0,0,x,y};
	double[] dest = new double[4];
	t.transform(tab,0,dest,0,2);
	this.x = dest[2]-dest[0];
	this.y = dest[3]-dest[1];
	}

/**
* Effectue un changement de repere du vecteur.
* @param OIJ rep, le nouveau repere.
*/
public void changeOIJ(OIJ rep)
	{
	Matrix2D mv = new Matrix2D();
	
	getOIJ().matPassage(rep,mv);
	
	double[] q = mv.compute(x,y);
	double[] p = mv.compute(0,0);

	set(q[0]-p[0],q[1]-p[1]);	
	setOIJ(rep);
	}
	
public String toString()
	{
	String s="("+this.x+"; "+this.y+")";
	return s;
	}

public void writeMore(PrintWriter f, OpaleSet p)
	{
	f.println("X " +x);
	f.println("Y " +y);
	}	
public int readKeyWord(String word, StreamTokenizer f, OpaleSet p)  throws java.io.IOException, InvalidFormatException
	{
	int type;
	if ( word.equals("X"))
		{
		type = f.nextToken();
		if (type != StreamTokenizer.TT_NUMBER) return -1;
		x = f.nval;
		return 0;
		}
	else if ( word.equals("Y"))
		{
		type = f.nextToken();
		if (type != StreamTokenizer.TT_NUMBER) return -1;
		y = f.nval;
		return 0;
		}
	return -1;
	}
}
