/*
Copyright (C) 2000  Groupe Opale (http://www.opale.ovh.org)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

You can visit the web site http://www.opale.ovh.org to obtain more informations about this program and/or to contact the coders.
*/


package opale.mathtools;
import opale.tools.Assert;
	
/**
* Cette classe modlise un vecteur de double et fournit des mthodes de calculs sur ce vecteur.
* @since Opale-Mathtools 0.1
* @author O.C.
*/
public final class DVect implements Cloneable
{
private double[] x;
private int n;

/**
* Constructeur par dfaut qui prend en argument le nombre de composantes et initialise toutes les composantes  0.
* @param int n, le nombre de composantes.
* @since Opale-Mathtools 0.11
*/
public DVect(int n)
	{
	x = new double[n];
	this.n = n;
	for(n=0;n<this.n;n++)
		x[n] =0 ;
	}

/**
* Constructeur qui prend en argument le nombre de composantes et les valeurs.
* @param int n, le nombre de composantes.
* @param double[] v, le tableau des valeurs.
*/
public DVect(int n, double[] v)
	{
	x = new double[n];
	this.n = n;
	for(n=0;n<this.n;n++)
		x[n] = v[n];
	}
	
/**
* Constructeur qui prend en argument un tableau de valeurs.
* @param double[] v, le tableau des valeurs.
* @since Opale-Mathtools 0.13
*/
public DVect(double[] v)
	{
	int i;
	this.n = v.length;
	x = new double[n];
	for(i=0;i<this.n;i++)
		x[i] = v[i];
	}
	
/**
* Constructeur de copie.
* @param DVect v, le vecteur a copier.
*/
public DVect(DVect v)
	{
	int i;
	
	this.n = v.n;
	x = new double[n];
	for(i=0;i<this.n;i++)
		x[i] = v.x[i];
	}
	
		
/**
* Fixe les valeurs du vecteur  partir d'un autre.
* @param DVect v, le vecteur  copier.
*/
public void set(DVect v)
        {
        //if (n != v.n) 
        //      throws new IllegalArgumentException("the DVect objects must have the same size");
 
        if (Assert.enabled) Assert.check(n== v.n);
        int i;
        for(i=0;i<this.n;i++)
                x[i] = v.x[i];
        }

/**
* Fixe la valeur d'une composante.
* @param int i, l'indice de la composante  modifier.
* @param double x, la nouvelle valeur.
*/      
public void set(int i, double x)
        {
        //if (i > n || i<0) 
        //      throws new IllegalArgumentException("Illegal index");
         if (Assert.enabled) Assert.check(i < n);
        this.x[i] = x;
        }       
	
/**
* Fixe toutes les valeurs du vecteur.
* @param double[] x, le tableau des nouvelles valeurs.
*/
public void set(double[] x)
        {
        //if (n!=x.length) 
        //      throws new IllegalArgumentException("Illegal index");
        if (Assert.enabled) Assert.check( n==x.length );

        int i;
        for(i=0;i<this.n;i++)
                this.x[i] = x[i];
        }
	


/**
* Renvoie la valeur d'une composante.
* @param int i, le numero de la composante.
* @return double, la valeur de la composante.
*/
public double get(int i)
        {
        //if (i > n || i<0) 
        //      throws new IllegalArgumentException("Illegal index");
        if (Assert.enabled) Assert.check( (i < n) && (i>=0) );
        return x[i];
        //else throws...;
        }       

/**
* Renvoie toutes les valeurs du vecteur sous forme de tableau.
* @param double[] a, le tableau qui contiendra les valeurs des composantes.
*/      
public void get(double[] a)
        {
        //if (n != a.length) 
        //      throws new IllegalArgumentException("The array must have the same size");

        if (Assert.enabled) Assert.check(n == a.length) ;
        int i;
        for(i=0;i<n;i++)
                a[i]=x[i];
        }
/**
* Get the pointer to the one-dimensional array of elements.
* @return double[], the pointer to array.
* @since Opale-Mathtools 0.11
*/	
public double[] toArray()
	{
	//if ( a.size() != n)
	//	throws ...:
	return x;
	}

/**
* Renvoie la taille du vecteur.
* @return int n, la taille.
* @deprecated
*/
public int getSize()
	{
	return n;
	}

/**
* Renvoie la taille du vecteur.
* @return int n, la taille.
* @since Opale-Mathtools 0.11
*/
public int size()
	{
	return n;
	}

/**
* Additionne un vecteur.
* @param DVect v, le vecteur  additionner.
*/
public void add(DVect v)
        {
        //if (n != v.n) 
        //      throws new IllegalArgumentException("the DVect objects must have the same size");
        if (Assert.enabled) Assert.check( n == v.n) ;
         
        int i;
        
        for(i=0;i<n;i++)
                x[i]+=v.x[i];
        }               
/**
* Additionne un tableau.
* @param double[] v, le tab  additionner.
* @since Opale-Mathtools 0.13
*/
public void add(double[] v)
        {
        //if (n != v.length) 
        //      throws new IllegalArgumentException("The array must have the same size");
         if (Assert.enabled) Assert.check( n == v.length) ;
        
        int i;
        
        for(i=0;i<n;i++)
                x[i]+=v[i];
        }               

/**
* X=X+l*V.
* @param double l, un rel.
* @param DVect v, le tab  additionner.
* @since Opale-Mathtools 0.13
*/
public void acc(double l, DVect v)
        {
        //if (n != v.n) 
        //      throws new IllegalArgumentException("the DVect objects must have the same size");
         if (Assert.enabled) Assert.check( n == v.n) ;
        
        int i;
        
        for(i=0;i<n;i++)
                x[i]+=l*v.x[i];
        }               
	
/**
* Soustrait un vecteur.
* @param DVect v, le vecteur  soustraire.
*/
public void sub(DVect v)
        {
        //if (n != v.n) 
        //      throws new IllegalArgumentException("the DVect objects must have the same size");
         if (Assert.enabled) Assert.check( n == v.n) ;
        
        int i;
        
        for(i=0;i<n;i++)
                x[i]=x[i]-v.x[i];
        }               



public void acc(DVect v, double a)
        {
        //if (n != v.n) 
        //      throws new IllegalArgumentException("the DVect objects must have the same size");
        if (Assert.enabled) Assert.check( n == v.n) ;
        
        int i;
        
        for(i=0;i<n;i++)
                x[i]+=a*v.x[i];
        }               


/**
* Multiplie les composantes par un rel.
* @param double a, le rel  multiplier.
*/
public void mul(double a)
	{
	int i;
	
	for(i=0;i<n;i++)
		x[i]*=a;
	}		

/**
* Renvoie la norme 2 au carr du vecteur.
* @return double, le carre de la norme.
*/
public double norm2Sq()
	{
	int i;
	double s=0;
	for(i=0;i<n;i++)
		s+=x[i]*x[i];
	return s;
	}		
	
/**
* Renvoie la norme 2 du vecteur.
* @return double, la norme.
*/
public double norm2()
	{
	return Math.sqrt(norm2Sq());
	}

/**
* Renvoie la norme infinie du vecteur.
* @return double, la norme.
* @since Opale-mathtools 0.14
*/
public double normInf()
	{
	double r=0;
	for (int i=0;i<n;i++)
		if (Math.abs(x[i])>r) r=Math.abs(x[i]);
	return r;
	}

/**
* Renvoie la norme 1 du vecteur.
* @return double, la norme.
*/
public double norm1()
	{
	int i;
	double s=0;
	for(i=0;i<n;i++)
		s+=Math.abs(x[i]);
	return s;
	}		


/**
* Renvoie la norme p du vecteur.
* @param int p, l'indice de la norme voulue.
* @return double, la norme.
*/
public double normp(int p)
	{
	if (p == 0)
		return -1;
	else 
		{
		int i;
		double s=0;
		for(i=0;i<n;i++)
			s+=Math.pow(Math.abs(x[i]),p);
		return Math.pow(s,1./p);
		}
	}
	
/**
* Renvoie le maximum des valeurs des composantes.
* @return double max, le maximum.
*/
public double getMax()
	{
	return Sort.getMax(n,x);
	}
	
/**
* Renvoie le minimum des valeurs des composantes.
* @return double min, le minimum.
*/
public double getMin()
	{
	return Sort.getMin(n,x);
	}

/**
* surcharge la mthode toString pour l'affichage.
* @return String, la chaine  afficher.
* @since Opale-Mathtools 0.11
*/
public String toString()
	{
	String s="";
	int i ;
	
	for (i=0;i<n-1;i++)
		s=s+x[i]+"\n";
	s+=x[n-1];	
	return s;
	}
	
/**
* Return a copy of this DVect.
* @return DVect, a copy of this object.
*/

public DVect copy()
	{
	DVect c = new DVect(this);
	return c;
	}
	
/**
* Cre un nouvel objet de mme classe et de mme contenu.
* @return Object un clone de l'objet.
* @exception  OutOfMemoryError s'il n'y a pas assez de mmoire.
* @see        java.lang.Cloneable
* @since Opale mathtools 0.14
*/
public Object clone()
	{
	try
		{
		DVect v = (DVect) super.clone();
		v.n = n;
		v.x = new double[n];
		for (int i =0;i<n;i++)
			v.x[i] = x[i];
		return v;
		}
	catch (CloneNotSupportedException e) // ne devrait jamais arriver
		{
		throw new InternalError();
		}
   	}
	
			
}

