/*
 * OPALE is a scientific library under LGPL. Its main goal is to
 * develop mathematical tools for any scientist.
 *
 * Copyright (C) 2002 Opale Group
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * You can visit the web site http://opale.tuxfamily.org to obtain more
 * informations about this program and/or to contact the authors by mail
 * developers@opale.tuxfamily.org.
 */




package opale.matrix.solver;

import opale.tools.Debug;
import opale.mathtools.DVect;
import opale.matrix.*;

/**
* This class implements Gauss-Seidel solver according to the book "Analyse matricielle applique  l'art de l'ingnieur", Lascaux & Thodor, Ed. Masson.
* @author O.C.
* @date 05/2001
* @since Opale-matrix 0.11
*/


public class SolverGaussSeidel implements IterativeSolver
{

/**
* Solve Ax=b.
* @param Matrix A.
*/
public void solve(Matrix A,DVect b, DVect x)
	{
	int n = A.getRowDim();
	if (n != A.getColumnDim() ) throw new IllegalArgumentException("Gauss-Seidel solver implemented only for square matrix");
	if ((n != b.size()) || (n != x.size())) throw new IllegalArgumentException("matrix and vector dimension are not equals.");
	
	double[][] AA=A.getInternalArray();
	double[] B = b.toArray();
	double[] X = x.toArray();
	double normr=1,normb=b.normInf();
	int it=0;
	
	while ((it++ < MAXITER) && (normr/normb>PRECISION))
	{
	normr=0.;
	for (int i=0;i<n;i++)
		{
		double c=B[i];
		for (int j=0;j<n;j++)
			c-=AA[i][j]*X[j];
		normr=Math.max(Math.abs(c),normr);
		X[i] = X[i]+c/AA[i][i];
		}
	if (Debug.On) Debug.print("residu = "+normr);		
	}
	
	if (Debug.On) Debug.print("n iter = "+it);		
	}
	

public static void main(String[] arg)
	{
	double[][] tab = { {4,1}, {1,4} };
	Matrix A=new Matrix(tab);
	
	DVect b=new DVect(2);
	DVect x=new DVect(2);

	System.out.println(A);
	
	b.set(0,1);
	b.set(1,0);
	
	
	SolverGaussSeidel gs = new SolverGaussSeidel();
	gs.solve(A,b,x);
		

	
	System.out.println(x);
	}
	
			
}
	
	
