#
# $Id: directswap.py 86 2009-05-02 19:02:20Z lxp $
#
# This file is part of OpenClone.
#
# Copyright (C) 2009  David Gnedt
#
# OpenClone is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenClone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with OpenClone.  If not, see <http://www.gnu.org/licenses/>.
#

from __future__ import with_statement
import logging
import subprocess
import threading

MODE_BACKUP, MODE_RESTORE = range(2)

logger = logging.getLogger('directswap')

# TODO: Clone swap version, page size?
class DirectSwap(threading.Thread):
    mode = None
    input = None
    output = None
    status = None
    status_lock = None
    
    def __init__(self, mode, **kw):
        threading.Thread.__init__(self)
        
        if mode is MODE_BACKUP:
            if 'input' not in kw:
                raise Exception('Keyword input expected in backup mode')
            
            if 'output' not in kw:
                raise Exception('Keyword output expected in backup mode')
        
        elif mode is MODE_RESTORE:
            if 'output' not in kw:
                raise Exception('Keyword output expected in restore mode')
            
            if 'input' not in kw:
                raise Exception('Keyword input expected in restore mode')
        
        self.mode = mode
        self.input = kw['input']
        self.output = kw['output']
        self.status = {}
        self.status_lock = threading.RLock()
        
        self.start()
    
    def run(self):
        if self.mode is MODE_BACKUP:
            uuid = None
            label = None
            
            # TODO: Get path (configuration?)
            args = ['/lib/udev/vol_id', '--type', self.input]
            logger.debug('Exec: %s' % ' '.join(args))
            proc = subprocess.Popen(args, stdout=subprocess.PIPE, close_fds=True)
            if proc.wait() != 0 or proc.stdout.read().rstrip('\n\r') != 'swap':
                raise Exception('Unknown partition type, swap expected')
            
            # TODO: Get path (configuration?)
            args = ['/lib/udev/vol_id', '--uuid', self.input]
            logger.debug('Exec: %s' % ' '.join(args))
            proc = subprocess.Popen(args, stdout=subprocess.PIPE, close_fds=True)
            status = proc.wait()
            if status == 0:
                uuid = proc.stdout.read().rstrip('\n\r')
            
            elif status == 4:
                # Null uuid found
                uuid = '00000000-0000-0000-0000-000000000000'
                pass
            
            else:
                raise Exception('Swap UUID read failed')
            
            # TODO: Get path (configuration?)
            args = ['/lib/udev/vol_id', '--label-raw', self.input]
            logger.debug('Exec: %s' % ' '.join(args))
            proc = subprocess.Popen(args, stdout=subprocess.PIPE, close_fds=True)
            status = proc.wait()
            if status == 0:
                # Label set
                label = proc.stdout.read().rstrip('\n\r')
            
            elif status == 3:
                # No label set
                pass
            
            else:
                raise Exception('Swap label read failed')
            
            self.output.write('#OpenClone directswap file\n')
            if uuid is not None:
                self.output.write('UUID=%s\n' % uuid)
            
            if label is not None:
                self.output.write('LABEL=%s\n' % label)
            
            self.output.close()
        
        elif self.mode is MODE_RESTORE:
            uuid = None
            label = None
            
            if self.input.readline() != '#OpenClone directswap file\n':
                raise Exception('Unkonwn image format')
            
            for line in self.input:
                # TODO: Fix split count (, 1)
                part = line.split('=')
                if len(part) < 2:
                    continue
                
                if part[0] == 'UUID':
                    # TODO: Fix rstrip count (, 1)
                    uuid = part[1].rstrip('\n')
                
                elif part[0] == 'LABEL':
                    # TODO: Fix rstrip count (, 1)
                    label = part[1].rstrip('\n')
                
                else:
                    continue
            
            self.input.close()
            
            # TODO: Get path (configuration?)
            args = ['/sbin/mkswap']
            if uuid is not None:
                args.append('-U')
                args.append(uuid)
            
            if label is not None:
                args.append('-L')
                args.append(label)
            
            args.append(self.output)
            
            logger.debug('Exec: %s' % ' '.join(args))
            proc = subprocess.Popen(args, close_fds=True)
            if proc.wait() != 0:
                raise Exception('Swap creation failed')
        
        with self.status_lock:
            self.status['success'] = True
        
        logger.debug('success: True')
    
    def getInput(self):
        return self.input
    
    def getOutput(self):
        return self.output
    
    def getStatus(self):
        with self.status_lock:
            return dict(self.status)
    
    def wait(self, timeout=None):
        self.join(timeout)
