#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

##  This file is part of orm, The Object Relational Membrane Version 2.
##
##  Copyright 2002-2006 by Diedrich Vorberg <diedrich@tux4web.de>
##
##  All Rights Reserved
##
##  For more Information on orm see the README file.
##
##  This program is free software; you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation; either version 2 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program; if not, write to the Free Software
##  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
##
##  I have added a copy of the GPL in the file COPYING

# Changelog
# ---------
#
# $Log: datasource.py,v $
# Revision 1.3  2006/04/15 23:15:07  diedrich
# Split up select() in select() and run_select()
#
# Revision 1.2  2006/01/01 20:45:38  diedrich
# The gadfly datasource got its own execute() method (un_escaper
# mechanism, see docstrings).
#
# Revision 1.1  2005/12/31 18:30:30  diedrich
# Initial commit
#
#
#
#

"""
This datasouce module uses the Python-only, filesystem based SQL
backend called gadfly by Aaron Watters, currently maintained by
Richard Jones <richard@users.sf.net>. It is available at

  http://gadfly.sourceforge.net

This database adapter is fully useable with orm, though probably not

very usefull except for testing purposes.
"""

__author__ = "Diedrich Vorberg <diedrich@tux4web.de>"
__version__ = "$Revision: 1.3 $"[11:-2]

# Python
from types import *
from sets import Set
import re

# orm
from orm2.datasource import datasource_base
from orm2.debug import sqllog, debug
from orm2.exceptions import *
from orm2.datatypes import common_serial
from orm2 import sql
from orm2.util import stupid_dict

import orm2.datasource

from gadfly import gadfly

class datasource(datasource_base):
    """
    An orm database adapter for gadfly.
    """

    def __init__(self, dbname="tmp", directory="/tmp",
                 encoding = "iso-8859-1"):
        """
        The default values will create a database called tmp in your
        /tmp directory. Gadfly will create a number of files called dbname.*
        there.
        """        
        orm2.datasource.datasource_base.__init__(self)

        self._conn = gadfly()
        self._conn.startup(dbname, directory)
        self.encoding = encoding

        self._update_cursor = self.cursor()

    def _from_params(params):
        database_name = params.get("dbname", "tmp")
        database_directory = params.get("directory", "/tmp")
        encoding = params.get("encoding", "iso-8859-1")

        return datasource(database_name, database_directory, encoding)

    from_params = staticmethod(_from_params)

    def backend_encoding(self):
        return self.encoding


    def select(self, dbclass, *clauses):
        """
        This is basically a copy of the datasource_base.select() method that
        has been modified to accomodate gadfly's non-standard (i.e. DB SiG)
        complient behaviour. (The fetchone() method raises an exception when
        no more results are available instead of returning None.
        """
        # The use of the stupid_dict class has become neccessary, because
        # sql._part instances are not hashable.
        columns = stupid_dict()
        for property in dbclass.__dbproperties__():
            if property.__select_this_column__():
                columns[property.column] = 0

        columns = list(columns)
        
        query = sql.select(columns, dbclass.__relation__, *clauses)
        return self.run_select(dbclass, query)
    
    def run_select(self, dbclass, select):
        """
        Run a select statement on this datasource that is ment to return
        rows suitable to construct objects of dbclass from them.

        @param dbclass: The dbclass of the objects to be selected
        @param select: sql.select instance representing the query
        """
        cursor = self.execute(select)

        columns = dbclass.__select_columns__()

        for tpl in cursor.fetchall():
            info = stupid_dict(zip(columns, tpl))
            yield dbclass.__from_result__(self, info)
                
    def insert(self, dbobj, dont_select=False):
        """
        The gadfly backend does not provide a mechanism to create unique
        keys for new rows. Values for the common_serial() datatype must be
        determined by the insert() function. It will query the maximum value
        of the id column and increment it.
        """
        # does the dbobj have a common_serial property?
        query_id = False
        for property in dbobj.__dbproperties__():
            if isinstance(property, common_serial):
                common_serial_property = property
                query_id = True
                break

        if query_id:
            query = "SELECT COUNT(*) FROM %s" % dbobj.__relation__
            cursor = self.execute(query)
            tpl = cursor.fetchone()
            count = tpl[0]

            if count == 0:
                new_id = 1
            else:            
                query = "SELECT MAX(id) FROM %s" % dbobj.__relation__
                cursor = self.execute(query)
                tpl = cursor.fetchone()
                max_id = tpl[0]

                new_id = max_id + 1
                
            common_serial_property.__set_from_result__(self, dbobj, new_id)

        datasource_base.insert(self, dbobj, dont_select)

    def select_one(self, dbclass, *clauses):
        """
        Gadfly doesn't support the LIMIT clause. 
        """
        result = self.select(dbclass, *clauses)
        result = list(result)
        
        if len(result) == 0:
            return None
        else:
            return result[0]

        
    def select_after_insert(self, dbobj):
        """
        The gadfly backend neither supports default values for columns
        not owns a mechanism to provide unique keys for new rows. So the
        select_after_insert() mechanism is useless.
        """
        pass
    
    def select_after_insert_where(self, dbobj):
        """
        See select_after_insert() above.
        """
        raise NotImplemented()

    def execute(self, command, modify=False):
        """
        @param A string containing an SQL command of any kind or an
               sql.statement instance.
        
        Execute COMMAND on the database. If modify is True, the command
        is assumed to modify the database. All modifying commands
        will be executed on the same cursor.
        """
        if type(command) == UnicodeType:
            raise TypeError("Database queries must be strings, not unicode")
        
        if isinstance(command, sql.statement):
            command = sql.sql(self)(command)

        print >> sqllog, command

        cursor = self.cursor()

        # for an explainantion for the _un_escaper mechanism see the
        # class' docstring below.
        un_escaper = _un_escaper(self)
        command = re.sub(r"'(.*?)[^\\]'", un_escaper, command)

        cursor.execute(command, tuple(un_escaper.info))

        return cursor

class _un_escaper:
    """
    The problem: Gadfly doesn't accept binary strings (even when properly
    escaped) in the SQL statements it executes. This wouldn't hurt so much, if
    it wouldn't view utf-8 encoded strings as binary. Anyway, here's the
    solution: since the whole of orm2 uses ' as quotes for SQL string literal
    I can seek string literals from SQL statements using regular expresisons.
    This happens in gadfly.datasource.datasource.execute(). The un_escaper is
    used as the repl parameter of re.sub() (See Python Library Reference
    chap. 4.2.3 for details). Much like the sql module's sql() mechanism,
    this is basically a 'curryed' function. It will replace all strings with
    ?s and retain a list of the strings. The strings are then 'un-escaped'
    (the reverse of orm2.sql.datasource.escape_string()) and used as
    variable parameters for gadfly's cursor.execute() method. This way gadfly
    gets the binary strings as Python objects and inserts them semmlessly
    into its own datastructures with no need to parse them.
    """
    def __init__(self, ds):
        self.ds = ds
        self.info = []
        
    def __call__(self, match_object):
        s = match_object.group(0)

        for a in self.ds.escaped_chars:
            s = s.replace(a[1], a[0])
        
        self.info.append(s[1:-1])
        return "?"
