#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

##  This file is part of orm, The Object Relational Membrane Version 2.
##
##  Copyright 2002-2006 by Diedrich Vorberg <diedrich@tux4web.de>
##
##  All Rights Reserved
##
##  For more Information on orm see the README file.
##
##  This program is free software; you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation; either version 2 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program; if not, write to the Free Software
##  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
##
##  I have added a copy of the GPL in the file COPYING

# Changelog
# ---------
#
# $Log: datatypes.py,v $
# Revision 1.12  2006/04/28 09:49:27  diedrich
# Docstring updates for epydoc
#
# Revision 1.11  2006/04/28 08:40:31  diedrich
# - Changed validator handling
# - __set__ and __get__ check if they fit the dbobj passed
# - Added widget handling
# - Varchar now has a length_validator, use text for unlimited-length strings
#
# Revision 1.10  2006/04/21 18:53:45  diedrich
# Added validator exceptions
#
# Revision 1.9  2006/04/15 23:17:46  diedrich
# Changed widget_spec handling
#
# Revision 1.8  2006/02/25 17:59:55  diedrich
# Made the many2one work with multi column keys.
#
# Revision 1.7  2006/02/25 00:20:20  diedrich
# - Added and tested the ability to use multiple column primary keys.
# - Some small misc bugs.
#
# Revision 1.6  2006/01/01 20:39:55  diedrich
# - data_attribute_name() has been made a function
# - Actually update the database on property access
#
# Revision 1.5  2005/12/31 18:24:58  diedrich
# - Updated year in copyright header ;)
# - The difference between __set__ and __set_from_result__ has be re-introduced
#   to accomodate for the relationships and other situations
# - took care of situationsin which two properties refer to the same column
#   (the data_attribute_name is based on the column name instead of the
#    attribute name)
# - added Long datatype
# - Enforce the id name convention for the common_serial type.
#
# Revision 1.4  2005/12/31 09:56:11  diedrich
# - comments
# - added the common_serial datatype
#
# Revision 1.3  2005/12/18 22:35:46  diedrich
# - Inheritance
# - pgsql adapter
# - first unit tests
# - some more comments
#
# Revision 1.2  2005/11/21 19:57:51  diedrich
# - The hidden attributes of the dbobj start with a space character now,
#   not with a _ to avoid name clashes
# - added the has_default and __select_after_insert__() mechanism
# - added thorogh error checking in __get__()
# - added isset() method to be able to tell the difference between <not-set>,
#   None and a value
# - Wrote a more meaningfull error message when __delete__() is called
#
# Revision 1.1.1.1  2005/11/20 14:55:46  diedrich
# Initial import
#
#
#

"""
Datatype classes for the default SQL datatypes.
===============================================

  Each of the classes in this module models an SQL datatype. Their
  instances will be responsible for managing an attribute in a
  dbclass. The classes accept a number of arguments for you to
  influence what exactly they do. Refer to L{datatype.__init__} for these.
  
"""

from types import *
from orm2 import sql
from orm2.validators import validator, length_validator
from orm2.exceptions import *
from orm2.ui import widget_spec as widget_spec_cls

class datatype(property):
    """
    This class encapsulates a dbclass' property (=attribute). It takes
    care of the SQL column name, the widget_spec and the information
    actually stored in the database/the dbobject.
    """
    python_class = None    
    sql_literal_class = None
    
    def __init__(self, column=None, title=None,
                 validators=(), widget_specs=(), has_default=False):
        """
        @param column: A orm2.sql column instance or a string containing a SQL
            column name pointing to the column this property is
            responsible for. Defaults to the column with the same name
            as the attribute.
        @param title: The title of this column used in dialogs, tables and
            validator error messages (among other things). This must be a
            unicode object or None. 
        @param widget_specs: Either an instance of orm2.ui.widget_spec or a
            sequence of several of them.
        @param validators: A sequence of objects of validators.validator
            children. (A single validator is ok, too)
        @param has_default: Boolean property that determines whether this
            dbproperty is retrieved from the database after the dbobject has
            been INSERTed. (So has_default referrs to the SQL column really).
        """     
        if type(column) == StringType:
            self.column = sql.column(column)
        elif isinstance(column, sql.column):
            self.column = column
        elif column is None:
            self.column = None
        else:
            raise TypeError("Column must either be a string or an sql.column"+\
                            " instance, not %s (%s)" % ( repr(type(column)),
                                                         repr(column),) )
        self.title = title

        if isinstance(validators, validator):
            self.validators = ( validators, )
        else:
            self.validators = tuple(validators)
        
        self.widget_specs = []
        if type(widget_specs) == InstanceType:
            self.add_widget_specs(widget_specs)
        else:
            for a in widget_specs:
                self.add_widget_spec(a)
        
        self.has_default = has_default
        
    def __init_dbclass__(self, dbclass, attribute_name):
        """
        This methods gets called by dbobject's metaclass. It supplies the
        db property with info about the class it belongs to and its attribute
        name. 
        """
        self.dbclass = dbclass
        self.attribute_name = attribute_name
        # The actual data attribute's names start with a space character
        # to avoid name clashes.
        
        if self.column is None:
            self.column = sql.column(attribute_name)
            
        self._data_attribute_name = " %s" % str(self.column)

        if self.title is None:
            self.title = unicode(self.attribute_name, "ascii")
            # It's save to use ascii, because Python does not allow non-ascii
            # identifyers and the attribute_name is an identifyer, of course.

    def data_attribute_name(self):
        try:
            return self._data_attribute_name
        except AttributeError:
            raise DatatypeMustBeUsedInClassDefinition(self.__class__.__name__)
        
    def __get__(self, dbobj, owner="owner? Like owner of what??"):
        """
        See the Python Language Reference, chapter 3.3.2.2 for details on
        how this works. Be sure to be in a relaxed, ready-for-hard-figuring
        mood.
        """
        # The error checking in this method may seem overblown. But
        # working with orm1 showed me that informative error messages,
        # that precisely say what's going on make development a lot
        # more fun.
        
        if dbobj is None:
            raise NotImplementedError(
                "datatype properties work on instances only")

        self.check_dbobj(dbobj)
            
        if self.isset(dbobj):
            return getattr(dbobj, self.data_attribute_name())
        else:
            primary_key_property = repr(tuple(dbobj.__primary_key__.\
                                                         attribute_names()))
            if not dbobj.__primary_key__.isset():
                pk_literal = "<unset>"
            else:
                pk_literal = repr(tuple(dbobj.__primary_key__.values()))
                    
            tpl = ( self.attribute_name,
                    dbobj.__class__.__name__,
                    primary_key_property,
                    pk_literal, )
            
            msg = "Attribute '%s' of '%s' [ %s=%s ] has not yet been set" % tpl
                
            raise AttributeError( msg )
        
    def __set__(self, dbobj, value):
        """
        Set the attribute managed by this datatype class on instance
        to value.  This will be called by Python on attribute
        assignment. The __set_from_result__ method does the same thing
        for data retrieved from the RDBMS. See below.
        """
        self.check_dbobj(dbobj)
        if value is not None: value = self.__convert__(value)

        for validator in self.validators:
            validator.check(dbobj, self, value)
                
        setattr(dbobj, self.data_attribute_name(), value)
        
        if dbobj.__is_stored__():
            dbobj.__ds__().update( dbobj.__relation__,
                                   self.column,
                                   self.sql_literal(dbobj),
                                   dbobj.__primary_key__.where() )
            

    def __set_from_result__(self, ds, dbobj, value):
        setattr(dbobj, self.data_attribute_name(), value)

    def check_dbobj(self, dbobj):
        if self.attribute_name is not None and \
               not self in dbobj.__dbproperties__():
            msg = "dbclass '%s' does not have attribute '%s' (wrong " + \
                  "dbclass for this dbproperty!)"
            msg = msg % ( dbobj.__class__.__name__, self.attribute_name, )
            raise AttributeError(msg)

    def isset(self, dbobj):
        """
        @returns: True, if this property is set, otherwise... well.. False.
        """
        return hasattr(dbobj, self.data_attribute_name())
    
    def __convert__(self, value):
        """
        Return value converted as a Python object of the class assigned to
        this datatype.
        """
        if not isinstance(value, self.python_class):
            return self.python_class(value)
        else:
            return value

    def sql_literal(self, dbobj):
        """
        Return an SQL literal representing the data managed by this property
        in dbobj.
        
        @param ds: datasource object of the parent dbclass
        @return: SQL literal as a string.
        """

        if not self.isset(dbobj):
            msg = "This attribute has not been retrieved from the database."
            raise AttributeError(msg)
        else:        
            value = getattr(dbobj, self.data_attribute_name())

            if value is None:
                return sql.NULL
            else:
                return self.sql_literal_class(value)
    
    def __select_this_column__(self):
        """
        Indicate whether this column shall be included in SELECT statements.
        True by default, it will return False for most relationships.
        """
        return True

    def __select_after_insert__(self, dbobj):
        """
        Indicate whether this column needs to be SELECTed after the dbobj has
        been inserted to pick up information supplied by backend as by SQL
        default values and auto increment columns.
        """
        return self.has_default and not self.isset(dbobj)

    def __delete__(self, dbobj):
        raise NotImplementedError(
            "Can't delete a database property from a dbobj.")

    def add_widget_spec(self, widget_spec):
        if not isinstance(widget_spec, widget_spec_cls):
            raise TypeError("Widgets must be instances of a subclass of " + \
                            "orm2.ui.widget_spec")

        self.widget_specs.append(widget_spec)
        widget_spec.__init_datatype__(self)

    def widget(self, module, **kw):
        """
        This function takes arbitrary keyword arguments that will be
        passed to the constructor of the wrapper class. See
        orm2.ui.widget.__call__() for details!
        
        @param module: A regular Python string naming a module from
           orm2.ui.wrappers
        @returns: The widget defined for this datatype from `module', or
           None if not defined. <b>This refers to a real widget, an instance
           of the wrapper class!</b>
        """
        ret = filter(self.widget_specs, lambda spec: spec.module == module)
        
        if len(ret) == 0:
            return None
        else:
            widget_spec = ret[0]
            return widget_spec(self, **kw)
    
class integer(datatype):
    """
    dbclass property for INTEGER SQL columns.
    """    
    python_class = int
    sql_literal_class = sql.integer_literal

class Long(datatype):
    """
    dbclass property for INTEGER SQL columns.
    """    
    python_class = long
    sql_literal_class = sql.long_literal

class Float(datatype):
    """
    dbclass property for FLOAT and DOUBLE (etc) SQL columns.
    """    
    python_class = float
    sql_literal_class = sql.float_literal

class string(datatype):
    """
    dbclass property for TEXT etc. SQL columns.
    """    
    python_class = str
    sql_literal_class = sql.string_literal

text = string

class varchar(string):
    """
    dbclass property for string values with a fixed (maximum-)length.
    This is the string class above with a length_validator added.
    """    
    def __init__(self, max_length, column=None, title=None,
                 validators=(), widget_specs=(), has_default=False):
        validators = list(validators)
        validators.append(length_validator(max_length))
        datatype.__init__(self, column, title,
                          validators, widget_specs, has_default)

char = varchar        

class Unicode(datatype):
    """
    dbclass property for TEXT, VARCHAR, CHAR() etc. SQL columns that
    are to be converted from SQL literals (i.e. encoded strings) to
    Python Unicode objectes and vice versa.

    When setting a Unicode property of a dbobj, you might want to convert
    the value to Unicode yourself. This class uses Python's default encoding
    (see documentation for sys.getdefaultencoding()) to convert things *to*
    Unicode, which may or may not be what you want. 
    """
    python_class = unicode
    sql_literal_class = sql.unicode_literal

    def __set_from_result__(self, ds, dbobj, value):
        value = unicode(value, ds.backend_encoding())
        setattr(dbobj, self.data_attribute_name(), value)

class common_serial(integer):
    """
    The common_serial datatype is an primary key integer column whoes
    value is supplied by the backend using its default mechanism. The
    default mechanism for each backend is defined by the adapter's
    datatype module (see there). The name of the common_serial column is
    alway 'id'.

    This class used by some of the test cases to define data models that
    work on every backend.
    """
    
    def __init__(self):
        datatype.__init__(self, column="id",
                          widget_specs=(),
                          has_default=True)

    def __init_dbclass__(self, dbclass, attribute_name):
        if attribute_name != "id":
            raise ORMException("All common_serial columns must be called 'id'")
        integer.__init_dbclass__(self, dbclass, "id")

    def __set__(self, dbobj, value):
        raise NotImplementedError("common_serial values are always " + \
                                  "retrieved from the database backend")

    def __set_from_result__(self, ds, dbobj, value):
        integer.__set_from_result__(self, ds, dbobj, value)
