#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

##  This file is part of orm, The Object Relational Membrane Version 2.
##
##  Copyright 2002-2006 by Diedrich Vorberg <diedrich@tux4web.de>
##
##  All Rights Reserved
##
##  For more Information on orm see the README file.
##
##  This program is free software; you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation; either version 2 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program; if not, write to the Free Software
##  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
##
##  I have added a copy of the GPL in the file COPYING

# Changelog
# ---------
#
# $Log: dbobject.py,v $
# Revision 1.9  2006/04/28 09:49:27  diedrich
# Docstring updates for epydoc
#
# Revision 1.8  2006/04/28 08:42:03  diedrich
# - __init__dbclass__ is called by dbobject.__init__ for those properties
#   that are not part of the class definition now
# - Changed widget handling
# - Added __eq__() and __ne__() based on keys.primary_key.__eq__()
#
# Revision 1.7  2006/04/21 18:58:58  diedrich
# Added __select_columns__ and widgets() methods to dbobject class.
#
# Revision 1.6  2006/02/25 00:20:20  diedrich
# - Added and tested the ability to use multiple column primary keys.
# - Some small misc bugs.
#
# Revision 1.5  2006/01/01 20:41:11  diedrich
# Added __is_stored__() and __dbproperty__()
#
# Revision 1.4  2005/12/31 18:27:22  diedrich
# - Updated year in copyright header ;)
# - The difference between __set__ and __set_from_result__ has be re-introduced
#   to accomodate for the relationships and other situations
# - __dbproperties__() and __paimary_key_column__() are classmethods now
#
# Revision 1.3  2005/12/18 22:35:46  diedrich
# - Inheritance
# - pgsql adapter
# - first unit tests
# - some more comments
#
# Revision 1.2  2005/11/21 19:59:11  diedrich
# - renamed __columns__() to __dbproperties__() and have it return a list of
#   properties
# - added the __repr__() method from orm's old dbclass module
#
# Revision 1.1.1.1  2005/11/20 14:55:46  diedrich
# Initial import
#
#
#

"""
This module defines one class, L{dbobject}, which is the base class
for all objects orm retrieves from or stores in the database.
"""

from string import *
from types import *

from orm2 import sql, keys
from orm2.util import stupid_dict
from orm2.exceptions import *
from orm2.datatypes import datatype
class dbobject(object):
    """
    Base class for all database aware classes.

    It contains a number of helper methods which are called like this:
    __help__. You may safely add db-aware properties, regular properties
    and methods.
    """
    
    class __metaclass__(type):
        def __new__(cls, name, bases, dict):
            ret = type.__new__(cls, name, bases, dict)
            
            if name != "dbobject":
                for attr_name, property in dict.items():
                    if isinstance(property, datatype):
                        property.__init_dbclass__(cls, attr_name)

                # Add (=inherit) db-properties from our parent classes
                for base in bases:
                    for attr_name, property in base.__dict__.items():
                        if isinstance(property, datatype):
                            setattr(ret, attr_name, property)
                            
                if not hasattr(ret, "__relation__"):
                    ret.__relation__ = sql.relation(name)
                elif type(ret.__relation__) == StringType:
                    ret.__relation__ = sql.relation(ret.__relation__)
                elif type(ret.__relation__) == UnicodeType:
                    raise TypeError("Unicode not allowed as SQL identifyer")
                elif isinstance(ret.__relation__, sql.relation):
                    pass
                else:
                    msg = "Relation name must be a string or an" + \
                          "sql.relation() instance, not %s (%s)"
                    raise TypeError(msg % ( repr(type(ret.__relation__)),
                                            repr(ret.__relation__),) )

            return ret


    # The primary key must be either
    #  - a keys.primary_key instance
    #  - a tuple of strings indicating attribute (not column!) names of this
    #    class that form a multi column primary key
    #  - a simple string indicating the attribute that manages the primary
    #    key column of this dbclass
    #  - None if the class does not have a primary key (which makes it
    #    impossible to update rows by updating an instance's attributes
    #    through orm)
    
    __primary_key__ = "id"

    def __init__(self, **kw):
        """
        Construct a dbobj from key-word arguments.
        """
        for name, prop in self.__class__.__dict__.items():
            if isinstance(prop, datatype) and  \
                   not hasattr(prop, "dbclass"):
                prop.__init_dbclass__(self.__class__, name)

        
        for name, value in kw.items():
            if self.__class__.__dict__.has_key(name):
                self.__class__.__dict__[name].__set__(self, value)
            else:
                raise NoSuchAttributeOrColumn(name)

        self._ds = None

        if self.__primary_key__ is not None:
            self.__primary_key__ = keys.primary_key(self)


    def __from_result__(cls, ds, info):
        """
        @param ds: datasource we are created by (see select() method)
        @param info: dictionary as { 'column_name': <data> }        
        """
        self = cls()
        for property in cls.__dbproperties__():
            if info.has_key(property.column):
                property.__set_from_result__(ds, self, info[property.column])

        self._ds = ds

        return self

    __from_result__ = classmethod(__from_result__)

    def __insert__(self, ds):
        """
        This method is called by datasource.insert() after the insert
        query has been performed. It sets the dbobj's _ds attribute.
        
        @param ds: datasource that just inserted us        
        """
        self._ds = ds
    
    def __ds__(self):
        """
        Return this dbobject's datasource.
        """
        return self._ds

    def __is_stored__(self):
        """
        @returns: Wheather this dbobj has been stored in the database already
           or retrieved from it
        """
        if getattr(self, "_ds", None) is not None:
            return True
        else:
            return False

    def __dbproperties__(cls):
        """
        Return the datatype objects among this dbobjects attributes as a dict
        like { name: property, ... }
        """
        for prop in cls.__dict__.values():
            if isinstance(prop, datatype):
                yield prop
                
    __dbproperties__ = classmethod(__dbproperties__)


    def __dbproperty__(cls, name=None):
        """
        Return a dbproperty by its name. Raise exceptions if
        
          - there is no property by that name
          - it's not a dbproperty

        name defaults to the dbclass' primary key.  
        """
        if name is None:
            if cls.__primary_key__ is None:
                raise NoPrimaryKey()
            else:
                name = cls.__primary_key__

        try:
            property = cls.__dict__[name]
        except KeyError:
            raise AttributeError("No such attribute: %s" % repr(name))

        if not isinstance(property, datatype):
            raise NoDbPropertyByThatName(name + " is not a orm2 datatype!")

        return property

    __dbproperty__ = classmethod(__dbproperty__)


    def __select_columns__(cls):
        # The use of the stupid_dict class has become neccessary, because
        # sql._part instances are not hashable.
        columns = stupid_dict()
        for property in cls.__dbproperties__():
            if property.__select_this_column__():
                columns[property.column] = 0
                
        columns = list(columns.keys())

        return columns
    
    __select_columns__ = classmethod(__select_columns__)

        
    def __repr__(self):
        """
        Return a human readable (more or less) representation of this
        dbobject.
        """
        ret = []

        ret.append("pyid=" + str(id(self)))
        
        #if self.oid():
        #    ret.append("oid=%i" % self.oid())
        #else:
        #    ret.append("oid=NULL")

        attribute_names = []
        for name, value in self.__dict__.items():
            if isinstance(value, datatype):
                attribute_names.append(name)
                
        for a in attribute_names:
            b = a + "="

            try:
                val = getattr(self, a)
                
                #if not isinstance(val, relationships.relationshipColumn):
                #    b += repr(val.get())
                #else:
                b += repr(val)
            except AttributeError:
                b += "<not set>"

            ret.append(b)
            
        return "<" + self.__class__.__name__ + " (" + \
               join(ret, " ") + ")>"


    def __eq__(self, other):
        if self.__primary_key__ is None or other.__primary_key__ is None:
            raise ValueError("Can't check equality on dbclasses that don't have a primary key")

        if not self.__primary_key__.isset() or \
           not other.__primary_key__.isset():
            raise ValueError("Can't check equality on a dbobj whoes primary key is not yet set")
            
        return self.__primary_key__.__eq__(other.__primary_key__)

    def __ne__(self, other):
           return (not self == other)
