#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

##  This file is part of orm, The Object Relational Membrane Version 2.
##
##  Copyright 2002-2006 by Diedrich Vorberg <diedrich@tux4web.de>
##
##  All Rights Reserved
##
##  For more Information on orm see the README file.
##
##  This program is free software; you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation; either version 2 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program; if not, write to the Free Software
##  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
##
##  I have added a copy of the GPL in the file gpl.txt.

#
# $Log: xist.py,v $
# Revision 1.1  2006/04/28 09:56:41  diedrich
# Initial commit
#
#

import sys
from string import *

from orm2.ui import widget_wrapper

from ll.xist.ns import html, xml, chars
from ll.xist.xsc import *
from ll.xist import parsers
from ll.xist.errors import NodeNotFoundError


class xist_widget(widget_wrapper):
    def __init__(self, datatype, name=None, title=None, help="", default=None,
                 **kw):
        """
        @param datatype: The datatype this widget represents
        @param name: The name attribute of the HTML <input> (or so) element
          that will be created by this. Defaults to dbclass.attribute, which
          will do for most circumstances... A regular String.
        @param title: Will overwrite the datatype.title attribute for the
          widget's caption. Unicode string required.
        @param help: Help text that will be displayed along with the widget.
          May be a XIST element, of course, but you may not want to use block
          level elements in it.
        @param default: Unicode String or other Python object suitable  to
          determine the default setting for this widget if dbobj is None
          (see widget() below)

        The key word arguments may be used to pass arbitrary attributes to
        the widget element.
        """  
        widget_wrapper.__init__(self, datatype)

        if name is None:
            self.name = "%s_%s" % ( datatype.dbclass.__name__,
                                    datatype.attribute_name, )
        else:
            self.name = name

        if title is None:
            self.title = self.datatype.title
        else:
            self.title = title

        if help is None:
            self._help = Frag()
        else:
            self._help = help

        self.extra_args = kw

    def widget(self, dbobj=None, **kw):
        """
        Returns a xist frag that represents the HTML to create the widget.
        Ideally this is a single HTML element. The key word arguments will be
        passed as attributes, overwriting the constructor's key word arguments
        (see above).

        @param dbobj: The (current) dbobj whoes value is supposed to be
            displayed. Defaults to None, which indicates the default setting
            (see constructor above).
        """
        raise NotImplementedError()

    def help(self, **kw):
        """
        @returns: A html.div instance containing the help message. The key word
          arguments will be added to the <div> as arguments. The css class
          defaults to 'help'
        """
        if not kw.has_key("class_"):
            kw["class_"] = "help"

        return html.div(self.help, **kw)

    def label(self, **kw):
        """
        @returns: A html.div instance containing the widget's label
            (title attribute). The key word arguments will be added to the
            <div> as arguments. The css class defaults to 'label'
        """    
        if not kw.has_key("class_"):
            kw["class_"] = "label"

        return html.div(self.title, **kw)

    def row(self, dbobj=None, plone=False, error=None):
        '''
        Return a row for an HTML form

        <div class="row">
          <div class="label">self.title</div>
          <div class="help">self._help if non-null</div>
          <div class="error">error if non-null</div>

          <div class="field">
            self.widget()
          </div>
        </div>

        @param dbobj: The current dbobj (for self.widget(), see above)
        @param plone: Set attributes so that this form may be used with
          CMFPlone`s JavaScript
        @param error: Error message to be displayed along with the widget.  
        '''
        if error is None:
            error = Frag()
        else:
            error = html.div(error, class_="error")

        return html.div(self.label(), self.help, error, self.widget(dbobj))



class input(xist_widget):
    '''
    Base class for <input /> element based widgets. The type= attribute is
    determined by the class name.
    '''
    def widget(self, dbobj=None, **kw):
        extra_args = self.extra_args.copy()
        extra_args.update(kw)

        if dbobj is None:
            value = self.default
        else:
            value = self.value(dbobj)

        return html.input(type=self.__class__.__name__, value=value,
                          name=self.name, **extra_args)

    def value(self, dbobj):
        """
        Return the contents of the value= attribute for dbobj. dbobj may not
        be None.
        """
        raise NotImplementedError()

class text(input):
    def value(self, dbobj):
        u = self.datatype.__get__(dbobj)
        
        if u is None:
            return ""
        elif type(u) != UnicodeType:
            msg = "dbattributes managed by a %s.%s widget must " + \
                  "return a Unicode object! (%s.%s doesn't)"
            msg = msg % ( self.__class__.__module__.__name__,
                          self.__class__.__name__,
                          dbobj.__class__.__name__,
                          self.datatype.attribute_name, )
                          
            raise TypeError(msg)
        else:
            return u

class checkbox(input):
    def widget(self, dbobj=None, **kw):
        extra_args = self.extra_args.copy()
        extra_args.update(kw)

        if dbobj is None:
            value = bool(self.default)
        else:
            value = self.value(dbobj)

        if value:
            checked = "checked"
        else:
            checked = None

        return html.input(type=self.__class__.__name__, 
                          checked=checked, name=self.name, **extra_args)
    
    
    def value(self, dbobj):
        value = self.datatype.__get__(dbobj)
        return bool(value)
    
    

# Local variables:
# mode: python
# ispell-local-dictionary: "english"
# End:

