#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

##  This file is part of orm, The Object Relational Membrane Version 2.
##
##  Copyright 2002-2006 by Diedrich Vorberg <diedrich@tux4web.de>
##
##  All Rights Reserved
##
##  For more Information on orm see the README file.
##
##  This program is free software; you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation; either version 2 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program; if not, write to the Free Software
##  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
##
##  I have added a copy of the GPL in the file gpl.txt.


# Changelog
# ---------
# $Log: __init__.py,v $
# Revision 1.4  2006/04/28 09:49:26  diedrich
# Docstring updates for epydoc
#
# Revision 1.3  2006/01/01 20:48:40  diedrich
# Added the stupid_dict class.
#
# Revision 1.2  2005/12/31 18:33:06  diedrich
# Updated year in copyright header ;)
#
# Revision 1.1  2005/11/21 19:50:23  diedrich
# Initial commit
#
#

"""
This module defines a number of miscellaneous helper functions and classes.
"""

from types import *

class stupid_dict:
    """
    This class implements the mapping (dict) interface. It uses a
    simple list to store its data and sequential search to access
    it. It does not depend on __hash__() to manage contained
    objects. (See Python Reference Manual Chapter 3.3)

    The actual data is stored in self.data as a list of tuples like
    (key, value).

    See the docstring of orm2.sql._part for details on why this is here.
    """
    def __init__(self, initdata=[]):
        if type(initdata) in (ListType, TupleType,):
            self.data = []
            for tpl in initdata:
                if type(tpl) not in (ListType, TupleType) or len(tpl) != 2:
                    raise ValueError("Cannot inittiate stupid_dict from "+\
                                     "that data")
                else:
                    self.data.append(tpl)
        elif type(initdata) == DictType:
            self.data = initdata.items()
        else:
            raise ValueError("A stupid_dict must be initialized either by "+\
                             "a list of pairs or a regular dictionary.")

    def __len__(self):
        return len(self.data)

    def __getitem__(self, which):
        for key, value in self.data:
            if key == which:
                return value

        if hasattr(self, "default"):
            return self.default
        else:
            raise KeyError(what)

    def __setitem__(self, which, what):
        if self.has_key(which):
            self.__delitem__(which)
            
        self.data.append( (which, what,) )

    def __delitem__(self, which):
        if self.has_key(which):
            idx = self.keys().index(which)
            del self.data[idx]
        else:
            raise KeyError(which)


    def __iter__(self):
        for key, value in self.data: yield key


    def __contains__(self, which):
        return which in self.keys()

    def __cmp__(self, other):
        raise NotImplementedError("I have no idea on how to do this...")
        
    def __eq__(self, other):
        self.data.sort()
        other.data.sort()
        
        if self.data == other.data:
            return True
        else:
            return False

    def __repr__(self):
        return "stupid_dict(%s)" % repr(self.data)

    def clear(self):
        self.data = []

    def copy(self):
        return stupid_dict(self.data[:])

    def get(self, which, default=None):
        if self.has_key(which):
            return self[which]
        else:
            return default

    def has_key(self, which):
        if which in self.keys():
            return True
        else:
            return False

    def items(self):
        return self.data[:]

    def iteritems(self):
        for tpl in self.data: yield tpl

    iterkeys = __iter__
    
    def itervalues(self):
        for key, value in self.data: yield value

    def keys(self):
        return list(self.iterkeys())

    def values(self):
        return list(self.itervalues())

    def pop(self):
        raise NotImplementedError("This doesn't make sense in a stupid_dict,"+\
                                  " or does it? No, seriously...")

    popitem = pop

    def setdefault(self, default):
        self.default = default

    def update(self, other):
        """
        Other must implement the mapping (i.e. dict) interface.
        """
        for key, value in other.items():
            self[key] = value

    

# Local variables:
# mode: python
# ispell-local-dictionary: "english"
# End:

