/* pam_bemyguest.c
   Copyright (C) 2004, 2005 g10 Code GmbH
   Copyright (C) 2006 Moritz Schulte <moritz@gnu.org>
 
   This file is part of PAM Bemyguest.
  
   PAM Bemyguest is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.
  
   PAM Bemyguest is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
  
   You should have received a copy of the GNU Lesser General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.  */

#define _GNU_SOURCE

#include <config.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <stdarg.h>
#include <errno.h>
#include <pwd.h>
#include <assert.h>
#include <time.h>

#define PAM_SM_AUTH
#define PAM_SM_SESSION
#include <security/pam_modules.h>



#define PAM_BEMYGUEST "pam-bemyguest"

#define GUEST_ACCOUNT_PREFIX "guest"
#define GUEST_ACCOUNT_ID_SIZE 12
#define GUEST_ACCOUNT_SIZE \
  (sizeof (GUEST_ACCOUNT_PREFIX) + GUEST_ACCOUNT_ID_SIZE)



static void
guest_account_name_generate (char *name)
{
  const struct tm *brokentime;
  time_t current_time;
  int ret;

  current_time = time (NULL);
  brokentime = localtime (&current_time);

  strcpy (name, GUEST_ACCOUNT_PREFIX);
  ret = strftime (name + sizeof (GUEST_ACCOUNT_PREFIX) - 1,
		  GUEST_ACCOUNT_SIZE - sizeof (GUEST_ACCOUNT_PREFIX),
		  "%s", brokentime);
  assert (ret);
}

static int
user_add (const char *account)
{
  char *command;
  int ret;

  assert (! strncmp (account, GUEST_ACCOUNT_PREFIX,
		     sizeof (GUEST_ACCOUNT_PREFIX) - 1));

  ret = asprintf (&command, "/usr/sbin/useradd -m '%s'", account);
  if (ret < 0)
    {
      syslog (LOG_ERR,
	      "failed to construct user add command for account `%s'", account);
      return 1;
    }

  ret = system (command);
  free (command);

  return !! ret;
}

static int
user_delete (const char *account)
{
  char *command;
  int ret;

  assert (! strncmp (account, GUEST_ACCOUNT_PREFIX,
		     sizeof (GUEST_ACCOUNT_PREFIX) - 1));

  ret = asprintf (&command, "/usr/sbin/userdel -r '%s'", account);
  if (ret < 0)
    {
      syslog (LOG_ERR,
	      "failed to construct user del command for account `%s'", account);
      return 1;
    }

  ret = system (command);
  free (command);

  /* FIXME/TODO: call all user processes?  */

  return !! ret;
}

/* Option structure layout.  */
struct pam_bemyguest_opt
{
  unsigned int debug; /* Enable debugging.  */
};

/* Option structure definition.  */
struct pam_bemyguest_opt pam_bemyguest_opt =
  {
    0
  };

#define DEBUGGING (pam_bemyguest_opt.debug)



/* This function parses the PAM argument vector ARGV of size ARGV */
static int
parse_argv (int argc, const char **argv)
{
  unsigned int i;
  int ret;

  ret = 0;
  
  for (i = 0; i < argc; i++)
    {
      if (! strcmp (argv[i], "debug"))
	{
	  /* Handle "debug" option.  */
	  pam_bemyguest_opt.debug = 1;
	}
      else
	{
	  syslog (LOG_ERR, "unknown PAM argument: %s", argv[i]);
	  ret = 1;
	  break;
	}
    }

  return ret;
}




/*
 * PAM interface.
 */

/* Authenticate a user.  */
PAM_EXTERN int
pam_sm_authenticate (pam_handle_t *pam_handle,
		     int flags, int argc, const char **argv)
{
  const char *username;
  char *guestname;
  int ret;

  openlog (PAM_BEMYGUEST, LOG_PID, LOG_AUTH);

  /* Parse argument vector provided by PAM.  */
  if (parse_argv (argc, argv))
    {
      ret = PAM_AUTH_ERR;
      goto out;
    }

  if (DEBUGGING)
    syslog (LOG_ERR, "Starting...\n");

  /*
   * Retrieve information from PAM.
   */

  guestname = NULL;
  username = NULL;

  ret = pam_get_user (pam_handle, &username, NULL);
  if (ret != PAM_SUCCESS)
    goto out;

  if ((! username) || strcmp (username, GUEST_ACCOUNT_PREFIX))
    {
      ret = PAM_AUTH_ERR;
      goto out;
    }

  guestname = malloc (GUEST_ACCOUNT_SIZE);
  assert (guestname);

  guest_account_name_generate (guestname);
  /* FIXME: error checking.  */

  /* Make username available to application.  */
  ret = pam_set_item (pam_handle, PAM_USER, guestname);
  if (ret != PAM_SUCCESS)
    goto out;

  if (user_add (guestname))
    {
      syslog (LOG_INFO, "failed to add user '%s' to the system\n", guestname);
      ret = PAM_AUTH_ERR;
    }
  else
    syslog (LOG_INFO, "user '%s' has been added to the system\n", guestname);

  guestname = NULL;

  /* FIXME: memory leak?  should GUESTNAME be deallocated or not?  */
  
  /* Done.  */

 out:

  /* Release resources.  */

  closelog ();			/* FIXME: is this correct?  */

  return ret;
}




/* PAM's `set-credentials' interface.  */
PAM_EXTERN int
pam_sm_setcred (pam_handle_t *pam_handle,
		int flags, int argc, const char **argv)
{
  /* Dunno what to do here.  */
  return PAM_SUCCESS;
}



PAM_EXTERN int
pam_sm_open_session(pam_handle_t *pamh, int flags, int argc, const char **argv)
{
  /* Hmm.  */
  return PAM_SUCCESS;
}



PAM_EXTERN int
pam_sm_close_session(pam_handle_t *pamh, int flags, int argc, const char **argv)
{
  const void *username_void;
  const char *username;
  int ret;

  ret = pam_get_item (pamh, PAM_USER, &username_void);
  if (ret != PAM_SUCCESS)
    goto out;

  username = username_void;

  if (strncmp (username,
	       GUEST_ACCOUNT_PREFIX, sizeof (GUEST_ACCOUNT_PREFIX) - 1))
    {
      syslog (LOG_ERR, "close_session: skipping deleting user");
      goto out;
    }

  ret = user_delete (username);

  /* delete user.  */

 out:

  return ret;
}

/* END */
