/*-
 * Copyright (c) 2006 Fredrik Lindberg. <fli at shapeshifter dot se>
 * Copyright (c) 2006 Josef Hajas <josef at hajas dot net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: int_bioapi.c,v 1.3 2006/12/18 14:41:08 nax Exp $
 */

#include <int_bioapi.h>

/*
 * Attach a BSP to the BioAPI framework
 * Arguments
 *     bsp_uuid - BioAPI UUID string identifying the BSP
 *
 * Returns a BioAPI handle which can be used to reference this
 * BSP for further operations.
 */
BioAPI_HANDLE *bioapi_attach_bsp(const char *bsp_uuid)
{
    BioAPI_RETURN bioReturn;
    BioAPI_VERSION Version;
    const BioAPI_UUID uuid;
    BioAPI_HANDLE *handle = malloc(sizeof(BioAPI_HANDLE));

    bioReturn = BioAPI_GetStructuredUUID(bsp_uuid, (BioAPI_UUID *) & uuid);
    bioReturn = BioAPI_ModuleLoad(&uuid, 0, NULL, 0);
    if (bioReturn != BioAPI_OK) {
        bioapi_error_code ("load module", bioReturn);
	return (NULL);
    }

    Version.Major = BioAPI_MAJOR;
    Version.Minor = BioAPI_MINOR;

    bioReturn = BioAPI_ModuleAttach(&uuid, &Version, &BioAPIMemoryFuncs,
				    0, 0, 0, 0, NULL, 0, NULL, handle);

    if (bioReturn != BioAPI_OK) {
        bioapi_error_code ("attach module", bioReturn);
	return (NULL);
    }

    return (handle);
}


/*
 * Detach a BSP from the BioAPI framework
 * Arguments
 *    handle - BioAPI handle returned from bioapi_attach_bsp()
 *    bsp_uuid - BioAPI UUID string identifying the BSP
 */
void
bioapi_detach_bsp(BioAPI_HANDLE *handle, const char *bsp_uuid)
{
	const BioAPI_UUID uuid;
        BioAPI_RETURN bioReturn;

	bioReturn = BioAPI_GetStructuredUUID(bsp_uuid, (BioAPI_UUID *)&uuid);
	bioReturn = BioAPI_ModuleUnload(&uuid, NULL, 0);
	if (bioReturn != BioAPI_OK) {
            bioapi_error_code ("module unload", bioReturn);
	}
	bioReturn = BioAPI_ModuleDetach(*handle);
	free(handle);
}

/*
 * Initialize the BioAPI framework 
 */
int bioapi_init()
{
    BioAPI_RETURN bioReturn;
    BioAPI_VERSION bioVersion;

    bioVersion.Major = BioAPI_MAJOR;
    bioVersion.Minor = BioAPI_MINOR;
    bioReturn = BioAPI_Init(&bioVersion, 0, NULL, 0, NULL);
    if (bioReturn != BioAPI_OK) {
        bioapi_error_code ("init BioAPI framework", bioReturn);
	return (1);
    }

    return 0;
}

/*
 * Close the BioAPI framework 
 */
void bioapi_destroy()
{

    BioAPI_Terminate();
}

/*
 * Capture for enroll
 * Arguments
 *     handle - BioAPI BSP handle of the BSP we want to talk to
 */
BioAPI_BIR_PTR bioapi_capture_for_enroll (BioAPI_HANDLE * handle)
{
    BioAPI_BIR_HANDLE captured;
    BioAPI_BIR_PTR bir_data = NULL;
    BioAPI_RETURN bioReturn;

    bioReturn = BioAPI_Capture (*handle,
		      BioAPI_PURPOSE_ENROLL,
		      &captured, -1, NULL);
    if (bioReturn != BioAPI_OK) {
        bioapi_error_code ("capture fingerprint", bioReturn);
	return (NULL);
    }

    bioReturn =
	BioAPI_GetBIRFromHandle(*handle, captured, &bir_data);
    if (bioReturn != BioAPI_OK) {
        bioapi_error_code ("getting fingerprint from template", bioReturn);
	return (NULL);
    }
    return bir_data;
}

/*
 * Enroll a new user
 * Arguments
 *     handle - BioAPI BSP handle of the BSP we want to talk to
 */
struct birdb_rec *bioapi_enroll (BioAPI_HANDLE * handle, 
    BioAPI_BIR_PTR bir,
    const char *user, int type, 
    int payloadToBir, const char *payload)
{
    BioAPI_INPUT_BIR input_bir;
    BioAPI_RETURN bioReturn;
    BioAPI_BIR_HANDLE captured_template;
    BioAPI_BIR_PTR bir_data = NULL;
    BioAPI_DATA bioapiPayload;
    BioAPI_DATA_PTR bioapiPayloadPtr = NULL;
    struct birdb_rec *rec;
    int error;

    rec = malloc(sizeof(struct birdb_rec));
    if (rec == NULL)
	return (NULL);

    input_bir.InputBIR.BIR = bir;
    input_bir.Form = BioAPI_FULLBIR_INPUT;

    bioapiPayload.Length = 0;
    if (payload) {
	bioapiPayload.Length = strlen (payload);
	bioapiPayload.Data = (uint8 *) payload;
	if (payloadToBir && (bioapiPayload.Length > 0)) {
	    bioapiPayloadPtr = &bioapiPayload;
	}
    }
    /*bioReturn = BioAPI_Enroll(*handle,
		      BioAPI_PURPOSE_ENROLL,
		      NULL, &captured_template, bioapiPayloadPtr, -1, NULL);
    if (bioReturn != BioAPI_OK) {
        bioapi_error_code ("enroll", bioReturn);
	return (NULL);
    }*/
    
    bioReturn = BioAPI_CreateTemplate (*handle,
		      &input_bir, NULL, &captured_template, bioapiPayloadPtr);
    if (bioReturn != BioAPI_OK) {
        bioapi_error_code ("enroll", bioReturn);
	return (NULL);
    }

    bioReturn =
	BioAPI_GetBIRFromHandle(*handle, captured_template, &bir_data);
    if (bioReturn != BioAPI_OK) {
        bioapi_error_code ("getting template from template", bioReturn);
	return (NULL);
    }

    /* Set up our record */
    rec->br_key = strdup(user);
    rec->br_type = type;
    rec->br_bir = bir_data;
    rec->br_other = NULL;
    if ((! payloadToBir) && (bioapiPayload.Length > 0)) {
    	rec->br_other = malloc ( sizeof (br_other_t));
	rec->br_other->Length = bioapiPayload.Length;
	rec->br_other->Data = bioapiPayload.Data;
    }

    return (rec);
}

/*
 * Free resources allocated to a BIR
 */
void bioapi_freebir(BioAPI_BIR * bir)
{

    free(bir->BiometricData);
    if (bir->Signature) {
	free(bir->Signature->Data);
	free(bir->Signature);
    }
    free(bir);
}

/*
 * Identify a user based on biometric data
 * Arguments
 *     handle - BioAPI BSP handle of the BSP we want to talk to 
 *     username - Will be set to the matching user if any, it's the
 *                responsibility of the called to free this memory.
 *
 * Returns 0 on success, non-zero on failure.
 */
int
bioapi_identify(BioAPI_HANDLE * handle, struct birdb_mod *bm, void *bmh,
		char **username, char **payload)
{
    BioAPI_BIR_HANDLE template, bir_processed;
    BioAPI_INPUT_BIR input_bir, bir_capture, bir_input_proc;
    BioAPI_BIR_HEADER bir_header;
    BioAPI_BOOL resp = 0;
    BioAPI_BOOL prec = BioAPI_TRUE;
    BioAPI_FAR far_max, far_ach;
    BioAPI_RETURN bioReturn;
    struct birdb_rec br;
    struct birdb_rec *rec;
    int retn = -1;

    br.br_key = *username;
    br.br_type = BioAPI_NOT_SET;
    br.br_ctime = -1;

    /* do we have any records in DB for testing arain? */
    rec = birdb_backend_seqgetfirst(bm, bmh, &br);
    if (rec == NULL) {
        birdb_backend_seqfree(bm, bmh, rec);
        return (-1);
    }
    birdb_backend_seqfree(bm, bmh, rec);

    bioReturn = BioAPI_Capture(*handle, BioAPI_PURPOSE_VERIFY,
			       &template, -1, NULL);
    if (bioReturn != BioAPI_OK) {
        bioapi_error_code ("capture fingerprint", bioReturn);
	retn = -1;
	goto out;
    }

    bioReturn = BioAPI_GetHeaderFromHandle(*handle, template, &bir_header);
    if (bioReturn != BioAPI_OK) {
        bioapi_error_code ("getting template from image", bioReturn);
	retn = -1;
	goto out;
    }

    if (bir_header.Type == BioAPI_BIR_DATA_TYPE_INTERMEDIATE) {
	bir_capture.Form = BioAPI_BIR_HANDLE_INPUT;
	bir_capture.InputBIR.BIRinBSP = &template;

	bioReturn = BioAPI_Process(*handle, &bir_capture, &bir_processed);
	if (bioReturn != BioAPI_OK) {
	    retn = -1;
            bioapi_error_code ("fingerprint processing", bioReturn);
	    goto out;
	}
	bir_input_proc.Form = BioAPI_BIR_HANDLE_INPUT;
	bir_input_proc.InputBIR.BIRinBSP = &bir_processed;
    } else {
	bir_input_proc.Form = BioAPI_BIR_HANDLE_INPUT;
	bir_input_proc.InputBIR.BIRinBSP = &template;
    }

    for (rec = birdb_backend_seqgetfirst(bm, bmh, &br);
	 rec != NULL; rec = birdb_backend_seqgetnext(bm, bmh, rec)) {
        far_max = 1;

	input_bir.InputBIR.BIR = rec->br_bir;
	input_bir.Form = BioAPI_FULLBIR_INPUT;
	bioReturn = BioAPI_VerifyMatch(*handle, &far_max, NULL, &prec,
				       &bir_input_proc, &input_bir, NULL,
				       &resp, &far_ach, NULL, NULL);

	if (bioReturn != BioAPI_OK) {
	    bioapi_error_code ("verify match", bioReturn);
	    retn = -1;
	    break;
	}

	if (resp) {
	    if (! *username) {
	    	*username = strdup(rec->br_key);
	    }
	    if (rec->br_other) {
		*payload  = strndup (rec->br_other->Data, rec->br_other->Length);
	    }
	    retn = 0;
	    fprintf (stderr, "g: %d Operation succeeded\n", PAM_TEXT_INFO);
	    break;
	}
    }
    if (! resp) {
        fprintf (stderr, "g: %d Operation failed\n", PAM_TEXT_INFO);
    }

    birdb_backend_seqfree(bm, bmh, rec);
  out:
    return (retn);
}

/* Check database for identical bir given. If any found
 * it is returned. NULL otherwise.
 */
struct birdb_rec *bioapi_bir_in_db(dbstuff_t dbstuff,
				   BioAPI_HANDLE bspHandle,
				   BioAPI_BIR_PTR bir)
{
    struct birdb_rec *foundRec = NULL;
    struct birdb_rec *recptr;
    BioAPI_INPUT_BIR inputBir, inputBirFromDb;
    BioAPI_RETURN bRet;

    inputBir.InputBIR.BIR = bir;
    inputBir.Form = BioAPI_FULLBIR_INPUT;
    inputBirFromDb.Form = BioAPI_FULLBIR_INPUT;
    BioAPI_FAR maxFAR = 1;
    BioAPI_FAR achievedFAR;
    BioAPI_BOOL result = BioAPI_FALSE;
    BioAPI_BOOL bPrecedence = BioAPI_TRUE;

    struct birdb_rec br;
    br.br_key = NULL;
    br.br_type = BioAPI_NOT_SET;
    br.br_ctime = -1;

    for (recptr = birdb_backend_seqgetfirst(dbstuff.bm, dbstuff.beh, &br);
	 recptr != NULL;
	 recptr =
	 birdb_backend_seqgetnext(dbstuff.bm, dbstuff.beh, recptr)) {
	inputBirFromDb.InputBIR.BIR = recptr->br_bir;
	bRet = BioAPI_VerifyMatch(bspHandle, &maxFAR, NULL,
				  &bPrecedence, &inputBir,
				  &inputBirFromDb, NULL, &result,
				  &achievedFAR, NULL, NULL);
	if (bRet != BioAPI_OK) {
	    bioapi_error_code ("verify match", bRet);
    	    return NULL;
	}
	if (result == BioAPI_TRUE) {
	    birdb_backend_seqfree(dbstuff.bm, dbstuff.beh, NULL);
	    return recptr;
	}
    }
    birdb_backend_seqfree(dbstuff.bm, dbstuff.beh, recptr);
    return NULL;
}

/*
 * GUI callback for message translation.
 */
BioAPI_RETURN
bioapi_guicallback(void *chptr, BioAPI_GUI_STATE guistate,
		   BioAPI_GUI_RESPONSE response, BioAPI_GUI_MESSAGE msg,
		   BioAPI_GUI_PROGRESS progress,
		   BioAPI_GUI_BITMAP_PTR buffer)
{
    struct callbackh *ch = chptr;
    int ret = BioAPI_OK;

    if ((ch->showAll) || (ch->prevMsg != msg)) {
	if (msg < 38) {
	    if (CallBackMessages[msg][0] != '\0') {
		fprintf(stderr, "g: %d %s\n", PAM_TEXT_INFO,
			CallBackMessages[msg]);
	    }
	} else {
	    fprintf(stderr, "g: %d %dGuiMsg\n", PAM_TEXT_INFO, msg);
	}
	ch->prevMsg = msg;
    }
    if (cancel_bioapi) {
        cancel_bioapi = 0;
	ret = BioAPI_CANCEL;
    }

    return (ret);
}

/*
 * Set BioAPI GUI callback
 *
 * Returns 0 on success, non-zero on failure.
 */
int bioapi_set_gui_callback(BioAPI_HANDLE bspHandle, struct callbackh *ch)
{
    int bRet;
    BioAPI_RETURN ret = BioAPI_OK;

    ch->prevMsg = -1;
    bRet = BioAPI_SetGUICallbacks(bspHandle, NULL, NULL,
				  (void *) bioapi_guicallback, ch);
    if (bRet != BioAPI_OK) {
	bioapi_error_code("set GUI callback", bRet);
	return (1);
    }

    return ret;
}

/*
 * Send error code and operation which generated this code to application.
 *
 */
void
bioapi_error_code(const char *sOperation, const BioAPI_RETURN bioReturn)
{
    syslog(LOG_ALERT, "Operation %s failed, BioAPI Error Code: 0x%lXh\n",
	   sOperation, bioReturn);
    fprintf(stderr, "o: %s\n", sOperation);
    fprintf(stderr, "g: %d %Xh BioAPI Error\n", PAM_ERROR_MSG,
	    bioReturn);
}
