/*
   bioapi_chbird - helper program for the pam_bioapi module that
   biometricly verifies user. It is not intended to be run directly from
   the command line and logs a security violation if done so.

   Copyright (C) 2005 Michael R. Crusoe <michael at qrivy dot net>
   Copyright (C) 2006 Josef Hajas <josef at hajas dot net>

   This program is free software; you can redistribute it and/or modify it 
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any
   later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License along 
   with this program; if not, write to the Free Software Foundation, Inc., 
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 */

#include "bioapi_chbird.h"

static char*
parse_error_get_operation (char *msg, char **errorMsg)
{
    char *default_operation = strdup("pam authentisation");
    char *operation = NULL;
    int op_len = 0;
    char *end = 0;

    if ((msg == NULL) || 
        (msg[0] != 'O') ||
        (msg[1] != 'p') ||
        (msg[2] != 'e') ||
        (msg[3] != 'r') ||
        (msg[4] != 'a') ||
        (msg[5] != 't') ||
        (msg[6] != 'i') ||
        (msg[7] != 'o') ||
        (msg[8] != 'n') ||
        (msg[9] != ' ')) {
        operation = default_operation;
    } else {
        msg += 10;
        end = strstr (msg, " failed: "); 
	if (end != NULL) {
	    op_len = end - msg;
	    free (operation);
	    operation = strndup (msg, op_len);
	    msg = end + 9;
	}
    }
    *errorMsg = strdup (msg);

    return operation;
}

static int
pam_conversation(int nmsgs,
		 const struct pam_message **msg,
		 struct pam_response **resp, void *closure)
{
    int i;
    int rc = PAM_SUCCESS;
    int length = 0;
    char response[MAXMSG + 1];
    char *content;
    char *errorMsg, *sOperation;

    for (i = 0; i < nmsgs; i++) {
        if (msg[i]->msg_style == PAM_ERROR_MSG) {
	    sOperation = parse_error_get_operation ((char *) msg[i]->msg, &errorMsg);
	    fprintf(stderr, "o: %s\n", sOperation);
	    free (sOperation);
	    fprintf(stderr, "g: %d %s\n", PAM_ERROR_MSG, errorMsg);
	    free (errorMsg);
	} else {
	    fprintf(stderr, "g: %d %s\n", msg[i]->msg_style, msg[i]->msg);
	}
	if ((msg[i]->msg_style == PAM_PROMPT_ECHO_OFF)
	    || (msg[i]->msg_style == PAM_PROMPT_ECHO_ON)) {
	    if (fgets(response, sizeof(response), stdin) != NULL) {
		if ((response[0] == 'r') && (response[1] == ':')
		    && (response[2] == ' ')) {
		    response[MAXMSG] = '\0';	/* NULL terminate */
		    length = strlen(response);
		    response[length - 1] = '\0';	// without end of
		    // line
		    resp[i] = (struct pam_response *)
			malloc(sizeof(struct pam_response));
		    resp[i]->resp = strdup(response + 3);	// without 
		    // 'r: '
		    resp[i]->resp_retcode = 0;
		    rc = PAM_SUCCESS;
		} else {
		    syslog(LOG_ERR, "Wrong pam_bioapi protocol! Got: %s",
			   response);
		    //rc = PAM_CONV_ERR;
		    exit(PAM_SYSTEM_ERR);
		}
	    }
	}
    }
    return rc;

}

static char *getuidname(uid_t uid)
{
    struct passwd *pw;
    static char username[MAXUSERNAME + 1];

    pw = getpwuid(uid);
    if (pw == NULL)
	return NULL;

    strncpy(username, pw->pw_name, sizeof(username));
    username[sizeof(username) - 1] = '\0';

    return username;
}

static void setup_int_signal(void)
{
    struct sigaction action;

    (void) memset((void *) &action, 0, sizeof(action));
    action.sa_handler = su_sighandler;
    action.sa_flags = 0;
    (void) sigaction(SIGINT, &action, NULL); //for cancelation of biometric operation
}

static void su_sighandler(int sig)
{
#ifndef SA_RESETHAND
    /*
     * emulate the behaviour of the SA_RESETHAND flag 
     */
    if (sig == SIGILL || sig == SIGTRAP || sig == SIGBUS || sig = SIGSERV)
	signal(sig, SIG_DFL);
#endif
    if (sig == SIGINT) {
        cancel_bioapi = 1; //for cancelation of biometric operation
	setup_int_signal();
    } else  if (sig > 0) {
	syslog(LOG_NOTICE, "caught signal %d.", sig);
	exit(sig);
    }
}

static void setup_signals(void)
{
    struct sigaction action;	/* posix signal structure */

    /*
     * Setup signal handlers
     */
    (void) memset((void *) &action, 0, sizeof(action));
    action.sa_handler = su_sighandler;
#ifdef SA_RESETHAND
    action.sa_flags = SA_RESETHAND;
#endif
    (void) sigaction(SIGILL, &action, NULL);
    (void) sigaction(SIGTRAP, &action, NULL);
    (void) sigaction(SIGBUS, &action, NULL);
    (void) sigaction(SIGSEGV, &action, NULL);
    action.sa_flags = 0;
    action.sa_handler = SIG_IGN;
    (void) sigaction(SIGTERM, &action, NULL);
    (void) sigaction(SIGHUP, &action, NULL);
    //(void) sigaction(SIGINT, &action, NULL);
    (void) sigaction(SIGQUIT, &action, NULL);
    //(void) sigaction(SIGCHLD, &action, NULL);
    setup_int_signal();
}

char *getXAuthorityFile (void) {
    struct passwd *pw;
    pw = getpwuid(getuid());
    char *xauthfile = NULL;

    asprintf (&xauthfile, "%s/.Xauthority", pw->pw_dir);
    //free (pw);

    return xauthfile;
}

void parseArgs(int argc, const char **argv, options_t * options)
{
    extern int optind;
    int opt;
    const char *display = NULL;
    
    const struct option long_options[] = {
      { "bsp", 1, NULL, 'b' },
      { "backend", 1, NULL, 'B' },
      { "display", 1, NULL, 'd' },
      { "tries", 1, NULL, 't' },
      { "translation", 1, NULL, 'T' },
      { "ask-username", 1, NULL, 'u' },
      { "gui", 0, NULL, 'g' },
      { "payload-to-bir", 0, NULL, 'p' },
      { "show-all-messages", 0, NULL, 's' },
      { "power-save", 0, NULL, 'w' },
      { "debug", 0, NULL, 'D' },
      { NULL, 0, NULL, 0 }
    };

    /*
     * default values 
     */
    options->backendConf = DEFCONFPATH;
    options->useDriverGui = 0;	// false
    options->getUsername = 0;	// false
    options->payloadToBIR = 0;	// false
    options->showAllMessages = 0;	// false
    options->tries = 1;
    options->type = -1;
    options->uuidString = NULL;
    options->user = NULL;
    options->dbbackend = "sqlite3";

    optind = 1;

    if (options != NULL) {
	while ((opt =
		getopt_long (argc, (char **) argv, 
		  "b:B:d:t:T:u:gpswD", long_options, NULL)) != -1) {
	    switch (opt) {
		/*
		 * BSP UUID 
		 */
	    case 'b':
		options->uuidString = (char *) argv[optind - 1];
		if (debug == 1)
		    syslog(LOG_DEBUG, "BSP UUID: %s", options->uuidString);
		break;
		/*
		 * DB backend 
		 */
	    case 'B':
		options->dbbackend = (char *) argv[optind - 1];
		if (debug == 1)
		    syslog(LOG_DEBUG, "DB backend set to: %s",
			   options->dbbackend);
		break;
		/*
		 * display variable 
		 */
	    case 'd':
		display = (char *) argv[optind - 1];
		setenv("DISPLAY", display, 1);
		if (debug == 1)
		    syslog(LOG_DEBUG, "Setting DISPLAY variable to: %s",
			   display);
		break;
		/*
		 * tries 
		 */
	    case 't':
		options->tries = atoi(argv[optind - 1]);
		if (debug == 1)
		    syslog(LOG_DEBUG, "Got number of tries: %d",
			   options->tries);
		break;
		/*
		 * username specified 
		 */
	    case 'u':
		options->getUsername = 1;
		options->user = (char *) argv[optind - 1];
		if (debug == 1)
		    syslog(LOG_DEBUG, "Authenticate specified username");
		break;
		/*
		 * payload to BIR
		 */
	    case 'p':
		options->payloadToBIR = 1;
		if (debug == 1)
		    syslog(LOG_DEBUG, "Save payload to BIR");
		break;
		/*
		 * show all gui state messages even if they repeting
		 */
	    case 's':
		options->showAllMessages = 1;
		if (debug == 1)
		    syslog(LOG_DEBUG, "Show all messages");
		break;
		/*
		 * use driver build in gui and don't use pam_conv() 
		 */

	    case 'g':
		options->useDriverGui = 1;
		char *xauthority = getXAuthorityFile();
		setenv ("XAUTHORITY", xauthority, 1);
		if (debug == 1) {
		    syslog(LOG_DEBUG,
			   "Use driver's GUI instead of pam_conv()");
		    syslog(LOG_DEBUG,
			   "XAUTHORITY set to %s",xauthority);
	        }
		free (xauthority);
		break;
	    case 'D':
		debug = 1;
		break;
	    }
	}
    } else {
	syslog(LOG_ERR,
	       "Internal application error: can't parse pam module arguments!");
	exit(PAM_SYSTEM_ERR);
    }
}

void listEnrolledRecords (dbstuff_t dbstuff, const char* user)
{
    int i;
    struct birdb_rec keyrec;
    struct birdb_rec **recs = NULL;
    
    keyrec.br_key = (char *) user;
    keyrec.br_type = BioAPI_NOT_SET;
    recs = birdb_backend_get(dbstuff.bm, dbstuff.beh, &keyrec);
    fprintf (stderr, "l:");
    if (recs) {
      for (i = 0; recs[i] != NULL; i++) {
	  fprintf (stderr, " %d", recs[i]->br_type);
      }
    }
    fprintf (stderr, "\n");
    fflush(stderr);
    birdb_backend_freegetres(dbstuff.bm, dbstuff.beh, recs);
}

char getAction()
{
    char action = '\0';
    char line[MAXMSG];

    if (!feof(stdin)
	   && !ferror(stdin) && (fgets(line, sizeof(line), stdin) != NULL)) {
	if ((line[0] == 'a') && (line[1] == ':') && (line[2] == ' ')) {
	    action = line[3];
	    if (debug)
		syslog(LOG_DEBUG, "Got action: '%c'", action);
	} else {
	    line[MAXMSG] = '\0';
	    syslog(LOG_ERR, "Wrong pam_bioapi protocol! Got: %s", line);
	    exit (PAM_AUTHTOK_ERR);
	}
    }
    return action;
}

int getType()
{
    int success = 0;
    int type = -1;
    char line[MAXMSG+1];
    
    if (!feof(stdin)
	   && !ferror(stdin) && (fgets(line, sizeof(line), stdin) != NULL)) {
	line[MAXMSG] = '\0';
	success = sscanf (line, "t: %d", &type);
	if (! success) {
	    syslog(LOG_ERR, "Wrong pam_bioapi protocol! Got: %s", line);
	    exit (PAM_AUTHTOK_ERR);
	}
    }
    return type;
}

char *getPayload()
{
    int success = 0;
    int type = -1;
    char line[MAXMSG+1];
    char *beginOfPayload = NULL;
    char *payload = NULL;
    
    /* ask parent payload */
    fprintf(stderr, "p?\n");
    fflush (stderr);
    
    if (!feof(stdin)
	   && !ferror(stdin) && (fgets(line, sizeof(line), stdin) != NULL)) {
	line[MAXMSG] = '\0';
	if (! ((line[0] == 'p') && (line[1] == ':') && (line[2] == ' ')) ) {
	    syslog(LOG_ERR, "Wrong pam_bioapi protocol! Got: %s", line);
	    exit (PAM_AUTHTOK_ERR);
	}
	beginOfPayload = line+3;
	if (line[0] != '\0') {
	  payload = strdup (beginOfPayload);
    	}
    }
    return payload;
}

int verifyUserByPam(const char *user)
{
    pam_handle_t *pamh = NULL;
    struct pam_conv conv;
    int retcode;

    if (debug == 1)
	syslog(LOG_DEBUG, "verifyUserByPam (%s);", user);

    if (getuid() == 0) {
	if (debug == 1)
	    syslog(LOG_DEBUG, "username (uid==0): %s.", user);
	retcode = PAM_SUCCESS;
    } else {
	char *realUser = (char *) getuidname(getuid());
	if (debug == 1)
	    syslog(LOG_DEBUG, "username (uid!=0): %s.", user);
	if (strcmp(realUser, user)) {
	    syslog(LOG_ALERT,
		   "User %s want to authenticate as (diffrent user) %s!",
		   realUser, user);
	    closelog();
	    return PAM_AUTH_ERR;
	}
	fprintf(stderr, "g: %d Verifying user identity\n", PAM_TEXT_INFO);
	conv.conv = &pam_conversation;
	conv.appdata_ptr = NULL;

	retcode = pam_start("bioapi_chbird", user, &conv, &pamh);

	if (retcode == PAM_SUCCESS)
	    retcode = pam_authenticate(pamh, 0);
	if (retcode == PAM_SUCCESS)
	    retcode = pam_acct_mgmt(pamh, 0);

	if (pam_end(pamh, retcode) != PAM_SUCCESS) {	/* close Linux-PAM 
							 */
	    pamh = NULL;
	    syslog(LOG_ERR,
		   "Failed to release authenticator since verifing user identity.");
	}

	if (retcode != PAM_SUCCESS) {
	    fprintf(stderr, "g: %d Access denied!\n", PAM_TEXT_INFO);
	}
    }

    return retcode;
}

/*
   call birdb_init(), birdb_cfgparse() birdb_addmod() and
   birdb_backend_open(). Also hendle errors. 
 */
int initDbBackend(dbstuff_t * dbstuff, options_t options)
{
    int error;

    dbstuff->bdb = birdb_init();
    if (dbstuff->bdb == NULL) {
	syslog(LOG_ALERT, "Unable to initilize database backend!");
	fprintf(stderr, "g: %d Unable to initilize database backend!"
		"Is libbirdb.so really installed?\n",
		PAM_ERROR_MSG, options.dbbackend);
	return PAM_AUTH_ERR;
    }
    error = birdb_cfgparse(dbstuff->bdb, options.backendConf);
    if (error < 0) {
	syslog(LOG_ALERT,
	       "Unable to parse backend config file [%s]!",
	       options.backendConf);
	fprintf(stderr,
		"g: %d Unable to parse backend config file [%s]!\n",
		PAM_ERROR_MSG, options.backendConf);
	return PAM_AUTH_ERR;
    }

    dbstuff->bm = birdb_findmod(dbstuff->bdb, options.dbbackend);
    if (dbstuff->bm == NULL) {
	syslog(LOG_ALERT, "Unable to find backend %s", options.dbbackend);
	fprintf(stderr,
		"g: %d Unable to find backend %s. "
		"Is this backend really installed?\n", PAM_ERROR_MSG,
		options.dbbackend);
	return PAM_AUTH_ERR;
    }

    dbstuff->beh = birdb_backend_open(dbstuff->bm, options.uuidString,
				      dbstuff->bm->bm_argc,
				      dbstuff->bm->bm_argv);
    if (dbstuff->beh == NULL) {
	syslog(LOG_ALERT, "Unable to open module %s. "
	       "Does path for database defined in birdb config really exist?",
	       options.dbbackend);
	fprintf(stderr, "g: %d Unable to open module %s. "
		"Does path for database defined in birdb config really exist?\n",
		PAM_ERROR_MSG, options.dbbackend);
	return PAM_AUTH_ERR;
    }
    return PAM_SUCCESS;
}

/*
   Ask user if he really want to delete this record and if so delete given 
   record from DB. return 0 if delete were really performed. 
 */
int
deleteFromDb(dbstuff_t dbstuff, struct birdb_rec *rec, const int purpose)
{
    int notDeleted = 1;
    int agreed = 1;
    char line[MAXMSG];


    if (purpose) {
      agreed = 0;
      fprintf(stderr, "d? %d\n", rec->br_type);
      fflush (stderr);

      if (!feof(stdin)
	     && !ferror(stdin) && (fgets(line, sizeof(line), stdin) != NULL)) {
	  if ((line[0] == 'r') && (line[1] == ':') && (line[2] == ' ')) {
	      if (line[3] == 'y') {
		agreed = 1;
	      }
	  } else {
	      syslog(LOG_ERR, "Wrong pam_bioapi protocol! Got: %s", line);
	      exit (PAM_SYSTEM_ERR);
	  }
      } 
    }
    if (agreed) {
      notDeleted = birdb_backend_del(dbstuff.bm, dbstuff.beh, rec);
    }
    return notDeleted;
}

/*
   Check if enrolled bir insn't already in database and if so ask user if
   he wants to delete an old record, but only if found record is attached
   to same user. In case found with other user's biometrics record found
   deny adding of this record. 
     Arguments in: dbstuff - backend handle and module 
               in: bspHandle - bioapi handle to loaded module 
               in: rec - record with username and bir set. It is used for
   comparing with found record. 
   
   Returns 0 if not allowed (found bir in
   other users records or user not agree with delete of previous record)
   or 1 if it is allowed. 
 */
int
isInsertAllowed(struct birdb_rec *rec, dbstuff_t dbstuff, BioAPI_HANDLE bspHandle)
{
    int notFoundOrUserAgree = 1;
    struct birdb_rec *foundRec = bioapi_bir_in_db(dbstuff, bspHandle, rec->br_bir);

    if (foundRec != NULL) {
	if (!strncmp(foundRec->br_key, rec->br_key, MAXUSERNAME)) {
	    notFoundOrUserAgree =
		!deleteFromDb (dbstuff, foundRec, 1);
	    if (notFoundOrUserAgree) {
		fprintf(stderr, 
		    "g: %d Record successfully deleted -> new record saved.\n",
			PAM_TEXT_INFO);
	    } else {
		fprintf(stderr, 
		    "g: %d Record wasn't deleted -> new record save canceled.\n",
			PAM_TEXT_INFO);
	    }
	} else {
	    notFoundOrUserAgree = 0;
	    fprintf(stderr,
		    "g: %d Record found for other user! Insert denied!\n",
		    PAM_TEXT_INFO);
	}
	birdb_freerec(foundRec);
    }

    return notFoundOrUserAgree;
}

/*
   Determine what the current user's name is. On a SELinux enabled system
   with a strict policy leaving the existing check prevents shadow
   password authentication from working. We must thus skip the check if
   the real uid is 0. 
 */
int checkUserIdentity(const char *user)
{
    int ret = PAM_SUCCESS;
    int uid = getuid();
    if (uid == 0) {
	if (debug == 1)
	    syslog(LOG_DEBUG, "username (uid==0): %s.", user);
    } else {
	char *realUser = (char *) getuidname(uid);
	if (debug == 1)
	    syslog(LOG_DEBUG, "username (uid!=0): %s.", user);
	if (strcmp(realUser, user)) {
	    syslog(LOG_ALERT,
		   "User [%s] want to authenticate as (diffrent user) [%s]!",
		   realUser, user);
	    ret = PAM_AUTH_ERR;
	}
    }
    return ret;
}

int verifyUser (char *user, int tries, 
    dbstuff_t dbstuff, BioAPI_HANDLE *bspHandle) {
  int error = 1;
  int i;
  char *payload = NULL;
    
  if (debug)
	syslog(LOG_ALERT, "Begin of verification");
  do {
      error =
	  bioapi_identify(bspHandle, dbstuff.bm, dbstuff.beh, &user, &payload);
      if ((! error) && (payload)) {
	  fprintf(stderr, "p: %s\n", payload);
	  tries = 0;
	  for (i = 0; i < strlen(payload); i++) payload[i] = 'x';
	  //free (payload);
      }
      if (cancel_bioapi) tries = 0;

  tries--;
  }
  while (tries > 0);

  return error;
}

int identify (int tries, 
    dbstuff_t dbstuff, BioAPI_HANDLE *bspHandle) {
  int error = 1;
  int i;
  char *user = NULL;
  char *payload = NULL;
    
  if (debug)
	syslog(LOG_ALERT, "Begin of identification");
  do {
      error = bioapi_identify(bspHandle, dbstuff.bm, dbstuff.beh, &user, &payload);
      if (! error) {
	  fprintf(stderr, "u: %s\n", user);
	  if (payload) {
	    fprintf(stderr, "p: %s\n", payload);
            for (i = 0; i < strlen(payload); i++) payload[i] = 'x';
	    free (payload);
	  }
	  tries = 0;
      }
      if (cancel_bioapi) tries = 0;

  tries--;
  }
  while (tries > 0);

  return error;
}


int
authenticate (options_t options, BioAPI_HANDLE *bspHandle,
	     dbstuff_t dbstuff)
{
    int i, error;
    BioAPI_RETURN bRet;
    int ret;

    if (debug)
	syslog(LOG_ALERT, "called authenticate()");

    if (options.getUsername == 1) {
	error = checkUserIdentity(options.user);
	if (error != PAM_SUCCESS) {
	    return error;
	}
    }
    error = 1;
    if (options.getUsername == 1) {
	error = verifyUser (options.user, options.tries, dbstuff, bspHandle);
    } else {
	error = identify (options.tries, dbstuff, bspHandle);
    }
    if (!error) {
	ret = PAM_SUCCESS;
    } else {
	ret = PAM_AUTH_ERR;
    }

    return ret;

}

int enroll (options_t options, BioAPI_HANDLE * bspHandle, dbstuff_t dbstuff)
{
    int i, error = 1;
    //int tries = options.tries;
    int tries = 1;

    struct birdb_rec *rec;

    do {
      BioAPI_BIR_PTR bir = bioapi_capture_for_enroll (bspHandle);
      if (bir == NULL)
	break;
      
      options.payload = getPayload();

      rec = bioapi_enroll (bspHandle, bir, options.user, options.type, 
	  options.payloadToBIR, options.payload);

      if (rec) {
	if (isInsertAllowed (rec, dbstuff, *bspHandle)) {
	    error = birdb_backend_ins(dbstuff.bm, dbstuff.beh, rec);
	    birdb_freerec(rec);
	}
	break;
      }
      if (cancel_bioapi) tries = 0;
      tries--;
    }
    while (tries > 0);

    if (error) {
	return PAM_AUTHTOK_ERR;
    } else {
	return PAM_SUCCESS;
    }
}

int main(int argc, const char **argv)
{
    options_t options;
    char *action = NULL;
    char op = '\0';
    BioAPI_HANDLE *bspHandlePtr = NULL;
    dbstuff_t dbstuff;
    int error;
    struct callbackh ch;
    int ret = PAM_AUTH_ERR;
    BioAPI_RETURN bRet;

    /*
     * Catch or ignore as many signal as possible.
     */
    cancel_bioapi = 0;
    setup_signals();

    openlog("bioapi_chbird", LOG_PID | LOG_CONS, LOG_AUTHPRIV);

    if (debug == 1)
	syslog(LOG_DEBUG, "Starting bioapi_chbird process");

    /*
     * we establish that this program is running with non-tty stdin.
     * this is to discourage casual use. It does *NOT* prevent an
     * intruder from repeatadly running this program to determine the
     * password of the current user (brute force attack, but one for
     * which the attacker must already have gained access to the user's
     * account).
     */

    if (isatty(STDIN_FILENO) && (!debug)) {
	syslog(LOG_NOTICE,
	       "inappropriate use of BioAPI PAM helper binary [UID=%d]",
	       getuid());
	fprintf(stderr,
		"This binary is not designed for running in this way\n"
		"-- the system administrator has been informed\n");
	sleep(10);		/* this should discourage/annoy the user */
	closelog();
	return PAM_SYSTEM_ERR;
    }

    action = (char *) argv[1];
    parseArgs(argc, argv, &options);

    error = initDbBackend(&dbstuff, options);
    if (error != PAM_SUCCESS) {
	goto main_error2;
    }

    error = bioapi_init();
    if (error) {
	goto main_error2;
    }

    bspHandlePtr = bioapi_attach_bsp(options.uuidString);
    if (bspHandlePtr == NULL) {
	goto main_error1;
    }

    ch.showAll = options.showAllMessages;
    if (options.useDriverGui == 0) {
	error = bioapi_set_gui_callback (*bspHandlePtr, &ch);
	if (error) 
	  goto main_out;
    }

    if (debug == 1)
	syslog(LOG_DEBUG, "Action: %s", action);

    if (action[0] == 'a') {

	ret = authenticate (options, bspHandlePtr, dbstuff);

	if (debug) {
	     if (ret == PAM_SUCCESS) {
	       syslog(LOG_DEBUG, "User successfully verified");
	     } else {
	       syslog(LOG_DEBUG, "User filed to verify");
	     }
	}
    } else {
	ret = verifyUserByPam(options.user);
	if (ret == PAM_SUCCESS) {
	   if (debug == 1)
	     syslog(LOG_DEBUG, "User [%s] successfully verified",
		     options.user);
	   while (op != 'x') {
	      // send to pam_bioapi which fingers are enrolled
	      listEnrolledRecords (dbstuff, options.user);
	      fprintf (stderr, "a?\n");
	      op = getAction();
	      // get type from calling pam_module (from stdin)
	      switch (op) {
	        case 'e':
	          options.type = getType();
		  ret = enroll (options, bspHandlePtr, dbstuff);
		  break;
	        case 'd':
	          options.type = getType();
		  struct birdb_rec rec;
		  rec.br_ctime = 0;
		  rec.br_key = options.user;
		  rec.br_type = options.type;
		  deleteFromDb (dbstuff, &rec, 0);
		  break;
	        case 'x':
	          syslog(LOG_DEBUG, "bioapi_chbird exiting");
		  ret = PAM_SUCCESS;
		  op = 'x';
		  break;
		default:
	          syslog (LOG_ERR, "Unknown action [%c]",op);
		  goto main_out;
	      }
	   }
	}
    }

  main_out:
    bioapi_detach_bsp(bspHandlePtr, options.uuidString);
  main_error1:
    bioapi_destroy();
  main_error2:
    birdb_backend_close(dbstuff.bm, dbstuff.beh);
    birdb_close(dbstuff.bdb);
    
    closelog();
    fclose(stderr);
    return ret;
}

