/*-
 * birdb_sqlite3 - Backend for pam-bioapi used to store BIR's in SQLite3 db.
 *
 * Copyright (C) 2006 Josef Hajas <josef at hajas dot net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/*
 * MySQL backend to libbirdb
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/queue.h>
#include <fcntl.h>
#include <limits.h>
#include <errno.h>
#include <err.h>
#include <syslog.h>
#include <bioapi.h>
#include <sqlite3.h>
#include <pwd.h>

#include <birdb.h>

static void *sqlite3db_open (const char *, int, char *[]);
static void sqlite3db_close (void *);
static struct birdb_rec **sqlite3db_get (void *, struct birdb_rec *);
static void sqlite3db_freegetres (void *, struct birdb_rec **);
static int sqlite3db_ins (void *, struct birdb_rec *);
static int sqlite3db_del (void *, struct birdb_rec *);
static struct birdb_rec *sqlite3db_seq_getfirst (void *, struct birdb_rec *);
static struct birdb_rec *sqlite3db_seq_getnext (void *, struct birdb_rec *);
static void sqlite3db_seq_free (void *handle, struct birdb_rec *);

static BioAPI_BIR *buf2bir (char *, size_t);
static int getNextRecord (sqlite3_stmt *, struct birdb_rec **);
static char *composeQuery (const char *, const char *, struct birdb_rec *);

/*
 * Default database "birdb", default table "bir"
 */
#define DEFPATH	 "/etc/bioapi/pam/bioshadow.db"
#define DEFTABLE "biodata"
#define MAXSQLSELECT 300

/* Backend module glue */
BIRDB_BACKEND_PARAMS =
{
"sqlite3",
    "SQLite3 backend",
    sqlite3db_open,
    sqlite3db_close,
    sqlite3db_get,
    sqlite3db_freegetres,
    sqlite3db_ins,
    sqlite3db_del,
    sqlite3db_seq_getfirst, 
    sqlite3db_seq_getnext, 
    sqlite3db_seq_free,};

/*
 * sqlite3 handle
 */
struct sqlite3db_handle
{
  sqlite3 *sqlite3;
  char *bspid;
  sqlite3_stmt *queryResult;
  struct
  {
    char *path;
    char *table;
  } db;
};

/*
 * Open and initialization routine
 * Arguments
 *   bspid - BSP uuid
 *	 argc - Number of arguments
 *	 argv - Argument array
 * 
 * Requires two argument
 *  path and table
 * both might be left out for default or blank values. 
 * 
 * If there is no database found in path, new one is created.
 *
 * Returns a sqlite3 handle on success, otherwise NULL.
 *
 */
static void *
sqlite3db_open (const char *bspid, int argc, char *argv[])
{
  struct sqlite3db_handle *mh;
  sqlite3 *db;
  int isNewFile = 0;		//false
  struct stat status;
  char *query;
  char *zErrMsg;
  int rc;

  if (bspid == NULL)
    {
      syslog (LOG_ALERT, "sqlite3db backend - Missing BSP ID");
      return (NULL);
    }

  mh = malloc (sizeof (struct sqlite3db_handle));
  if (mh == NULL)
    return (NULL);

  mh->bspid = strdup (bspid);
  if (mh->bspid == NULL)
    goto error1;

  mh->db.path = (argc == 0) ? strdup (DEFPATH) : strdup (argv[0]);

  if (mh->db.path == NULL)
    {
      syslog (LOG_ALERT, "sqlite3db backend - Missing database path");
      goto error2;
    }

  mh->db.table = (argc < 2) ? strdup (DEFTABLE) : strdup (argv[1]);

  if (mh->db.table == NULL)
    {
      syslog (LOG_ALERT, "sqlite3db backend - Missing database table");
      goto error2;
    }

  (void) stat (mh->db.path, &status);
  // Is path provided regular file?
  if (!S_ISREG (status.st_mode))
    {
      if (S_ISDIR (status.st_mode))
	{
	  syslog (LOG_ALERT,
		  "sqlite3db backend - Directory specified! I need file!");
	  goto error2;
	}
      // we have to make new table
      isNewFile = 1;
    }

  // open database
  rc = sqlite3_open (mh->db.path, &db);
  if (rc)
    {
      syslog (LOG_ALERT,
	      "Error since opening file [%s]: %s. Make sure at least path exist!\n",
	      mh->db.path, sqlite3_errmsg (db));
      goto error2;
    }

  if (isNewFile)
    {
      chmod (mh->db.path, S_IRUSR | S_IWUSR);
      asprintf (&query, "CREATE TABLE %s "
		"(id INTEGER PRIMARY KEY,user TEXT, uuid TEXT,header BLOB,"
		"data BLOB,signature BLOB,type INTEGER,other BLOB);",
		mh->db.table);
      if (query == NULL)
	goto error3;
      //syslog (LOG_DEBUG, "DEUG: %s", query);
      rc = sqlite3_exec (db, query, NULL, NULL, &zErrMsg);
      if (rc != SQLITE_OK)
	{
	  syslog (LOG_ALERT, "SQL error while creating table biodata: %s\n",
		  zErrMsg);
	  free (query);
	  goto error3;
	}
    }
  mh->sqlite3 = db;

  return (mh);
error3:
  sqlite3_close (db);
  if (zErrMsg)
    sqlite3_free (zErrMsg);
error2:
  free (mh->bspid);
error1:
  free (mh);
  return (NULL);
}

/*
 * Close a sqlite3 handle
 */
static void
sqlite3db_close (void *handle)
{
  struct sqlite3db_handle *mh = handle;
  sqlite3 *db = mh->sqlite3;

  sqlite3_close (db);
  free (mh->db.path);
  free (mh->db.table);
  free (mh->bspid);
  free (mh);
}

/*
 * Get a records from the database
 * Arguments
 *     handle - sqlite3 handle (defined in birdb.h)
 *     br - BIRdb record (only br_key and br_bspid is used
 *     but only br_bspid is mandatory)
 *
 * Returns an array of birdb records that matches the given key.
 * This array must be freed with freegetres()
 */
static struct birdb_rec **
sqlite3db_get (void *handle, struct birdb_rec *br)
{
  struct sqlite3db_handle *mh = handle;
  struct birdb_rec **recarray = NULL;
  struct birdb_rec *rec = NULL;
  int error, sz, idx;
  sqlite3 *db = mh->sqlite3;
  sqlite3_stmt *queryResult;
  char *query;

  query = composeQuery (mh->db.table, mh->bspid, br);

  if (query == NULL)
    return (NULL);

  error = sqlite3_prepare (db, query, -1, &queryResult, NULL);
  //free(query);
  sqlite3_free (query);
  if ((error != 0) || (queryResult == NULL))
    return (NULL);

  sz = idx = 0;
  for (error = getNextRecord (queryResult, &rec);
       error != SQLITE_DONE; error = getNextRecord (queryResult, &rec))
    {

      sz += sizeof (struct birdb_rec *);
      recarray = realloc (recarray, sz);
      if (recarray == NULL)
	goto fail;

      recarray[idx] = rec;
      //rec->br_key = strdup (br->br_key);
      if (rec->br_key == NULL)
	goto fail;

      rec = NULL;
      idx++;
    }

  sqlite3_finalize (queryResult);

  /*
   * Add a final NULL entry in the array to indicate the end
   */
  if (recarray != NULL)
    {
      sz += sizeof (struct birdb_rec *);
      recarray = realloc (recarray, sz);
      if (recarray == NULL)
	goto fail;
      recarray[idx] = NULL;
    }

  return (recarray);
fail:
  sqlite3db_freegetres (handle, recarray);
  return (NULL);
}

/*
 * Free results returned by fdb_get()
 * Arguments
 *     handle - filedb handle 
 *     recarray - result array returned by fdb_get 
 */
static void
sqlite3db_freegetres (void *handle, struct birdb_rec **recarray)
{
  int idx = 0;

  if (recarray == NULL)
    return;

  while (recarray[idx] != NULL)
    {
      free (recarray[idx++]);
    }
  free (recarray);
}

/*
 * Insert a record into the database
 * Arguments
 *     handle - sqlite3 handle
 *     br - the birdb record to be inserted 
 *
 * Returns 0 on success, non-zero on failure.
 */
static int
sqlite3db_ins (void *handle, struct birdb_rec *br)
{
  struct sqlite3db_handle *mh = handle;
  sqlite3 *db = mh->sqlite3;
  BioAPI_BIR *bir = br->br_bir;
  sqlite3_stmt *queryResult = NULL;
  int rc;
  char *query = NULL;
  char *bsp;
  //int uid = user2uid (br->br_key);

  query =
    sqlite3_mprintf
    ("insert into %s (uuid,user,header,data,signature,other,type)"
     "values('%q', '%q', ?, ?, ?, ?, %d);", strdup (mh->db.table),
     strdup (mh->bspid), strdup (br->br_key), br->br_type);
  if (query == NULL)
    return (-1);

  rc = sqlite3_prepare (db, query, -1, &queryResult, NULL);
  sqlite3_free (query);
  if ((rc != SQLITE_OK) || (queryResult == NULL))
    {
      goto sqlite3db_ins_error;
    }

  rc =
    sqlite3_bind_blob (queryResult, 1, &(bir->Header),
		       sizeof (BioAPI_BIR_HEADER), SQLITE_STATIC);
  if (rc != SQLITE_OK)
    {
      goto sqlite3db_ins_error;
    }

  rc =
    sqlite3_bind_blob (queryResult, 2, bir->BiometricData,
		       bir->Header.Length - sizeof (BioAPI_BIR_HEADER),
		       SQLITE_STATIC);
  //free);
  if (rc != SQLITE_OK)
    {
      goto sqlite3db_ins_error;
    }

  if (bir->Signature)
    {
      rc =
	sqlite3_bind_blob (queryResult, 3, bir->Signature->Data,
			   (size_t) bir->Signature->Length, free);
      if (rc != SQLITE_OK)
	{
	  goto sqlite3db_ins_error;
	}
    }
  else
    {
      rc = sqlite3_bind_blob (queryResult, 3, NULL, 0, SQLITE_STATIC);
      if (rc != SQLITE_OK)
	{
	  goto sqlite3db_ins_error;
	}
    }

  if (br->br_other)
    {
      rc =
	sqlite3_bind_blob (queryResult, 4, br->br_other->Data,
			   (int) br->br_other->Length, free);
      if (rc != SQLITE_OK)
	{
	  goto sqlite3db_ins_error;
	}
    }
  else
    {
      rc = sqlite3_bind_blob (queryResult, 4, NULL, 0, SQLITE_STATIC);
      if (rc != SQLITE_OK)
	{
	  goto sqlite3db_ins_error;
	}
    }

  rc = sqlite3_step (queryResult);
  if (rc != SQLITE_DONE)
    {
      goto sqlite3db_ins_error;
    }


  rc = sqlite3_finalize (queryResult);
  if (rc != SQLITE_OK)
    {
      goto sqlite3db_ins_error2;
    }

  return (0);
sqlite3db_ins_error:
  syslog (LOG_ALERT,
	  "SQL error while inserting data into table: %s\n",
	  sqlite3_errmsg (db));
  rc = sqlite3_finalize (queryResult);
  //free (query);
sqlite3db_ins_error2:
  return (-1);
}

/*
 * Remove a record from the database
 * Arguments
 *     handle - sqlite3 handle
 *     br - the birdb record to be removed 
 *
 * Returns 0 on success, non-zero on failure.
 */
static int
sqlite3db_del (void *handle, struct birdb_rec *br)
{
  struct sqlite3db_handle *mh = handle;
  sqlite3 *sqlite3 = mh->sqlite3;
  char *query, *errorMsg;
  int error = SQLITE_BUSY;

  if (br->br_ctime > 0)
    {
      query = sqlite3_mprintf ("delete from %q where user = '%q'"
			       " and id = %d;",
			       mh->db.table, br->br_key, br->br_ctime);
    }
  else
    {
      query = sqlite3_mprintf ("delete from %q where user = '%q'"
			       " and type = %d;",
			       mh->db.table, br->br_key, br->br_type);
    }

  if (query == NULL)
    return (-1);

  //syslog (LOG_DEBUG, "DEUG: %s", query);
  while (error == SQLITE_BUSY)
    error = sqlite3_exec (sqlite3, query, NULL, NULL, &errorMsg);
  free (query);
  if (error != SQLITE_OK)
    {
      syslog (LOG_ALERT, "SQL error while removing data from table %s: %s\n",
	      mh->db.table, errorMsg);
      sqlite3_free (query);
      sqlite3_free (errorMsg);
      return (-1);
    }

  return (0);
}

/*
 * Get the first record of the database for sequential traversal 
 * Arguments
 *     handle - sqlite3 handle
 *     br - the birdb record to be removed 
 *
 * Returns the first record if any, NULL otherwise 
 */
static struct birdb_rec *
sqlite3db_seq_getfirst (void *handle, struct birdb_rec *br)
{
  struct sqlite3db_handle *mh = handle;
  struct birdb_rec *rec = NULL;
  sqlite3 *db = mh->sqlite3;
  sqlite3_stmt *queryres;
  int error;
  char *query;

  query = composeQuery (mh->db.table, mh->bspid, br);
  //syslog (LOG_DEBUG, "DEUG: %s", query);

  error = sqlite3_prepare (db, query, -1, &queryres, NULL);
  sqlite3_free (query);

  //free(query);
  if ((error != 0) || (queryres == NULL))
    {
      return (NULL);
    }

  error = getNextRecord (queryres, &rec);
  //syslog (LOG_DEBUG, "DEUG: %d (%d), 0x%Xh", error, SQLITE_DONE, rec);
  if (error == SQLITE_DONE)
    {
      return (NULL);
    }

  mh->queryResult = queryres;

  return (rec);
}

/*
 * Get the next logical record for use with sequential traversial 
 * Arguments
 *     handle - sqlite3db handle
 *     prev - previous record from getnext or getfirst 
 *
 * Returns the next record if any, NULL otherwise 
 * Resources allocated for the previous record will be freed.
 */
static struct birdb_rec *
sqlite3db_seq_getnext (void *handle, struct birdb_rec *prev)
{
  struct sqlite3db_handle *mh = handle;
  struct birdb_rec *rec = NULL;
  sqlite3 *sqlite3 = mh->sqlite3;
  int error;
  char *query, *user;

  if (mh->queryResult == NULL)
    return (NULL);
  error = getNextRecord (mh->queryResult, &rec);
  if ((error == -1) || (rec == NULL))
    return (NULL);

  return (rec);
}

/*
 * Free the remains of an sequential traversial
 * Arguments
 *     handle - sqlite3 handle
 *     last - previous record from getnext or getfirst 
 * 
 * This is not needed if getnext returns NULL
 */
static void
sqlite3db_seq_free (void *handle, struct birdb_rec *last)
{
  struct sqlite3db_handle *mh = handle;

  sqlite3_finalize (mh->queryResult);
  mh->queryResult = NULL;
  if (last != NULL)
    birdb_freerec (last);
}

/* from queryResult get next row
 * in: queryResult - queryHandle 
 * out: record on success and NULL on any problem
 *
 * return sqlite3 return code.
 * */
static int
getNextRecord (sqlite3_stmt * queryResult, struct birdb_rec **record)
{
  void *data = NULL;
  int rc = SQLITE_BUSY;
  int sqltype = SQLITE_NULL;
  BioAPI_BIR *tempBir;

  while (rc == SQLITE_BUSY)	// until database is locked
    rc = sqlite3_step (queryResult);

  if (rc == SQLITE_ROW)
    {
      // get uid from query
      if (sqlite3_column_type (queryResult, 0) == SQLITE_TEXT)
	{
	  *record = (struct birdb_rec *) malloc (sizeof (struct birdb_rec));
	  (*record)->br_key = (char *) sqlite3_column_text (queryResult, 0);
	}
      else
	{
	  return (-1);
	}

      tempBir = (BioAPI_BIR *) malloc (sizeof (BioAPI_BIR));
      // get BIR from query
      if (sqlite3_column_type (queryResult, 1) == SQLITE_BLOB)
	{
	  data = (void *) sqlite3_column_blob (queryResult, 1);
	}
      else
	{
	  goto error_nextRecord;
	}
      memcpy (&(tempBir->Header), (BioAPI_BIR_HEADER *) data,
	      sizeof (BioAPI_BIR_HEADER));
      //free (data);

      if (sqlite3_column_type (queryResult, 2) == SQLITE_BLOB)
	{
	  data = (void *) sqlite3_column_blob (queryResult, 2);
	}
      else
	{
	  goto error_nextRecord;
	}
      tempBir->BiometricData = (BioAPI_BIR_BIOMETRIC_DATA_PTR) data;

      sqltype = sqlite3_column_type (queryResult, 3);
      if (sqltype == SQLITE_BLOB)
	{
	  data = (void *) sqlite3_column_blob (queryResult, 3);

	  tempBir->Signature =
	    (BioAPI_DATA_PTR) malloc (sizeof (BioAPI_DATA));
	  if (tempBir->Signature == NULL)
	    {
	      syslog (LOG_ALERT,
		      "Unable to allocate memory for BIR signature.");
	      free (tempBir->BiometricData);
	      goto error_nextRecord;
	    }
	  tempBir->Signature->Length =
	    (uint32) sqlite3_column_bytes (queryResult, 3);
	  tempBir->Signature->Data = data;
	}
      else if (sqltype == SQLITE_NULL)
	{
	  tempBir->Signature = NULL;
	}
      else
	{
	  goto error_nextRecord;
	}

      (*record)->br_bir = tempBir;

      // get record id (which finger) from query
      if (sqlite3_column_type (queryResult, 4) == SQLITE_INTEGER)
	{
	  (*record)->br_type = sqlite3_column_int (queryResult, 4);
	}
      else
	{
	  goto error_nextRecord;
	}

      // get payload from query
      if (sqlite3_column_type (queryResult, 5) == SQLITE_BLOB)
	{
	  (*record)->br_other = malloc (sizeof (br_other_t));
	  (*record)->br_other->Length = sqlite3_column_bytes (queryResult, 5);
	  (*record)->br_other->Data = (void *) sqlite3_column_blob (queryResult, 5);
	}

      // get record id (which finger) from query
      if (sqlite3_column_type (queryResult, 6) == SQLITE_INTEGER)
	{
	  (*record)->br_ctime = sqlite3_column_int (queryResult, 6);
	}
      else
	{
	  goto error_nextRecord;
	}

    }
  return rc;
error_nextRecord:
  free (tempBir);
error_nextRecord2:
  free (*record);
  *record = NULL;
  return (-1);

}

/* Compose query string
 * in: table, bspid, struct birdb_rec
 *
 * return string of first part of select
 * */
static char *
composeQuery (const char *table, const char *bspid, struct birdb_rec *br)
{
  char *query;

  query =
    sqlite3_mprintf ("SELECT user, header, data, signature, type, other, id "
		     "FROM %s WHERE uuid = '%s'", table, bspid);

  if (br->br_key)
    {
      query = sqlite3_mprintf ("%s AND user = '%q'", query, br->br_key);
    }

  if (br->br_type != BioAPI_NOT_SET)
    {
      query = sqlite3_mprintf ("%s AND type = %d", query, br->br_type);
    }
  query = sqlite3_mprintf ("%s;", query);

  return query;
}
