/* pam_sieĝo: PAM modulo por defendi vian kastelon
 * Copyright (C) 2005 Maarten Deprez
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


#define _GNU_SOURCE
#include <stdio.h>
#include <stdarg.h>
#include <syslog.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <time.h>

#define PAM_SM_AUTH

#include <security/pam_modules.h>
#include <security/_pam_macros.h>

#define UTF8 1

#ifdef UTF8
static char *supozata_regularo = "/etc/sieĝo";
static char *supozata_notlibro = "/var/log/sieĝo";
#else
static char *supozata_regularo = "/etc/siegho";
static char *supozata_notlibro = "/var/log/siegho";
#endif

typedef enum
  {
    REKONAJHO_ANONCO = 1, /* user */
    REKONAJHO_MILITISTO = 2, /* remote user */
    REKONAJHO_ARMEO = 4, /* remote host */
  } rekonajhoj_t;

typedef struct
{
  rekonajhoj_t rekonajhoj;
  char *rekonajho;
  int longo;
} rekonajho_t;

typedef struct
{
  char *memoro;
  int komenco;
  int fino;
  int legite;
  int spaco;
} memoro_t;

typedef struct
{
  rekonajhoj_t rekonajhoj;
  char *notlibro;
  char *ago;
  unsigned int timemo;
  unsigned int limo;
  unsigned int dauro;
} reguloj_t;

typedef struct
{
  char *regularo;
} ordonoj_t;

typedef struct
{
  unsigned int timo;
  time_t lasta;
} noto_t;

#define jhurnalu(graveco, formato, argumentoj...) _jhurnalu(graveco, "pam_siegho: " formato , ##argumentoj)
static void _jhurnalu(int graveco, const char *formato, ...)
{
  va_list argumentoj;
  va_start(argumentoj, formato);
  vsyslog(LOG_AUTHPRIV | graveco, formato, argumentoj);
  va_end(argumentoj);
}

static void forgesu_memoron(memoro_t *memoro)
{
  free(memoro->memoro);
}

static void pretigu_memoron(memoro_t *memoro)
{
  memoro->memoro = NULL;
  memoro->komenco = 0;
  memoro->fino = 0;
  memoro->legite = 0;
  memoro->spaco = 0;
}

static inline int legu_frazon(int libro, memoro_t *memoro, char **frazo)
{
  char *nova_memoro;
  int r;

  if (memoro->memoro)
    memoro->komenco = ++memoro->fino;

  while (1)
    {
      for (; memoro->fino < memoro->legite; memoro->fino++)
	switch (memoro->memoro[memoro->fino])
	  {
	  case '\n':
	    memoro->memoro[memoro->fino] = 0;
	  case 0:
	    *frazo = memoro->memoro + memoro->komenco;
	    return memoro->fino - memoro->komenco;
	  }

      if (memoro->spaco - memoro->legite < 128)
	{
	  if (memoro->komenco)
	    {
	      memmove(memoro->memoro, memoro->memoro + memoro->komenco, memoro->legite - memoro->komenco);
	      memoro->legite -= memoro->komenco;
	      memoro->fino -= memoro->komenco;
	      memoro->komenco = 0;
	    }

	  if (memoro->spaco - memoro->legite < 128)
	    {
	      nova_memoro = (char *) realloc(memoro->memoro, memoro->spaco = memoro->legite + 256);
	      if (nova_memoro == NULL)
		{
		  jhurnalu(LOG_CRIT, "memorpeto rifuzighis (%m)!");
		  return -1;
		}

	      memoro->memoro = nova_memoro;
	    }
	}

      r = read(libro, memoro->memoro + memoro->legite, memoro->spaco - memoro->legite);
      if (r == -1)
	{
	  jhurnalu(LOG_CRIT, "legi el dosiero malsukcesis (%m)!");
	  return -1;
	}

      if (r == 0)
	{
	  if (memoro->fino - memoro->komenco)
	    {
	      memoro->memoro[memoro->fino] = 0;
	      *frazo = memoro->memoro + memoro->komenco;
	      return memoro->fino - memoro->komenco;
	    }

	  return 0;
	}

      memoro->legite += r;
    }
}

static inline void komprenu(char *frazo, char **nomo, char **valoro)
{
  int f;

  *nomo = NULL;
  *valoro = NULL;

  if ((frazo[0] == 0) || (frazo[0] == '#')) return;
  *nomo = frazo;
  for (f = 0; frazo[f]; f++)
    if (frazo[f] == '=')
      goto valoro;

  return;

 valoro:
  frazo[f] = 0;
  *valoro = frazo + f + 1;
}

static int numero(const char *valoro)
{
  int n = 0;
  while ((*valoro >= '0') && (*valoro <= '9'))
    n = n * 10 + *(valoro++) - '0';

  if (*valoro)
    jhurnalu(LOG_WARNING, "mi ignoris malbonan ciferon \"%c\"!", *valoro);

  return n;
}

static int tempo(const char *valoro)
{
  int n = 0;
  while ((*valoro >= '0') && (*valoro <= '9'))
    n = n * 10 + *(valoro++) - '0';
  if (strcmp(valoro, "m") == 0)
    n *= 60;
  else if (strcmp(valoro, "h") == 0)
    n *= 60 * 60;
  else if (strcmp(valoro, "t") == 0)
    n *= 60 * 60 * 24;
  else if (strcmp(valoro, "s") != 0)
    jhurnalu(LOG_WARNING, "mi ignoris nekonatan tempunuon \"%s\"!", valoro);

  return n;
}

static void forgesu_regulojn(reguloj_t *reguloj)
{
  if (reguloj->notlibro && (reguloj->notlibro != supozata_notlibro))
    free(reguloj->notlibro);
  if (reguloj->ago)
    free(reguloj->ago);
}

static int legu_regulojn(const char *loko, reguloj_t *reguloj)
{
  int libro, r, i;
  memoro_t memoro;
  char *frazo, *nomo, *valoro, *sekvanta;
  struct flock baro;

  libro = open(loko, O_RDONLY);
  if (libro == -1)
    {
      jhurnalu(LOG_ERR, "malfermi regularon \"%s\" malsukcesis (%m)!", loko);
      goto eraro;
    }

  baro.l_type = F_RDLCK;
  baro.l_whence = SEEK_SET;
  baro.l_start = 0;
  baro.l_len = 0;

  if (fcntl(libro, F_SETLKW, &baro) == -1)
    {
      jhurnalu(LOG_CRIT, "bari regularon \"%s\" malsukcesis (%m)!", loko);
      goto eraro_malfermite;
    }

  pretigu_memoron(&memoro);

  reguloj->rekonajhoj = REKONAJHO_ARMEO;
  reguloj->notlibro = supozata_notlibro;
  reguloj->ago = NULL;
  reguloj->timemo = 60;
  reguloj->limo = 10;
  reguloj->dauro = 60 * 60 * 24;

  while ((r = legu_frazon(libro, &memoro, &frazo)) > 0)
    {
      komprenu(frazo, &nomo, &valoro);
      if (nomo == NULL) continue;
      if (valoro == NULL)
	{
	  jhurnalu(LOG_WARNING, "vi ne donis valoron por \"%s\", mi ignoros la regulon!", nomo);
	  continue;
	}

      if ((strcmp(nomo, "rekonaĵo") == 0) || (strcmp(nomo, "rekonajho") == 0))
	{
	  reguloj->rekonajhoj = 0;
	  do
	    {
	      sekvanta = NULL;
	      for (i = 0; valoro[i]; i++)
		if (valoro[i] == ',')
		  {
		    valoro[i] = 0;
		    sekvanta = valoro + i + 1;
		    break;
		  }

	      if (strcmp(valoro, "anonco") == 0)
		reguloj->rekonajhoj |= REKONAJHO_ANONCO;
	      else if (strcmp(valoro, "militisto") == 0)
		reguloj->rekonajhoj |= REKONAJHO_MILITISTO;
	      else if (strcmp(valoro, "armeo") == 0)
		reguloj->rekonajhoj |= REKONAJHO_ARMEO;
	      else
		jhurnalu(LOG_WARNING, "mi ignoris nekonatan rekonajhon \"%s\"!", valoro);
	    }
	  while ((valoro = sekvanta));
	}
      else if (strcmp(nomo, "notlibro") == 0)
	{
	  if (reguloj->notlibro && (reguloj->notlibro != supozata_notlibro))
	    free(reguloj->notlibro);
	  reguloj->notlibro = strdup(valoro);
	  if (reguloj->notlibro == NULL)
	    {
	      jhurnalu(LOG_CRIT, "kopii valoron de \"notlibro\" malsukcesis (%m)!");
	      goto eraro_legante;
	    }
	}
      else if (strcmp(nomo, "ago") == 0)
	{
	  if (reguloj->ago)
	    free(reguloj->ago);
	  reguloj->ago = strdup(valoro);
	  if (reguloj->ago == NULL)
	    {
	      jhurnalu(LOG_CRIT, "kopii valoron de \"ago\" malsukcesis (%m)!");
	      goto eraro_legante;
	    }
	}
      else if (strcmp(nomo, "timemo") == 0)
	reguloj->timemo = tempo(valoro);
      else if (strcmp(nomo, "limo") == 0)
	reguloj->limo = numero(valoro);
      else if ((strcmp(nomo, "daŭro") == 0) || (strcmp(nomo, "dauro") == 0))
	reguloj->dauro = tempo(valoro);
    }

  if (r == -1)
    {
      jhurnalu(LOG_CRIT, "legi el regularo malsukcesis!");
      goto eraro_legante;
    }

  if (reguloj->rekonajhoj == 0)
    {
      jhurnalu(LOG_ERR, "Vi difinis neniun rekonajhon!");
      forgesu_regulojn(reguloj);
      return 0;
    }

  return 1;

 eraro_legante:
  forgesu_memoron(&memoro);
  forgesu_regulojn(reguloj);
 eraro_malfermite:
  close(libro);
 eraro:
  return -1;
}

static void forgesu_ordonojn(ordonoj_t *ordonoj)
{
  if (ordonoj->regularo && (ordonoj->regularo != supozata_regularo))
    free(ordonoj->regularo);
}

static int auskultu_ordonojn(int n_argumentoj, const char **argumentoj, ordonoj_t *ordonoj)
{
  int i;
  char *argumento, *nomo, *valoro;

  ordonoj->regularo = supozata_regularo;

  for (i = 0; i < n_argumentoj; i++)
    {
      argumento = strdup(argumentoj[i]);
      if (argumento == NULL)
	{
	  if (ordonoj->regularo && (ordonoj->regularo != supozata_regularo))
	    free(ordonoj->regularo);
	  jhurnalu(LOG_CRIT, "kopii argumenton malsukcesis (%m)!");
	  return -1;
	}

      komprenu(argumento, &nomo, &valoro);
      if (nomo == NULL) continue;

      if (strcmp(nomo, "regularo") == 0)
	{
	  if (valoro)
	    {
	      if (ordonoj->regularo && (ordonoj->regularo != supozata_regularo))
		free(ordonoj->regularo);
	      ordonoj->regularo = strdup(valoro);
	      if (ordonoj->regularo == NULL)
		{
		  free(argumento);
		  jhurnalu(LOG_CRIT, "kopii valoron de argumento \"regularo\" malsukcesis (%m)!");
		  return -1;
		}
	    }
	  else
	    jhurnalu(LOG_WARNING, "vi ne diris lokon por la regularo, mi ignoros la argumenton!");
	}
      else
	jhurnalu(LOG_WARNING, "mi ignoris nekonatan argumenton \"%s\"!", nomo);	

      free(argumento);
    }

  return 1;
}

static int ricevu_rekonajhojn(pam_handle_t *pam, rekonajhoj_t rekonajhoj, rekonajho_t *rekonajho)
{
  int i, n;
  int longo, longoj[3];
  char *parto;
  const char *partoj[3];

  rekonajho->rekonajhoj = rekonajhoj;
  n = 0;
  
  if (rekonajhoj & REKONAJHO_ANONCO)
    {
      if ((pam_get_user(pam, &partoj[n], NULL) != PAM_SUCCESS) ||
	  (partoj[n] == NULL) || (*(partoj[n]) == '\0'))
	{
	  jhurnalu(LOG_ERR, "anonco nekonata!");
	  return 0;
	}
      n++;
    }

  if (rekonajhoj & REKONAJHO_MILITISTO)
    {
      if ((pam_get_item(pam, PAM_RUSER, (void *) (&partoj[n])) != PAM_SUCCESS) ||
	  (partoj[n] == NULL) || (*(partoj[n]) == '\0'))
	{
	  jhurnalu(LOG_ERR, "militisto nekonata!");
	  return 0;
	}
      n++;
    }

  if (rekonajhoj & REKONAJHO_ARMEO)
    {
      if ((pam_get_item(pam, PAM_RHOST, (void *) (&partoj[n])) != PAM_SUCCESS) ||
	  (partoj[n] == NULL) || (*(partoj[n]) == '\0'))
	{
	  jhurnalu(LOG_ERR, "armeo nekonata!");
	  return 0;
	}

      n++;
    }

  longo = 0;
  for (i = 0; i < n; i++)
    if (partoj[i])
      longo += longoj[i] = strlen(partoj[i]) + 1;

  rekonajho->rekonajho = (char *) malloc(longo);
  rekonajho->longo = longo;
  if (rekonajho->rekonajho == NULL)
    {
      jhurnalu(LOG_CRIT, "memorpeto rifuzighis (%m)!");
      return -1;
    }

  parto = rekonajho->rekonajho;
  for (i = 0; i < n; i++)
    {
      memcpy(parto, partoj[i], longoj[i]);
      parto += longoj[i];
    }

  return 1;
}

static void forgesu_rekonajhojn(rekonajho_t *rekonajho)
{
  free(rekonajho->rekonajho);
}

static int serchu_regularon(pam_handle_t *pam, ordonoj_t *ordonoj, char **regularo)
{
  struct stat informoj;
  const char *servo;

  if (stat(ordonoj->regularo, &informoj) == -1)
    {
      jhurnalu(LOG_ERR, "Regularo \"%s\" ne troveblas (%m)!", ordonoj->regularo);
      return 0;
    }

  if (S_ISREG(informoj.st_mode))
    {
      *regularo = strdup(ordonoj->regularo);
      if (*regularo == NULL)
	return -1;
    }
  else if (S_ISDIR(informoj.st_mode))
    {
      if ((pam_get_item(pam, PAM_SERVICE, (void *) &servo) != PAM_SUCCESS) ||
	  (servo == NULL) || (*servo == 0))
	{
	  jhurnalu(LOG_ERR, "La nomo de la servo ne estas konata!");
	  return 0;
	}

      if (asprintf(regularo, "%s/%s", ordonoj->regularo, servo) == -1)
	{
	  jhurnalu(LOG_CRIT, "serchi regularon malsukcesis (%m)!");
	  return -1;
	}

      if (stat(*regularo, &informoj) == -1)
	{
	  free(*regularo);
	  if (asprintf(regularo, "%s/%s", ordonoj->regularo, "aliaj") == -1)
	    {
	      jhurnalu(LOG_CRIT, "serchi regularon malsukcesis (%m)!");
	      return -1;
	    }

	  if (stat(*regularo, &informoj) == -1)
	    {
	      free(*regularo);
	      jhurnalu(LOG_ERR, "Regularo \"%s\" ne troveblas (%m)!", ordonoj->regularo);
	      return 0;
	    }
	}
    }
  else
    {
      jhurnalu(LOG_ERR, "Regularo \"%s\" ne estas (ligita al) dosiero au dosierujo!", regularo);
      return 0;
    }

  return 1;
}

static int malfermu_notlibron(const char *loko, int *notlibro)
{
  struct flock baro;

  *notlibro = open(loko, O_CREAT | O_RDWR, S_IRUSR | S_IWUSR);
  if (*notlibro == -1)
    {
      jhurnalu(LOG_ALERT, "malfermi notlibron \"%s\" malsukcesis (%m)!", loko);
      return -1;
    }

  baro.l_type = F_WRLCK;
  baro.l_whence = SEEK_SET;
  baro.l_start = 0;
  baro.l_len = 0;
  if (fcntl(*notlibro, F_SETLKW, &baro) == -1)
    {
      close(*notlibro);
      jhurnalu(LOG_CRIT, "bari notlibron \"%s\" malsukcesis (%m)!", loko);
      return -1;
    }

  return 1;
}

static int serchu_noton(int notlibro, rekonajho_t *rekonajho, noto_t *noto)
{
  int longo;
  rekonajhoj_t libro_rekonajhoj;
  char *libro_rekonajho = NULL;
  struct stat informoj;
  off_t sekvonta = 0;

  if (fstat(notlibro, &informoj) == -1)
    {
      jhurnalu(LOG_CRIT, "ricevi informojn pri la notlibro malsukcesis (%m)!");
      return -1;
    }

  while (sekvonta < informoj.st_size)
    {
      if (read(notlibro, &longo, sizeof(int)) != sizeof(int))
	{
	  jhurnalu(LOG_CRIT, "legi el la notlibro malsukcesis (%m)!");
	  goto eraro;
	}

      sekvonta += longo;
      if (read(notlibro, &libro_rekonajhoj, sizeof(rekonajhoj_t)) != sizeof(rekonajhoj_t))
	{
	  jhurnalu(LOG_CRIT, "legi el la notlibro malsukcesis (%m)!");
	  goto eraro;
	}

      longo -= sizeof(noto_t) + sizeof(rekonajhoj_t) + sizeof(int);
      if ((libro_rekonajhoj == rekonajho->rekonajhoj) && (longo == rekonajho->longo))
	{
	  if (libro_rekonajho == NULL)
	    {
	      libro_rekonajho = (char *) malloc(longo);
	      if (libro_rekonajho == NULL)
		{
		  jhurnalu(LOG_CRIT, "memorpeto rifuzighis (%m)!");
		  return -1;
		}
	    }

	  if (read(notlibro, libro_rekonajho, longo) != longo)
	    {
		  jhurnalu(LOG_CRIT, "legi el la notlibro malsukcesis (%m)!");
		  goto eraro;
	    }

	  if (memcmp(libro_rekonajho, rekonajho->rekonajho, longo) == 0)
	    {
	      free(libro_rekonajho);
	      if ((read(notlibro, noto, sizeof(noto_t)) != sizeof(noto_t)) || 
		  (lseek(notlibro, -sizeof(noto_t), SEEK_CUR) == -1))
		{
		  jhurnalu(LOG_CRIT, "legi el la notlibro malsukcesis (%m)!");
		  return -1;
		}

	      return 1;
	    }
	}

      if (lseek(notlibro, sekvonta, SEEK_SET) == -1)
	{
	  jhurnalu(LOG_CRIT, "legi el la notlibro malsukcesis (%m)!");
	  goto eraro;
	}
    }

  free(libro_rekonajho);

  noto->timo = 0;
  noto->lasta = 0;
  longo = rekonajho->longo + sizeof(noto_t) + sizeof(rekonajhoj_t) + sizeof(int);
  if ((write(notlibro, &longo, sizeof(int)) != sizeof(int)) ||
      (write(notlibro, &rekonajho->rekonajhoj, sizeof(rekonajhoj_t)) != sizeof(rekonajhoj_t)) ||
      (write(notlibro, rekonajho->rekonajho, rekonajho->longo) != rekonajho->longo))
    {
      jhurnalu(LOG_CRIT, "skribi en la notlibro malsukcesis (%m)!");
      return -1;
    }

  return 1;

 eraro:
  free(libro_rekonajho);
  return 0;
}

static int trovu_regulojn(pam_handle_t *pam, int n_argumentoj, const char **argumentoj, reguloj_t *reguloj)
{
  int r;
  ordonoj_t ordonoj;
  char *regularo;

  r = auskultu_ordonojn(n_argumentoj, argumentoj, &ordonoj);
  if (r <= 0) goto fino;

  r = serchu_regularon(pam, &ordonoj, &regularo);
  forgesu_ordonojn(&ordonoj);
  if (r <= 0) goto fino;

  r = legu_regulojn(regularo, reguloj);
  free(regularo);

 fino:
  return r;
}

static int trovu_noton(pam_handle_t *pam, reguloj_t *reguloj, int *notlibro, noto_t *noto)
{
  int r;
  rekonajho_t rekonajho;

  r = ricevu_rekonajhojn(pam, reguloj->rekonajhoj, &rekonajho);
  if (r <= 0) goto fino;

  r = malfermu_notlibron(reguloj->notlibro, notlibro);
  if (r <= 0) goto fino_rekonajho;
  r = serchu_noton(*notlibro, &rekonajho, noto);
  if (r <= 0)
    close(*notlibro);

 fino_rekonajho:
  forgesu_rekonajhojn(&rekonajho);
 fino:
  return r;
}

static int skribu_noton(int notlibro, noto_t *noto)
{
  int r;
  r = write(notlibro, noto, sizeof(noto_t));
  if (r != sizeof(noto_t))
    jhurnalu(LOG_CRIT, "skribi en la notlibro malsukcesis (%m)!");
  close(notlibro);
  return (r > 0) ? 1 : -1;
}

PAM_EXTERN int pam_sm_authenticate(pam_handle_t *pam, int flagoj, int n_argumentoj, const char **argumentoj)
{
  int r;
  reguloj_t reguloj;
  int notlibro;
  noto_t noto;
  time_t t;
  unsigned int n;
  const char *valoro;

  r = trovu_regulojn(pam, n_argumentoj, argumentoj, &reguloj);
  if (r <= 0) return (r == -1) ? PAM_AUTHINFO_UNAVAIL : PAM_AUTH_ERR;


  r = trovu_noton(pam, &reguloj, &notlibro, &noto);
  if (r <= 0)
    {
      forgesu_regulojn(&reguloj);
      return (r == -1) ? PAM_AUTHINFO_UNAVAIL : PAM_AUTH_ERR;
    }

  t = time(NULL);
  if ((noto.timo <= reguloj.limo) || (t - noto.lasta > reguloj.dauro))
    {
      n = (t - noto.lasta) / reguloj.timemo;
      noto.timo = (n < noto.timo) ? noto.timo - n : 0;
    }

  noto.timo++;
  noto.lasta = t;

  r = skribu_noton(notlibro, &noto);
  if (r <= 0)
    {
      forgesu_regulojn(&reguloj);
      return (r == -1) ? PAM_AUTHINFO_UNAVAIL : PAM_AUTH_ERR;
    }

  if (noto.timo > reguloj.limo)
    {
      jhurnalu(LOG_NOTICE, "mi riglas la pordon!");
      if (noto.timo == reguloj.limo + 1)
	{
	  if (reguloj.ago)
	    {
	      r = fork();
	      if (r == 0)
		{
		  if ((pam_get_user(pam, &valoro, NULL) == PAM_SUCCESS) && (valoro != NULL))
		    setenv("ANONCO", valoro, 1);
		  if ((pam_get_item(pam, PAM_RUSER, (void *) &valoro) == PAM_SUCCESS) && (valoro != NULL))
		    setenv("MILITISTO", valoro, 1);
		  if ((pam_get_item(pam, PAM_RHOST, (void *) &valoro) == PAM_SUCCESS) && (valoro != NULL))
		    setenv("ARMEO", valoro, 1);
		  if ((pam_get_item(pam, PAM_SERVICE, (void *) &valoro) == PAM_SUCCESS) && (valoro != NULL))
		    setenv("SERVO", valoro, 1);

		  execl("/bin/sh", "/bin/sh", "-c", reguloj.ago, (char *) NULL);
		  jhurnalu(LOG_CRIT, "fari agon \"%s\" malsukcesis (%m)!", reguloj.ago);
		  exit(1);
		}
	      else if (r > 0)
		{
		  if (wait(&r) == -1)
		    jhurnalu(LOG_CRIT, "atendi la agon malsukcesis (%m)!");
		  else if (!WIFEXITED(r) || (WEXITSTATUS(r) != 0))
		    jhurnalu(LOG_WARNING, "ago \"%s\" malsukcesis!", reguloj.ago);
		}
	      else
		jhurnalu(LOG_CRIT, "fari agon malsukcesis (%m)!");
	    }
	}
      r = PAM_MAXTRIES;
    }
  else
    {
      jhurnalu(LOG_INFO, "mi akceptas la vizitanton (timo: %d/%d).", noto.timo, reguloj.limo);
      r = PAM_SUCCESS;
    }

  forgesu_regulojn(&reguloj);
  return r;
}

PAM_EXTERN int pam_sm_setcred(pam_handle_t *pam, int flagoj, int n_argumentoj, const char **argumentoj)
{
  int r;
  reguloj_t reguloj;
  int notlibro;
  noto_t noto;

  if (((flagoj & ~PAM_SILENT) != PAM_ESTABLISH_CRED) && ((flagoj & ~PAM_SILENT) != PAM_REINITIALIZE_CRED))
    return PAM_SUCCESS;

  r = trovu_regulojn(pam, n_argumentoj, argumentoj, &reguloj);
  if (r <= 0) return (r == -1) ? PAM_CRED_UNAVAIL : PAM_CRED_ERR;

  r = trovu_noton(pam, &reguloj, &notlibro, &noto);
  forgesu_regulojn(&reguloj);
  if (r <= 0) return (r == -1) ? PAM_CRED_UNAVAIL : PAM_CRED_ERR;

  noto.timo = 0;
  r = skribu_noton(notlibro, &noto);
  if (r <= 0) return (r == -1) ? PAM_CRED_UNAVAIL : PAM_CRED_ERR;

  jhurnalu(LOG_INFO, "mi trankvilighas pri la vizitanto.");
  return PAM_SUCCESS;
}

/* end of module definition */

#ifdef PAM_STATIC
struct pam_module _pam_permit_modstruct = {
#ifdef UTF8
  "pam_sieĝo",
#else
  "pam_siegho",
#endif
  pam_sm_authenticate,
  pam_sm_setcred,
  NULL,
  NULL,
  NULL,
  NULL
};
#endif
