/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef ARRAY_H
#define ARRAY_H

#include <libpandora/global.h>

#include <iostream>
#include <libpandora/serialize.h>
#include <libpandora/error.h>

template<class elt_t>
class Array
{
  typedef Array<elt_t> array_t;

private:
  elt_t		*elts;
  int		nelts;

public:
  Array(void)  : elts(NULL), nelts(0) { }
  inline Array(int n);
  inline Array(const array_t &x);
  inline Array &operator =(const array_t &x);
  inline elt_t &operator[](const int i);
  ~Array(void) { clear(); }
  
  inline elt_t *tab(void) 		{ return elts; }
  inline int size(void) const 		{ return nelts; }
  inline void init(int n);

  inline elt_t &get(int i);
  inline const elt_t &get_const(int i) const;

private:
  void clear(void);
};

template <class elt_t>
Array<elt_t>::Array(int n)
  : nelts(n)
{ 
  if (n > 0) {
    elts = new elt_t[n]; 
  } else {
    elts = NULL;
    nelts = 0;
  }
}

template <class elt_t>
Array<elt_t>::Array(const Array<elt_t> &x)
  : nelts(x.nelts)
{
  elts = new elt_t[nelts];
  for (int i = 0; i < nelts; ++i)
    elts[i] = x.elts[i];
}

template <class elt_t>
Array<elt_t> &Array<elt_t>::operator =(const Array<elt_t> &x)
{
  clear();
  nelts = x.nelts;
  elts = new elt_t[nelts];
  for (int i = 0; i < nelts; ++i)
    elts[i] = x.elts[i];
  return *this;
}

template <class elt_t>
void Array<elt_t>::init(int n)
{
  clear();
  nelts = n;
  elts = new elt_t[nelts];
  bzero(elts, nelts * sizeof(elt_t));
}

template <class elt_t>
void Array<elt_t>::clear(void)
{
  __DELETE_ARRAY(elts);
  nelts = 0;
}

template <class elt_t>
elt_t &Array<elt_t>::operator[](const int i)
{
  pandora_assert(i >= 0 && i < nelts);
  return elts[i];
}

template <class elt_t>
elt_t &Array<elt_t>::get(int i)
{
  pandora_assert(i >= 0 && i < nelts);
  return elts[i];  
}

template <class elt_t>
const elt_t &Array<elt_t>::get_const(int i) const
{
  pandora_assert(i >= 0 && i < nelts);
  return elts[i];  
}

template <class elt_t>
void serialize(char *str, size_t &count, const size_t maxlen, 
	       Array<elt_t> *var)
{
  int siz = var->size();
  serialVar(siz);
  for(int i = 0; i < siz; ++i)
    serialVar((*var)[i]);
}

template <class elt_t>
void unserialize(const char *str, size_t &count, Array<elt_t> *var)
{
  int siz;
  unserialVar(siz);
  if (var == NULL) var = new Array<elt_t>(siz);
  else var->init(siz);
  for(int i = 0; i < siz; ++i)
    unserialVar((*var)[i]);
}


#endif /* ARRAY_H */


