/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include <iomanip>

#include <libpandora/compentry.h>
#include <libpandora/dynloader.h>
#include <libpandora/serialize.h>

#define DEBUG_COMPENTRY 0

CompEntry::CompEntry(void) 
  : ctype(undefined), func(NULL), rank(-1), type(0),
    order(-1), depth(-1), mux(1), branch(0), 
    branches(0), nbOptions(0), param(0), needLookup(true)
{
  memset((char *)sw, 0, MAX_BRANCHES*sizeof(int));
}

CompEntry::CompEntry(compentryt_t ct, const char *name, const char *aka) 
  : ctype(ct), id(name), func(NULL), rank(-1), type(0),
    order(-1), depth(-1), mux(1), branch(0), 
    branches(0), nbOptions(0), param(0), needLookup(true)
{
  alias.init((aka != NULL) ? aka : name);
  memset((char *)sw, 0, MAX_BRANCHES*sizeof(int));
}

CompEntry::CompEntry(const CompEntry &x) 
  : ctype(x.ctype), id(x.id), alias(x.alias),
    func(x.func), rank(x.rank), type(x.type),
    order(x.order), depth(x.depth), mux(x.mux), branch(x.branch), 
    branches(x.branches), nbOptions(x.nbOptions), param(x.param),
    needLookup(true)
{
  for (int i = 0; i < branches; ++i) sw[i] = x.sw[i];
  for (int i = 0; i < nbOptions; ++i)
    optionList[i] = x.optionList[i];
}

CompEntry &CompEntry::operator=(const CompEntry &x) 
{
  ctype = x.ctype; id = x.id; alias = x.alias; 
  func = x.func; rank = x.rank; type = x.type;
  order = x.order; depth = x.depth;
  mux = x.mux; branch = x.branch; branches = x.branches; 
  nbOptions = x.nbOptions; param = x.param;
  needLookup = true;

  for (int i = 0; i < branches; ++i) sw[i] = x.sw[i];
  for (int i = 0; i < nbOptions; ++i)
    optionList[i] = x.optionList[i];
  return *this;
}

CompEntry::~CompEntry(void)
{
  if (func != NULL) {
    dynloader->unload((void *)func);
    func = NULL;
  }
}

bool CompEntry::setup(void)
{
  //pandora_debug("[compentry setup] " << id);
  switch(ctype) {
  case component:
    if (func == NULL) {
      if (lookup()) {
	func = (Component *(*)(void)) dynloader->load((symbol_id_t)id);
	if (func == NULL)
	  pandora_warning("cannot load symbol: " << id);	
      } else {
	pandora_warning("cannot lookup symbol: " << id);
      }
    }
    return (func != NULL);
  case macro:		break;
  default:		break;
  }

  return false;
}

bool CompEntry::getComponent(Component **comp) const
{
  if (comp == NULL) return false;
  
  switch(ctype) {
  case component:	
    if (func == NULL) 	break; 
    *comp = (*func)();
    return (*comp != NULL);
  case macro:		break;
  default:		break;
  }

  return false;
}

size_t CompEntry::write(char *str, size_t maxlen) const
{
  size_t count = 0;

  serialVar(ctype);
  serialVar(id);
  serialVar(alias);
  serialVar(type);
  serialVar(param);

  serialVar(nbOptions);
  for (int i = 0; i < nbOptions; ++i)
    serialVar(optionList[i]);

  return count;
}

size_t CompEntry::read(const char *str)
{
  size_t count = 0;

  unserialVar(ctype);
  unserialVar(id);
  unserialVar(alias);
  unserialVar(type);
  unserialVar(param);

  unserialVar(nbOptions);  
  for (int i = 0; i < nbOptions; ++i)
   unserialVar(optionList[i]);

  return count;
}

void CompEntry::print(ostream *f) const
{
  if (ctype == undefined) return;
  switch (ctype) {
  case component:	*f << "@" << (isParam() ? "!" : "") ; break;
  case macro:		*f << "&"; break;
  default:		return;
  }
  *f << id;
  if ((!alias.isNull()) && (alias != id)) {
    *f << ":" << alias;
  }
  *f << " ";
  if (nbOptions > 0) {
    *f << '[';
    for (int i = 0; i < nbOptions; ++i) {
      optionList[i].print(f);
      if (i < (nbOptions-1)) *f << ", ";
    }
    *f << "] ";
  }

  int typ = GET_COMP_TYPE(type),
    dmx = GET_COMP_DEMUX_FACT(type), 
    swi = GET_COMP_SWITCH_FACT(type);
  
  for (int i = 0; i < dmx; ++ i) 
    *f << ((typ == COMP_DEMUX) ? "< " : "> ");

#define isTypeOf(t, m)	(((t) & (m)) == (m))
  for (int i = 0; i < swi; ++ i) {
    if (isTypeOf(typ, COMP_SWITCH)) *f << "( ";
    if (isTypeOf(typ, COMP_LAST)) *f << ") ";
    if (isTypeOf(typ, COMP_INTER)) *f << "| ";
  }
#undef isTypeOf

  if ((dmx + swi) == 0) *f << "; ";

#if DEBUG_COMPENTRY
  *f << " { 0>" << mux << " ";
  for (int i = 1; i < branches; ++i) *f << i << ">" << sw[i] << " ";
  *f << "}";
#endif
}

bool CompEntry::setOption(const text &op, const MultiValue &mv)
{
  bool status = false;
  OptionEntry *oe;

  if (op.isNull()) 
    goto finished;

  if (mv.type == MultiValue::undefined) {
    status = removeOption(op);
    goto finished;
  }

  oe = getOptionEntry(op);
  if (oe == NULL) {
    OptionEntry new_oe(op);
    new_oe.set(mv);
    status = pushOption(new_oe);
  } else {
    oe->mv = mv;
    status = true;
  }

 finished:
  return status;
}


bool CompEntry::removeOption(const text &op)
{
  //pandora_debug("removing option: " << op);
  int off = 0;
  int n = getNbOptions();

  //pandora_debug(*this);
  for (int i = 0, j = 0; i < n; ++i) {
    OptionEntry *oe = getOptionEntry(i);

    pandora_assert (oe != NULL);
    if ((oe->id == op) || (oe->alias == op)) {
      ++off;
    } else {
      if (j < i) {
	//pandora_debug(*oe << " -> " << j);
	optionList[j] = *oe;
      }
      ++j;
    }
  }
  nbOptions -= off;
  
  //pandora_debug(*this);
  return (off > 0);
}

bool CompEntry::getOrigName(text &orig, const text &alias)
{
  OptionEntry *oe = getOptionEntry(alias);
  if (oe == NULL) 		return false;
  if (oe->alias != alias) 	return false;
  orig = oe->id;
  return true;
}

bool CompEntry::setVars(stack_env_t *env)
{
  //pandora_debug("setting variable in comp: " << id << " (" << param << ")");
  if (env == NULL) return false;
  if (env->size() == 0) return true;
  MultiValue mv;

  if (isParam()) {
    mv = env->atOrNil(alias);
    if (mv.type == MultiValue::textual) {
      id = text(mv.value.s);
      --param;
    }
  }

  if (needParam()) {
    for (int i = 0; i < nbOptions; ++i) {
      OptionEntry *oe = getOptionEntry(i);
      //pandora_debug("examining option: " << oe->id);
      if (!oe->isParam()) continue;
      mv = env->atOrNil(oe->alias);
      //pandora_debug("found in env: " << mv);
      if (mv.type == MultiValue::undefined) continue;
      //pandora_debug("setting option: " << oe->id << " to " << mv);
      oe->set(mv);
      param -= 2;
    }
  }

  return true;
}

ostream& operator<<(ostream& f, const CompEntry &ce)
{
  ce.print(&f);
  return f;
}
