/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef COMP_ENTRY_H
#define COMP_ENTRY_H

#include <libpandora/global.h>

#include <iostream>
#include <libpandora/optionentry.h>
#include <libpandora/dynloader.h>
#include <libpandora/pandora_limits.h>
#include <libpandora/multivalue.h>
#include <libpandora/map.h>
#include <libpandora/text.h>
#include <libpandora/error.h>


#define	COMP_SIMPLE		(1 << 0)
#define COMP_DEMUX		(1 << 1)
#define COMP_PREMUX		(1 << 2)
#define COMP_SWITCH		(1 << 3)
#define COMP_INTER		(1 << 4)
#define COMP_LAST		(1 << 5)
#define COMP_MACRO		(1 << 6)

#define GET_COMP_TYPE(x)	((x) & 0x000000ff)
#define GET_COMP_DEMUX_FACT(x)	(((x) & 0x0000ff00) >> 8)
#define GET_COMP_SWITCH_FACT(x)	(((x) & 0x00ff0000) >> 16)
#define INC_COMP_DEMUX_FACT(x)	((x) += 0x00000100) 
#define DEC_COMP_DEMUX_FACT(x)	((x) -= 0x00000100) 
#define INC_COMP_SWITCH_FACT(x)	((x) += 0x00010000) 
#define DEC_COMP_SWITCH_FACT(x)	((x) -= 0x00010000) 

class Component;

typedef Map<text, MultiValue> stack_env_t;

struct CompEntry {
  enum compentryt_t { undefined = -1, component, macro };

  compentryt_t	ctype;
  text 		id;
  text 		alias;
  Component 	*(*func)(void);
  int 		rank;
  int 		type;

  int 		order, depth;
  int 		mux, sw[MAX_BRANCHES];
  int 		branch, branches;
  int 		nbOptions;
  OptionEntry 	optionList[MAX_OPTIONS];
  int 		param;

  bool 		needLookup;

  CompEntry(void);
  CompEntry(compentryt_t, const char *name, const char *aka = NULL);
  CompEntry(const CompEntry &x);
  CompEntry &operator=(const CompEntry &x);
  ~CompEntry(void);

  inline void init(const char *name)	   { id.init(name); }
  inline void setAlias(const char *name)   { alias.init(name); }
  inline bool lookup(void);
  bool setup(void);

  bool isComponent(void) const 	{ return ctype == component; }
  bool isMacro(void) const 	{ return ctype == macro; }

  bool getComponent(Component **) const;

  size_t write(char *, size_t maxlen) const;
  size_t read(const char *);
  void   print(ostream *f = &cout) const;

  inline int getNbOptions(void) const 	{ return nbOptions; }

  inline OptionEntry *getOptionEntry(int index);
  inline OptionEntry *getOptionEntry(const text &id);
  inline bool pushOption(const OptionEntry &oe);
  inline void popOption(void);

  inline void clear(void) 		{ func = NULL; clearOptions(); }
  inline void clearOptions(void) 	{ nbOptions = 0; }
  
  bool setOption(const text &op, const MultiValue &mv);
  bool removeOption(const text &op);

  bool getOrigName(text &, const text &);

  inline void snarfOptions(const CompEntry *ce);
  inline void setParam(void) 		{ ++param; }
  inline bool isParam(void) const	{ return (param % 2); }
  inline void setOpParam(void) 		{ param += 2; }
  inline bool needParam(void) const  	{ return (param > 0); }
  bool setVars(stack_env_t *env);
};

inline void serialize(char *str, size_t &count, 
		 const size_t maxlen, const CompEntry *ce) 
{
  count += ce->write(str + count, maxlen - count);
}

inline void unserialize(const char *str, size_t &count, CompEntry *ce)
{
  count += ce->read(str + count);
}

extern ostream &operator<< (ostream &, const CompEntry&);

bool CompEntry::lookup(void) 		   
{ 
  if (ctype != component) {
    needLookup = false;
    return true;
  }
  if (needLookup)
    needLookup = !dynloader->lookup(id, id); 
  return !needLookup;
}

OptionEntry *CompEntry::getOptionEntry(int index)
{
  pandora_assert(index >= 0 && index < nbOptions);
  return &(optionList[index]);
}

OptionEntry *CompEntry::getOptionEntry(const text &id)
{
  for(int i = 0; i < nbOptions; ++i) {
    OptionEntry *oe = getOptionEntry(i);
    if ((oe->id == id) || (oe->alias == id)) return oe;
  }    
  return NULL;
}

bool CompEntry::pushOption(const OptionEntry &oe) 
{
  if (nbOptions >= MAX_OPTIONS) return false;
  optionList[nbOptions] = oe;
  ++nbOptions;
  return true;
}

void CompEntry::popOption(void) 
{
  --nbOptions;
}

void CompEntry::snarfOptions(const CompEntry *ce)
{
  nbOptions = ce->nbOptions;
  for (int i = 0; i < nbOptions; ++i)
    optionList[i] = (ce->optionList)[i];  
}

serialEnum(CompEntry::compentryt_t);

#endif /* COMP_ENTRY_H */
