/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <libpandora/conf/time.h>
#include <libpandora/conf/unistd.h>
#include <libpandora/conf/fcntl.h>
}

#include <libpandora/error.h>
#include <libpandora/slog.h>

static ofstream devnull("/dev/null");
static void getlname(char *name, int size);

static bool useSyslog  = false;
static const char *progname = "pandora";
static char localhost[256];
static const int __tmp__ = (getlname(localhost, sizeof(localhost)), 0);
static int verbosity = 
#ifdef NDEBUG
LOG_WARNING
#else
LOG_INFO
#endif
;

void use_syslog(void)
{
  if (useSyslog) return;
  useSyslog = true;
  slog(progname, SLOG_DAEMON);
  slog.clogEnable(false);
}

int set_verbosity(int verb)
{
  int old = verbosity;
  if (verb > LOG_DEBUG) verb = LOG_DEBUG;
  if (verb < LOG_ERR) verb = LOG_ERR;
  verbosity = verb;
  return old;
}

int get_verbosity(void)
{
  return verbosity;
}

void _abort(void)
{
  abort();
}

void _assert_(const char *file, int line, const char *func, const char *expr)
{
  _cerr(LOG_ERR, file, line, func) << "assertion failed: " << expr << "\n";
  _abort();
}

ostream &_cerr(int level, const char *file, int line, const char *func)
{
  ostream *s = (useSyslog ? (ostream *) &slog((slog_level_t)level) : &cerr);
  if (level > verbosity) return devnull;
#if 0
  time_t now;
  time(&now);
  struct tm *then= localtime(&now);
  char currentTime[32];
  strftime(currentTime, sizeof(currentTime), "%b %d %H:%M:%S", then);
  if (level < LOG_WARNING) *s << '\n';
  if (!useSyslog) cerr << currentTime << ' ' << localhost << ' '
		       << progname << '[' << getpid() << "]: ";
  if (level <= LOG_WARNING) *s << func << '[' << file << ':' << line << "]: ";
#else
  if (!useSyslog) *s << progname << ':';
  if (level <= LOG_WARNING) *s << file << ':' << line << ':';  
  *s << ' ';
#endif
  return *s;
}

static void getlname(char *name, int size)
{
  if (gethostname(name, size) < 0) {
    *name = '\0';
    return;
  }
  
  char *ptr = strchr(name, '.');
  if (ptr != NULL) *ptr = '\0';
}

