/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <stdlib.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <libpandora/conf/unistd.h>
#include <libpandora/conf/dirent.h>
#include <libpandora/conf/time.h>
#include <libpandora/conf/snprintf.h>
#include <glob.h>
#include <errno.h>
   }

#include <libpandora/fileutil.h>
#include <libpandora/error.h>
#include <libpandora/util.h>

void rotate_files(const char *prefix, FILE **f, time_t cur_time)
{
  pandora_assert( f != NULL );
  if (prefix == NULL) return;

  char file[256];

  tm *gmt = new tm;
  (void) gmtime_r(&cur_time, gmt);
  char cur_date[17];
  strftime(cur_date, 16, "%Y%m%d_%H%M%S", gmt);
  delete gmt;
  snprintf(file, 256, "%s_%s.dat",
	   prefix, cur_date);

  *f = fopen(file, "w");
  pandora_assert(*f != NULL);
  //pandora_debug("opening "<< file);
}

static char pdir[PATH_MAX] = "";

static bool make_pdir(void)
{
  int plen = sizeof(pdir);
  char *_pdir = pdir;
  const char *home = getenv("HOME");
  if (home == NULL) {
    pandora_warning("cannot find your home directory, set HOME variable");
    return false;
  }
  int len = strlen(home);
  if (plen < len) goto err;
  memcpy(_pdir, home, len);
  _pdir += len;
  plen -= len;
  if (plen < 11) goto err;
  memcpy(_pdir, "/.pandora/", 11);
  return true;

 err:
  *pdir = '\0';
  return false;
}

bool create_dir(const char *dir)
{
  struct stat st;

  if (lstat(dir, &st) == 0) {
    if (!S_ISDIR(st.st_mode)) {
      pandora_warning(dir << " exists and is not a directory");
      return false;
    } else {
      // directory exists
      return true;
    }
  } else {
    if (errno != ENOENT) {
      pandora_pwarning(dir);
      return false;
    }
  }

  if ((mkdir(dir, S_IRWXU) < 0) && (errno != EEXIST)) {
    pandora_pwarning(dir);
    return false;
  }

  pandora_debug("created directory: " << dir);
  return true;
}

const char *pandora_dir(void)
{
  if (*pdir == '\0') {
    if (make_pdir() && create_dir(pdir)) goto ok;
#if 0
    strcpy(pdir, "/var/tmp/.pandora/");
    pandora_warning("cannot create pandora directory in HOME, "
		    "trying with /var/tmp");
    if (create_pdir()) goto ok;
    pandora_warning("cannot create pandora directory in /var/tmp, giving up");
#else
    pandora_warning("cannot create pandora directory");
#endif
  }

 ok:
  return pdir;
}

extern bool rmdir_rec(const char *dirname)
{
  if (chdir(dirname) < 0) return false;
  
  bool status = true;
  
  DIR *dir = opendir(".");
  dirent *d;
  while((d = readdir(dir)) != NULL) {
    struct stat st;
    const char *file = d->d_name;
    if (!strcmp(".",  file)) continue;
    if (!strcmp("..", file)) continue;
    
    if (lstat(file, &st) != 0) {
      pandora_pwarning(file);
      return false;
    }

    if (S_ISDIR(st.st_mode)) {
      if (!rmdir_rec(file)) 
	return false;
    } else {
      if (unlink(file) < 0) {
	pandora_pwarning("unlink");
	status = false;
      }
    }
  }
  closedir(dir);

  if (chdir("..") < 0) {
    pandora_pwarning(dirname);
    return false;
  }
  if (rmdir(dirname) < 0) {
    pandora_pwarning(dirname);
    return false;    
  }

  return true;
}

bool create_empty_dir(const char *dir)
{
  (void) rmdir_rec(dir);
  if (!create_dir(dir)) return false;
  return true;
}

char *split_path_list(const char *plist, char *dest, int maxlen)
{
  if (plist == NULL || dest == NULL) return NULL;
  *dest = '\0';
  char *ptr = strchr(plist, ':');
  if (ptr == NULL) {
    strncpy(dest, plist, maxlen);
    return NULL;
  }
  int len = pandora_min((ptr - plist), maxlen-1);
  memcpy(dest, plist, len);
  dest[len] = '\0';
  return ptr+1;
}

void glob_list(const char *pattern, const char *plist)
{
  if (pattern == NULL || plist == NULL) return;

  int patlen = strlen(pattern);
  char path[2048];
  
  char *tmp = (char *)plist;
  glob_t *g = NULL;

  *path = '\0';
  do {
    g = (glob_t *)malloc(sizeof(glob_t));
    tmp = split_path_list(tmp, path, sizeof(path));
    strncat(path, "/", sizeof(path));
    strncat(path, pattern, sizeof(path));
    if (glob(path, 0, NULL, g) != 0) {
      fprintf(stderr, "%s: %s\n", path, strerror(errno));
      goto finished;
    }
    for (int i = 0; i < (int)g->gl_pathc; ++i) {
      // XXX find an usable API!
      printf("%s\n", g->gl_pathv[i]);
    }
  finished:
    globfree(g);
  } while (tmp != NULL);
  
}

char *read_in_string(const char *name)
{
  struct stat st;
  int len = 0;
  char *str;
  FILE *f = NULL;
  if (lstat(name, &st) < 0) 		goto failed;
  len = (int)st.st_size;
  if ((f = fopen(name, "r")) == NULL) 	goto failed;
  str = (char *)xmalloc((len+2)*sizeof(char));
  if ((int)fread(str, len, 1, f) != 1) 	goto failed;
  if (fgetc(f) != EOF) 			goto failed;
  str[len] = str[len+1] = '\0';
  if (fclose(f) == EOF) 		goto failed;

  return str;

 failed:
  pandora_pwarning(name);
  return NULL;  
}

bool file_is_modified(const char *name, time_t t)
{
  struct stat st;

  if (name == NULL) return false;
  if (stat(name, &st) != 0) {
    pandora_pwarning(name);
    return false;
  }

  return (st.st_mtime > t);
}

bool file_exists(const char *name)
{
  struct stat st;
  return (stat(name, &st) == 0);
}

bool make_default_uri(char *uri, int len, const char *file, const char *var)
{
  char *defenv = getenv(var);
  const char *defval = (defenv != NULL) ? defenv : file;

  char *pandora_home = getenv("PANDORA_HOME");
  if (pandora_home == NULL) pandora_home = ".";

  return (snprintf(uri, len, "file:%s/%s", pandora_home, defval) > 0);
}

int file_writeblock(int fd, const char *buffer, int nbytes)
{
  int n;
  int offset;
  
  for (offset = 0; offset < nbytes; offset += n) {
    if ((n = write(fd, buffer + offset, nbytes - offset)) < 0) {
      if (errno == EINTR) {
	n = 0;
      } else {
	pandora_pwarning("file_writeblock");
	return -1;
      }
    } else if (n==0) break;
  }

  return offset;
}

int file_readblock(int fd, char *buffer, int nbytes)
{
  register int n;
  register int offset;
  
  for (offset = 0; offset < nbytes; offset += n) {
    if ((n = read(fd, buffer + offset, nbytes - offset)) < 0) {
      if (errno == EINTR) {
	n = 0;
      } else {
	pandora_pwarning("file_readblock");
	return -1;
      }
    } else if (n==0) break;
  }
  
  return offset;
}
